/* Program to calculate CRCs identical to the LVM2 plugin.
 * Compile with:
 *    gcc -o lvm2_crc lvm2_crc.c
 * Run with:
 *   ./lvm2_crc <name_of_binary_file> <starting_offset_in_bytes> <size_in_bytes>
 */

#define _GNU_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>

#define INITIAL_CRC 0xf597a6cf

u_int32_t calc_crc(u_int32_t initial, void *buffer, u_int32_t size)
{
	static const u_int32_t lvm2_crc_table[] = {
		0x00000000, 0x1db71064, 0x3b6e20c8, 0x26d930ac,
		0x76dc4190, 0x6b6b51f4, 0x4db26158, 0x5005713c,
		0xedb88320, 0xf00f9344, 0xd6d6a3e8, 0xcb61b38c,
		0x9b64c2b0, 0x86d3d2d4, 0xa00ae278, 0xbdbdf21c
		};
	u_int32_t i, crc = initial;
	u_int8_t *data = (u_int8_t *) buffer;

	for (i = 0; i < size; i++) {
		crc ^= *data++;
		crc = (crc >> 4) ^ lvm2_crc_table[crc & 0xf];
		crc = (crc >> 4) ^ lvm2_crc_table[crc & 0xf];
	}

	return crc;
}

int main(int argc, char **argv)
{
	void *buffer;
	char *filename;
	u_int32_t crc, size;
	u_int64_t offset;
	int fd;

	if (argc < 4) {
		printf("Usage: %s <filename> <starting_offset_in_bytes> "
		       "<size_in_bytes>\n", argv[0]);
		return 1;
	}

	filename = argv[1];

	fd = open(filename, O_RDONLY);
	if (fd < 0) {
		printf("Error: Could not open %s.\n", filename);
		return 2;
	}

	offset = strtoull(argv[2], NULL, 0);
	size = strtoul(argv[3], NULL, 0);

	buffer = calloc(1, size+1);
	if (!buffer) {
		printf("Error: Could not allocate %u byte buffer.\n", size);
		return 3;
	}

	if (lseek64(fd, offset, SEEK_SET) != offset) {
		printf("Error: Could not seek to offset %lu in %s.\n",
		       (unsigned long)offset, filename);
		return 4;
	}

	if (read(fd, buffer, size) != size) {
		printf("Error: Could not read %u bytes from %s.\n",
		       size, filename);
		return 5;
	}

	crc = calc_crc(INITIAL_CRC, buffer, size);

	printf("CRC: %u (0x%x)\n", crc, crc);

	return 0;
}

