/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */ 

#include <sys/types.h>
#include <sys/wait.h>
#include <ctype.h>
#include <string.h>
#include <sys/mount.h>
#include <unistd.h>
#include <frontend.h>
#include <glib.h>
#include <ncurses.h>
#include <panel.h>

#include "common.h"
#include "window.h"
#include "menu.h"
#include "enum.h"
#include "dialog.h"
#include "clist.h"
#include "selwin.h"
#include "views.h"
#include "mount.h"
#include "logging.h"

/**
 *	filter_volume_for_mount_operation - keep the volumes that can perform given mount operation
 *	@handle: the volume handle
 *	@user_data: contains the mount operation code (MOUNT,REMOUNT,UNMOUNT)
 *
 *	This routine is a standard clist_filter_func function type that checks to see
 *	if the given volume can have the given mount/remount/unmount operation performed
 *	on it.
 */
int filter_volume_for_mount_operation(object_handle_t handle, void *user_data)
{
	int rc;
	mount_operation operation = GPOINTER_TO_UINT(user_data);

	switch (operation) {
	case MOUNT:
		rc = evms_can_mount(handle);
		break;
	case UNMOUNT:
		rc = evms_can_unmount(handle);
		break;
	case REMOUNT:
		rc = evms_can_remount(handle);
		break;
	default:
		rc = EINVAL;
		log_error("%s: Unknown mount operation %d received.\n", __FUNCTION__, operation);
		break;
	}
	return rc;	
}

/**
 *	get_mount_operation_text - return text for the given mount operation
 *	@operation: the mount operation (MOUNT,UNMOUNT,REMOUNT)
 *	@title: the address of the string pointer for the window title
 *	@action: the address of the string pointer for the action button text
 *	@alt_action: the address of the string pointer for the alternative action button text
 *
 *	This routine simply returns the text for the dialog for a certain type of mount operation.
 */
inline void get_mount_operation_text(mount_operation operation, char **title,
				char **action, char **alt_action)
{
	switch (operation) {
	case MOUNT:
		*action = _("_Mount");
		*title = _("Mount File System");
		*alt_action = _("Mount _Options");
		break;
	case REMOUNT:
		*action = _("_Remount");
		*title = _("Remount File System");
		*alt_action = _("Remount _Options");
		break;
	case UNMOUNT:
		*action = _("_Unmount");
		*title = _("Unmount File System");
		*alt_action = NULL;
		break;
	default:
		*action = "";
		*title = "";
		*alt_action = NULL;
		break;
	}
}

/**
 *	mount_operation_button_activated - invoke the mount operation
 *	@item: the menu item that was activated
 *
 *	This routine executes the mount operation to attach a filesystem
 *	to a given device, or unattach a filesystem or remount a filesystem.
 */
int mount_operation_button_activated(struct menu_item *item)
{
	int rc;
	object_handle_t volume;
	GHashTable *hash_table;
	mount_operation operation;
	struct selwin *selwin = item->user_data;
	struct dialog_window *dialog = item->user_data;

	hash_table = dialog->user_data;
	volume = get_selected_handle(selwin->clist);
	operation = GPOINTER_TO_UINT(g_hash_table_lookup(hash_table, "operation"));

	switch (operation) {
	case MOUNT:
		rc = evms_mount(volume,
			g_hash_table_lookup(hash_table, "mountpoint"),
			g_hash_table_lookup(hash_table, "options"));
		update_status(_("Mount"), rc);
		break;
	case UNMOUNT:
		rc = evms_unmount(volume);
		update_status(_("Unmount"), rc);
		break;
	case REMOUNT:
		rc = evms_remount(volume, g_hash_table_lookup(hash_table, "options"));
		update_status(_("Remount"), rc);
		break;
	default:
		rc = EINVAL;
		log_error("%s: Unexpected operation %d received.\n", __FUNCTION__, operation);
		break;
	}
	if (rc == 0) {
		dialog->status = DLG_STATUS_CLOSING;
		refresh_views();
	}
	return 0;
}

/**
 *	get_mount_options - prompt the user for mount command options
 *	@prev_options: the string containing any previous options or NULL if none
 *
 *	This routine presents a dialog for the user to be able to enter
 *	command line options for the mount command.
 */
char *get_mount_options(char *prev_options)
{
	char *options;
	int starty, len;
	WINDOW *win, *border_win;
	PANEL *panel, *border_panel;

	options = g_malloc0(EVMS_VOLUME_NAME_SIZE + 1);

	panel = create_centered_popup_window(getmaxy(stdscr) - 8, getmaxx(stdscr) - 10, WHITE_BKGD);
	border_panel = (PANEL *)panel_userptr(panel);
	win = panel_window(panel);
	border_win = panel_window(border_panel);

	if (prev_options != NULL) {
		char *prev_opts;

		starty = getmaxy(win) / 2 - 2;
		prev_opts = g_strdup_printf(_("Previous Mount Options: %s"), prev_options);
		print_centered(win, starty, prev_opts);
		g_free(prev_opts);
	}

	starty = getmaxy(win) / 2;
	len = strlen(_("Options:: "));

	print_centered(border_win, 0, _("Enter command line options for mount command"));
	mvwaddnstr(win, starty, 2, _("Options:: "), len);

	show_popup_window(panel);

	if (read_string(win, starty, len + 2, options, EVMS_VOLUME_NAME_SIZE) != OK) {
		g_free(options);
		options = NULL;
	}
	delete_popup_window(panel);

	return options;
}

/**
 *	mount_options_button_activated - get mount/remount options
 *	@item: the menu item that was activated
 *
 *	This routine displays a dialog to get additional arguments that
 *	will be passed to the mount/remount command.
 */
int mount_options_button_activated(struct menu_item *item)
{
	GHashTable *hash_table;
	char *options, *prev_options;
	struct dialog_window *dialog = item->user_data;

	hash_table = dialog->user_data;
	prev_options = g_hash_table_lookup(hash_table, "options");
	options = get_mount_options(prev_options);
	if (options != NULL) {
		g_hash_table_insert(hash_table, "options", options);
		g_free(prev_options);
	}

	return (options != NULL ? 0 : ENODATA);
}

/**
 *	get_mount_point - prompt the user for the name for an EVMS volume
 *	@old_dir: if non-NULL, the previous mount point
 *
 *	This routine prompts the user for a directory name for the mount point.
 */
char *get_mount_point(char *old_dir)
{
	char *dir;
	int starty, len;
	WINDOW *win, *border_win;
	PANEL *panel, *border_panel;

	dir = g_malloc0(EVMS_VOLUME_NAME_SIZE + 1);

	panel = create_centered_popup_window(getmaxy(stdscr) - 8, getmaxx(stdscr) - 10, WHITE_BKGD);
	border_panel = (PANEL *)panel_userptr(panel);
	win = panel_window(panel);
	border_win = panel_window(border_panel);

	if (old_dir != NULL) {
		char *prev_dir;

		starty = getmaxy(win) / 2 - 2;
		prev_dir = g_strdup_printf(_("Previous Mount Point: %s"), old_dir);
		print_centered(win, starty, prev_dir);
		g_free(prev_dir);
	}

	starty = getmaxy(win) / 2;
	len = strlen(_("Mount point:: "));

	print_centered(border_win, 0, _("Enter Mount Point Directory"));
	mvwaddnstr(win, starty, 2, _("Mount point:: "), len);

	show_popup_window(panel);

	if (read_string(win, starty, len + 2, dir, EVMS_VOLUME_NAME_SIZE) != OK) {
		g_free(dir);
		dir = NULL;
	}
	delete_popup_window(panel);

	return dir;
}

/**
 *	volume_to_mount_selected - callback activated when volume for mount selected
 *	@clist: the clist for the row item selected
 *	@item: the clist item that was selected
 *
 *	This routine is invoked when a volume in the clist is selected that contains
 *	a file system to mount. We prompt for a mount point and store that in the
 *	hash table associated with the dialog replacing any previous mount point.
 */
int volume_to_mount_selected(struct clist *clist, struct clist_item *item)
{
	GHashTable *hash_table;
	char *mountpoint, *prev_mountpoint;
	struct dialog_window *dialog = clist->user_data;

	hash_table = dialog->user_data;
	prev_mountpoint = g_hash_table_lookup(hash_table, "mountpoint");
	mountpoint = get_mount_point(prev_mountpoint);
	if (mountpoint != NULL) {
		row_selected(clist, item);
		g_hash_table_insert(hash_table, "mountpoint", mountpoint);
		g_free(prev_mountpoint);
	}

	return (mountpoint != NULL ? 0 : ENODATA);
}

/**
 *	on_delete_mount_operation_dialog - callback invoked when dialog is to be deleted
 *	@dialog: the mount operation dialog
 *
 *	This routine is invoked when the mount operation dialog is deleted. We
 *	take care of freeing any dynamically allocated entries in the hash table
 *	before freeing the hash table associated with the dialog.
 */
void on_delete_mount_operation_dialog(struct dialog_window *dialog)
{
	GHashTable *hash_table = dialog->user_data;

	if (hash_table != NULL) {
		g_free(g_hash_table_lookup(hash_table, "options"));
		g_free(g_hash_table_lookup(hash_table, "mountpoint"));
		g_hash_table_destroy(hash_table);
	}
}

/**
 *	show_mount_operation_dialog - display a dialog to allow various mount operations
 *	@handle: non-zero volume handle when invoked from context popup
 *	@operation: the mount operation code (MOUNT,REMOUNT,UNMOUNT)
 *
 *	This routine creates the dialog that displays one or more volumes that can have
 *	the given mount/unmount/remount operation performed on them.
 */
int show_mount_operation_dialog(object_handle_t handle, mount_operation operation)
{
	struct selwin *selwin;
	GHashTable *hash_table;
	struct dialog_window *dialog;
	char *title, *action, *alt_action;

	hash_table = g_hash_table_new(g_str_hash, g_str_equal);
	g_hash_table_insert(hash_table, "operation", GUINT_TO_POINTER(operation));

	get_mount_operation_text(operation, &title, &action, &alt_action);	
	
	selwin = create_selection_window(title,
					NULL, NULL,
					action,
					(menuitem_activate_cb)mount_operation_button_activated,
					alt_action,
					(menuitem_activate_cb)mount_options_button_activated,
					hash_table);

	dialog = (struct dialog_window *)selwin;
	
	set_dialog_delete_cb(dialog, (dialog_delete_cb)on_delete_mount_operation_dialog);

	print_clist_column_title(selwin->clist, 0, " ");
	print_clist_column_title(selwin->clist, 1, _("Volume"));
	print_clist_column_title(selwin->clist, 2, _("Size"));

	if (handle == 0)
		clist_populate(selwin->clist, enum_volumes,
				filter_volume_for_mount_operation,
				format_standard_item, NULL,
				GUINT_TO_POINTER(operation));
	else
		clist_populate_handle(selwin->clist, handle,
				format_standard_item, NULL, NULL);

	if (operation == UNMOUNT)
		set_menu_item_visibility(dialog->prev_button, FALSE);
	else if (operation == MOUNT)
		set_clist_select_item_cb(selwin->clist,
					(clist_select_item_cb)volume_to_mount_selected);

	if (g_list_length(selwin->clist->choices) == 1)
		select_item(selwin->clist, selwin->clist->choices->data);

	process_modal_dialog(dialog);

	return 0;
}

/**
 *	context_mount_filesystem_menuitem_activated - mount a file system on the supplied volume
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is called to mount a filesystem for a certain volume on
 *	a certain mount point.
 */
int context_mount_filesystem_menuitem_activated(struct menu_item *item)
{
	return show_mount_operation_dialog(GPOINTER_TO_UINT(item->user_data), MOUNT);
}

/**
 *	actions_mount_filesystem_menuitem_activated - mount a file system on a volume
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is invoked by the Actions->File System->Mount to mount one
 *	of any volumes that can be mounted.
 */
int actions_mount_filesystem_menuitem_activated(struct menu_item *item)
{
	return show_mount_operation_dialog(0, MOUNT);
}

/**
 *	context_unmount_filesystem_menuitem_activated - unmount a file system on the supplied volume
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is called to unmount a filesystem for a certain volume.
 */
int context_unmount_filesystem_menuitem_activated(struct menu_item *item)
{
	return show_mount_operation_dialog(GPOINTER_TO_UINT(item->user_data), UNMOUNT);
}

/**
 *	actions_unmount_filesystem_menuitem_activated - unmount a file system
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is invoked by the Actions->File System->Unmount to unmount one
 *	of any volumes that can be unmounted.
 */
int actions_unmount_filesystem_menuitem_activated(struct menu_item *item)
{
	return show_mount_operation_dialog(0, UNMOUNT);
}

/**
 *	context_remount_filesystem_menuitem_activated - remount a file system on the supplied volume
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is called to remount a filesystem for a certain volume on
 *	a certain mount point.
 */
int context_remount_filesystem_menuitem_activated(struct menu_item *item)
{
	return show_mount_operation_dialog(GPOINTER_TO_UINT(item->user_data), REMOUNT);
}

/**
 *	actions_remount_filesystem_menuitem_activated - remount a file system on a volume
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is invoked by the Actions->File System->Remount to remount one
 *	of any volumes that can be re-mounted.
 */
int actions_remount_filesystem_menuitem_activated(struct menu_item *item)
{
	return show_mount_operation_dialog(0, REMOUNT);
}
