/*
 *   (C) Copyright IBM Corp. 2004
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: Template Plugin
 * File: evms2/engine/plugins/template/template.c
 *
 * This file is a template for an EVMS plugin. It contains stubs for all of
 * the possible APIs that a plugin can support, along with comments about what
 * each API should do. If an API is marked OPTIONAL, it isn't necessary to
 * support the API, and the function stub can be deleted (along with the
 * corresponding entry in the function table at the end of the file). If an
 * API is *not* marked OPTIONAL, you *must* provide code for that API (even
 * if it's an empty function).
 *
 * Before starting a new plugin, it's a good idea to review some of the
 * global header files in the top-level "include" directory. Of particular
 * interest to plugin-writers are: common.h, enginestructs.h, plugin.h,
 * plugfuncs.h, devmapper.h, and options.h.
 *
 * You should also read the Architecture Overview, available on the EVMS
 * website at http://evms.sourceforge.net/architecture/.
 *
 * You can start writing your plugin by first copying the "template" directory
 * to another directory (within the "plugins" directory). Then add an
 * appropriate entry to the bottom of the top-level configure.ac file. Then
 * run "autoconf" and "./configure" in the top-level to generate a Makefile
 * for your directory. Initially, your plugin will not be built as part of the
 * normal engine build process. You can build your plugin by running "make"
 * within your plugin directory, and install it with "make install".
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include "template.h"

/* Pointer to the Engine-services API set. Must be called "EngFncs" in order
 * for the LOG_ macros to work.
 */
engine_functions_t *EngFncs;

/**
 * template_setup_evms_plugin
 *
 * Perform any global setup required when this plugin is loaded. For example,
 * allocate any global lists or data structures or register a namespace.
 *
 * The "EngFncs" pointer must be initialized first.
 *
 * Return:
 * - 0 for success.
 * - E_NOLOAD if the plugin should be unloaded.
 * - Other non-zero error-code for failure.
 **/
static int template_setup_evms_plugin(engine_functions_t *functions)
{
	EngFncs	= functions;

	LOG_ENTRY();

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * template_cleanup_evms_plugin
 *
 * Perform any global cleanup required when this plugin is unloaded. For
 * example, free any global data structures or lists.
 *
 * OPTIONAL
 **/
static void template_cleanup_evms_plugin(void)
{
	LOG_ENTRY();

	LOG_EXIT_VOID();
}

/**
 * template_can_add_feature
 *
 * Can you consume the input_object and create a new feature object?
 *
 * Return:
 * - 0 for yes, and also set the size of the object you would create on top of
 *   this input_object.
 * - Non-zero error-code for no.
 *
 * OPTIONAL. This API is normally only used by feature-plugins.
 **/
static int template_can_add_feature(storage_object_t *input_object,
				    sector_count_t *size)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_can_delete
 *
 * Can you delete this object? This will almost always be "yes", but
 * occasionally one object might need to wait for another object to be
 * deleted first, or an object might not be deletable at all (e.g. if
 * it's an automatically generated metadata or freespace object).
 *
 * Return:
 * - 0 for yes.
 * - Non-zero error-code for no.
 *
 * OPTIONAL
 **/
static int template_can_delete(storage_object_t *object)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_can_unassign
 *
 * Can you unassign your plugin from this object? A lot like can_delete, but
 * this is asking about a consumed object instead of a produced object.
 *
 * Return:
 * - 0 for yes.
 * - Non-zero error-code for no.
 *
 * OPTIONAL. This API is normally only used by segment-manager plugins.
 **/
static int template_can_unassign(storage_object_t *object)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_can_expand
 *
 * Can you expand this object?  If yes, build an expand_object_info_t and add
 * it to the expand_points list. The specified expand_limit is the maximum
 * change that is allowed for this object. If you cannot expand by an amount
 * less than or equal to this limit, then your object should not be added to
 * the expand_points list.
 *
 * If you can't expand directly, but can allow one of your child objects to
 * expand, call can_expand on whichever child you will allow to expand. If you
 * can not handle expanding below you, do not pass the command down to your
 * child.
 *
 * Return:
 * - Return 0, even if you can't expand and didn't add anything to the list.
 *   The engine will determine success or failure based on the contents of the
 *   expand_points list.  Return non-zero if something went wrong in the
 *   processing of the command, such as a failure to allocate memory.
 *
 * OPTIONAL
 **/
static int template_can_expand(storage_object_t *object,
			       sector_count_t expand_limit,	/* a delta size */
			       list_anchor_t expand_points)	/* of type expand_object_info_t */
{
	LOG_ENTRY();

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * template_can_expand_by
 *
 * Can you allow your child object to expand by "size"? "size" is the change
 * in size, not the resulting size.
 *
 * Return:
 * - 0 for yes. Update "size" if your object would expand by a different
 *   delta size when your child object expands by the given size.
 * - Non-zero error-code for no.
 *
 * OPTIONAL
 **/
static int template_can_expand_by(storage_object_t *object,
				  sector_count_t *size)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_can_shrink
 *
 * Can you shrink this object?  If yes, build a shrink_object_info_t and add
 * it to the shrink_points list. The specified shrink_limit is the maximum
 * change that is allowed for this object. If you cannot shrink by an amount
 * less than or equal to this limit, then your object should not be added to
 * the shrink_points list.
 *
 * If you can't shrink directly, but can allow one of your child objects to
 * shrink, call can_shrink on whichever child you will allow to shrink. If you
 * can not handle shrinking below you, do not pass the command down to your
 * child.
 *
 * Return:
 * - Return 0, even if you can't shrink and didn't add anything to the list.
 *   The engine will determine success or failure based on the contents of the
 *   shrink_points list.  Return non-zero if something went wrong in the
 *   processing of the command, such as a failure to allocate memory.
 *
 * OPTIONAL
 **/
static int template_can_shrink(storage_object_t *object,
			       sector_count_t shrink_limit,	/* a delta size */
			       list_anchor_t shrink_points)	/* of type shrink_object_info_t */
{
	LOG_ENTRY();

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * template_can_shrink_by
 *
 * Can you allow your child object to shrink by "size"? "size" is the change
 * in size, not the resulting size.
 *
 * Return:
 * - 0 for yes. Update "size" if your object would shrink by a different
 *   delta size when your child object shrinks by the given size.
 * - Non-zero error-code for no.
 *
 * OPTIONAL
 **/
static int template_can_shrink_by(storage_object_t *object,
				  sector_count_t *size)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_can_replace_child
 *
 * Can this object's child be replaced? If new_child is NULL, the question is
 * simply: can the child object be replaced? If new_child is not NULL, the
 * question is: can you replace this child with new_child?
 *
 * Return:
 * - 0 for yes.
 * - Non-zero error-code for no.
 *
 * OPTIONAL
 **/
static int template_can_replace_child(storage_object_t *object,
				      storage_object_t *child,
				      storage_object_t *new_child)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_can_set_volume
 *
 * Can this object to be made into a volume (flag == TRUE), or have the volume
 * removed from it (flag == FALSE)? The answer should almost always be yes.
 *
 * Return:
 * - 0 for yes.
 * - Non-zero error-code for no.
 *
 * OPTIONAL
 **/
static int template_can_set_volume(storage_object_t *object,
				   boolean flag)
{
	LOG_ENTRY();

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * template_discover
 *
 * Probe each object on the input_objects list for metadata belonging to your
 * plugin. Claim objects by removing them from the input list. Create a
 * storage_object_t for the new object(s) you are discovering. Fill in the
 * appropriate fields and put the new object(s) on the output_objects list.
 * If you do not claim an object from the input list, then just move it to the
 * output list. The input list can be modified at will (it will be discarded
 * after the call to this discover routine). The output list must contain all
 * the storage objects in the system after yours are discovered, i.e., it is
 * the input list, minus the objects you claim, plus the objects you produce.
 *
 * The overall discovery process is iterative, and thus this discover routine
 * can be called multiple times during the overall process, each time with a
 * potentially different set of input objects. When final_call == TRUE, that
 * will be the last time your discover routine will be called, and you should
 * perform any extra actions that might be required by your plugin.
 *
 * Return:
 * - Number of newly created objects you added to the output_objects list.
 **/
static int template_discover(list_anchor_t input_objects,
			     list_anchor_t output_objects,
			     boolean final_call)
{
	storage_object_t *child;
	list_element_t itr;

	LOG_ENTRY();

	LIST_FOR_EACH(input_objects, itr, child) {
		/* Use the READ() macro to probe the child object for
		 * metadata belonging to your plugin. If you find your
		 * metadata, create any appropriate new objects and use
		 * EngFncs->insert_thing() to add your new objects to the
		 * output_list. If an input object doesn't contain your
		 * metadata, simply insert it on the output list.
		 */

		EngFncs->insert_thing(output_objects, child, 0, NULL);
	}

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * template_create
 *
 * Create new storage_object_t(s) from the list of input_objects using the
 * given options. Place the newly allocated storage_object_t(s) on the
 * new_objects list.
 *
 * Return:
 * - 0 for success.
 * - Non-zero error-code for failure.
 *
 * OPTIONAL
 **/
static int template_create(list_anchor_t input_objects,
			   option_array_t *options,
			   list_anchor_t output_objects)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_assign
 *
 * Assign your plugin to produce storage objects from the given storage object.
 * This API is a lot like create, but you only get one input object, and the
 * output list is assumed to be the parent_objects list of this input_object.
 *
 * Return:
 * - 0 for success.
 * - Non-zero error-code for failure.
 *
 * OPTIONAL. This API is normally only used by segment managers to assign
 * themselves to disks (or other segments).
 **/
static int template_assign(storage_object_t *object,
			   option_array_t *options)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_discard
 *
 * Forget about these objects. Don't delete them. Just clean up any data
 * structures you may have associated with them. The Engine will call to
 * deactivate the objects during commit.
 *
 * Return:
 * - 0 for success.
 * - Non-zero error-code for failure.
 **/
static int template_discard(list_anchor_t objects)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_delete
 *
 * Delete the object. Free any privately allocated data. Remove your parent
 * pointer from your child objects. Do any cleanup necessary to remove your
 * plug-in from your child objects. Put your object's children from the object's
 * child_objects list_anchor_t onto the list_anchor_t provided in the second
 * parameter. Call the Engine's free_?????t() to free the object.
 *
 * Return:
 * - 0 for success.
 * - Non-zero error-code for failure.
 *
 * OPTIONAL
 **/
static int template_delete(storage_object_t *object,
			   list_anchor_t child_objects)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_unassign
 *
 * Unassign your plugin from producing storage objects from the given storage
 * object. A lot like delete.
 *
 * Return:
 * - 0 for success.
 * - Non-zero error-code for failure.
 *
 * OPTIONAL. This API is normally only used by segment-manager plugins that are
 * assigned and removed from disks (or other segments).
 **/
static int template_unassign(storage_object_t *object)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_expand
 *
 * Expand a storage object. If the "object" is not the "expand_object", then
 * your child is going to expand. Do any necessary work to get ready for your
 * child to expand (e.g., read in metadata), then call expand() on your child
 * object which will expand. Upon return from the call to your child's
 * expand(), do any work necessary to adjust this object to account for the
 * child object's new size (e.g., update the location of metadata).
 *
 * If the "object" is the same as the "expand_object", then this is the object
 * targeted for expanding. Expand the object according to the input_objects
 * given and the options selected.
 *
 * Return:
 * - 0 for success.
 * - Non-zero error-code for failure.
 *
 * OPTIONAL
 **/
static int template_expand(storage_object_t *object,
			   storage_object_t *expand_object,
			   list_anchor_t input_objects,
			   option_array_t *options)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_shrink
 *
 * Shrink a storage object. If the "object" is not the "shrink_object", then
 * your child is going to shrink. Do any necessary work to get ready for your
 * child to shrink (e.g., read in metadata), then call shrink() on your child
 * object which will shrink. Upon return from the call to your child's
 * shrink(), do any work necessary to adjust this object to account for the
 * child object's new size (e.g., update the location of metadata).
 *
 * If the "object" is the same as the "shrink_object", then this is the object
 * targeted for shrinking. Shrink the object according to the input_objects
 * given and the options selected.
 *
 * Return:
 * - 0 for success.
 * - Non-zero error-code for failure.
 *
 * OPTIONAL
 **/
static int template_shrink(storage_object_t *object,
			   storage_object_t *shrink_object,
			   list_anchor_t input_objects,
			   option_array_t *options)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_replace_child
 *
 * Replace the object's child with the new child object.
 *
 * Return:
 * - 0 for success.
 * - Non-zero error-code for failure.
 *
 * OPTIONAL
 **/
static int template_replace_child(storage_object_t *object,
				  storage_object_t *child,
				  storage_object_t *new_child)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_set_volume
 *
 * This call notifies you that your object is being made into (or part of) a
 * volume or that your object is no longer part of a volume.  The "flag"
 * parameter indicates whether the volume is being created (TRUE) or removed
 * (FALSE).
 *
 * Return: No return value.
 *
 * OPTIONAL
 **/
static void template_set_volume(storage_object_t *object,
				boolean flag)
{
	LOG_ENTRY();

	LOG_EXIT_VOID();
}

/**
 * template_add_sectors_to_kill_list
 *
 * The kill-list is a list of sectors that need to be zeroed at commit-time.
 * Translate the lsn and count into lsn(s) and count(s) for your child
 * object(s) and call the child object's add_sectors_to_kill_list(). The
 * Disk Manager (at the bottom of the object stack) calls the Engine's
 * add_sectors_to_kill_list service to put the sectors on the Engine's kill
 * list.
 *
 * Very similar to read and write, but without the data buffer.
 *
 * Return:
 * - 0 for success.
 * - Non-zero error-code for failure.
 **/
static int template_add_sectors_to_kill_list(storage_object_t *object,
					     lsn_t lsn,
					     sector_count_t count)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_commit_changes
 *
 * Write your plugin's metadata to disk. Clear the SOFLAG_DIRTY in the
 * storage_object_t(s). Committing changes in done in several phases.
 * "phase" says which phase of the commit is being performed.
 *
 * Write your first copy of metadata during the FIRST_METADATA_WRITE
 * phase; write your second copy of metadata (if you have one) during
 * the SECOND_METADATA_WRITE phase. Use the SETUP and POST_ACTIVATE
 * phases for appropriate setup/cleanup before/after all the metadata
 * has been written.
 *
 * Return:
 * - 0 for success.
 * - Non-zero error-code for failure.
 **/
static int template_commit_changes(storage_object_t *object,
				   commit_phase_t phase)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_can_activate
 *
 * Can you activate this object?
 *
 * Return:
 * - 0 for success.
 * - Non-zero error-code for failure.
 *
 * OPTIONAL
 **/
static int template_can_activate(storage_object_t *object)
{
	LOG_ENTRY();

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * template_activate
 *
 * Activate the specified storage object. This should be done by communicating
 * with the appropriate kernel driver. If your plugin uses Device-Mapper to
 * activate your storage objects (hopefully it does), use the DM services
 * provided by the engine to assist in creating the target mappings and
 * creating/activating the device.
 *
 * Return:
 * - 0 for success.
 * - Non-zero error-code for failure.
 *
 * OPTIONAL
 **/
static int template_activate(storage_object_t *object)
{
	LOG_ENTRY();

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * template_can_deactivate
 *
 * Can you deactivate this object?
 *
 * Return:
 * - 0 for success.
 * - Non-zero error-code for failure.
 *
 * OPTIONAL
 **/
static int template_can_deactivate(storage_object_t *object)
{
	LOG_ENTRY();

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * template_deactivate
 *
 * Deactivate the specified storage object. Just like activate, this should be
 * done by communicating with the appropriate kernel driver.
 *
 * Return:
 * - 0 for success.
 * - Non-zero error-code for failure.
 *
 * OPTIONAL
 **/
static int template_deactivate(storage_object_t *object)
{
	LOG_ENTRY();

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * template_get_option_count
 *
 * Get the total number of supported options for the specified task.
 *
 * Return:
 * - Number of options available for this task.
 * - -1 if the task is not supported.
 *
 * OPTIONAL
 **/
static int template_get_option_count(task_context_t *context)
{
	LOG_ENTRY();

	LOG_EXIT_INT(-1);
	return -1;
}

/**
 * template_init_task
 *
 * Initialize a user-task with the appropriate information for your plugin.
 * Fill in the initial list of acceptable objects. Fill in the minimum and
 * maximum number of objects that must/can be selected. Set up all initial
 * values in the option_descriptors in the context record for the given task.
 * Some fields in the option_descriptor may be dependent on a selected object.
 * Leave such fields blank for now, and fill in during the set_objects call.
 *
 * Return:
 * - 0 for success.
 * - Non-zero error-code for failure.
 *
 * OPTIONAL
 **/
static int template_init_task(task_context_t *context)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_set_option
 *
 * Examine the specified value, and determine if it is valid for the task and
 * option_descriptor index. If it is acceptable, set that value in the
 * appropriate entry in the option_descriptor. The value may be adjusted if
 * necessary/allowed. If so, set the effect return value accordingly.
 *
 * Return:
 * - 0 for success.
 * - Non-zero error-code for failure.
 *
 * OPTIONAL
 **/
static int template_set_option(task_context_t *context,
			       u_int32_t index,
			       value_t *value,
			       task_effect_t *effect)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_set_object
 *
 * Validate the objects in the selected_objects list in the task context.
 * Remove from the selected objects lists any objects which are not acceptable.
 * For unacceptable objects, create a declined_handle_t structure with the
 * reason why it is not acceptable, and add it to the declined_objects list.
 * Modify the acceptable_objects list in the task context as necessary based on
 * the selected objects and the current settings of the options. Modify any
 * option settings as necessary based on the selected objects. Return the
 * appropriate task_effect_t settings if the object list(s), minimum or maximum
 * objects selected, or option settings have changed.
 *
 * Return:
 * - 0 for success.
 * - Non-zero error-code for failure.
 *
 * OPTIONAL
 **/
static int template_set_objects(task_context_t *context,
				list_anchor_t declined_objects,	 /* of type declined_handle_t */
				task_effect_t *effect)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_get_info
 *
 * Return any additional information that you wish to provide about the object.
 * The Engine provides an external API to get the information stored in the
 * storage_object_t. This call is to get any other information about the object
 * that is not specified in the storage_object_t. Any piece of information you
 * wish to provide must be in an extended_info_t structure. Use the Engine's
 * engine_alloc() to allocate the memory for the extended_info_t. Also use
 * engine_alloc() to allocate any strings that may go into the extended_info_t.
 * Then use engine_alloc() to allocate an extended_info_array_t with enough
 * entries for the number of extended_info_t structures you are returning. Fill
 * in the array and return it in *info.
 *
 * If you have extended_info_t descriptors that themselves may have more
 * extended information, set the EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE flag in
 * the extended_info_t flags field. If the caller wants more information about
 * a particular extended_info_t item, this API will be called with a pointer to
 * the storage_object_t and with a pointer to the name of the extended_info_t
 * item. In that case, return an extended_info_array_t with further information
 * about the item. Each of those items may have the
 * EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE flag set if you desire. It is your
 * responsibility to give the items unique names so that you know which item the
 * caller is asking additional information for. If info_name is NULL, the caller
 * just wants top level information about the object.
 *
 * Return:
 * - 0 for success.
 * - Non-zero error-code for failure.
 *
 * OPTIONAL
 **/
static int template_get_info(storage_object_t *object,
			     char *info_name,
			     extended_info_array_t **info)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_set_info
 *
 * Apply the settings of the options to the given object.
 *
 * Return:
 * - 0 for success.
 * - Non-zero error-code for failure.
 *
 * OPTIONAL
 **/
static int template_set_info(storage_object_t *object,
			     option_array_t *options)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_get_plugin_info
 *
 * Return any additional information that you wish to provide about your plugin.
 * The Engine provides an external API to get the information stored in the
 * plugin_record_t. This call is to get any other information about the plugin
 * that is not specified in the plugin_record_t. Any piece of information you
 * wish to provide must be in an extended_info_t structure. Use the Engine's
 * engine_alloc() to allocate the memory for the extended_info_t.  Also use
 * engine_alloc() to allocate any strings that may go into the extended_info_t.
 * Then use engine_alloc() to allocate an extended_info_array_t with enough
 * entries for the number of extended_info_t structures you are returning. Fill
 * in the array and return it in *info.
 *
 * If you have extended_info_t descriptors that themselves may have more
 * extended information, set the EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE flag in
 * the extended_info_t flags field. If the caller wants more information about
 * a particular extended_info_t item, this API will be called with a pointer to
 * the name of the extended_info_t item. In that case, return an
 * extended_info_array_t with further information about the item. Each of those
 * items may have the EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE flag set if you
 * desire. It is your responsibility to give the items unique names so that you
 * know which item the caller is asking additional information for. If
 * info_name is NULL, the caller just wants top level information about the
 * plugin.
 *
 * Return:
 * - 0 for success.
 * - Non-zero error-code for failure.
 *
 * OPTIONAL
 **/
static int template_get_plugin_info(char *info_name,
				    extended_info_array_t **info)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_read
 *
 * Convert lsn and count to lsn and count on the child object(s) and call the
 * read function of child objects (using the READ() macro).
 *
 * There is no limit on how big a single read request can be. If your object
 * has internal boundaries (e.g. chunks in a striped object), you may need to
 * break up the request into multiple smaller requests before submitting them
 * to your child objects.
 *
 * Return:
 * - 0 for success.
 * - Non-zero error-code for failure.
 **/
static int template_read(storage_object_t *object,
			 lsn_t lsn,
			 sector_count_t count,
			 void *buffer)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_write
 *
 * Convert lsn and count to lsn and count on the child object(s) and call the
 * write function of child objects (using the WRITE() macro).
 *
 * There is no limit on how big a single write request can be. If your object
 * has internal boundaries (e.g. chunks in a striped object), you may need to
 * break up the request into multiple smaller requests before submitting them
 * to your child objects.
 *
 * Return:
 * - 0 for success.
 * - Non-zero error-code for failure.
 **/
static int template_write(storage_object_t *object,
			  lsn_t lsn,
			  sector_count_t count,
			  void *buffer)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_get_functions
 *
 * Return an array of plugin functions that you support for this object.
 * Plugin functions are plugin-specific additions to the plugin API set.
 * Any functionality you need to provide for your plugin that isn't
 * covered by the standard API should be implemented as a plugin function.
 * Existing examples: RAID-hot-add, Snapshot-rollback, and LVM PV-move.
 *
 * Return:
 * - 0 for success.
 * - Non-zero error-code for failure.
 *
 * OPTIONAL
 **/
static int template_get_functions(storage_object_t *object,
				  function_info_array_t **actions)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_function
 *
 * Execute a plugin function on the object. The definitions of your plugin's
 * functions were provided by get_plugin_function. This API executes one
 * of those functions.
 *
 * Return:
 * - 0 for success.
 * - Non-zero error-code for failure.
 *
 * OPTIONAL
 **/
static int template_function(storage_object_t *object,
			     task_action_t action,
			     list_anchor_t objects,
			     option_array_t *options)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_backup__metadata
 *
 * Save the metadata needed to build this object. The plug-in can call
 * the Engine's save_metadata() service for each contiguous chunk of
 * metadata that it writes to each child object.
 *
 * Return:
 * - 0 for success
 * - Non-zero error-code for failure.
 **/
static int template_backup_metadata(storage_object_t *object)
{
	LOG_ENTRY();

	LOG_EXIT_INT(0);
	return 0;
}

/*
 * Container APIs. If your plugin doesn't use containers, delete all of
 * the following stubs until the plugin_functions table.
 */

/**
 * template_can_delete_container
 *
 * Can you destroy the container?  You must check to be sure that no objects
 * are exported from this container.
 *
 * Return:
 * - 0 for yes
 * - Non-zero error-code for no.
 **/
static int template_can_delete_container(storage_container_t *container)
{
	LOG_ENTRY();
	
	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_can_expand_container
 *
 * Can you expand this container?  If yes, build an expand_object_info_t
 * and add it to the expand_points list.  If you allow your consumed
 * objects to expand, call can_expand() on whichever consumed objects
 * you will allow to expand.  If you can not handle expanding below you,
 * do not pass the command down to your consumed objects.
 *
 * Return:
 * - Return 0, even if you can't expand and didn't add anything to the list.
 *   The engine will determine success or failure based on the contents of the
 *   expand_points list.  Return non-zero if something went wrong in the
 *   processing of the command, such as a failure to allocate memory.
 **/
static int template_can_expand_container(storage_container_t *container,
					 list_anchor_t expand_points);	/* of type expand_object_info_t, */
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_can_expand_container_by
 *
 * Can you allow the specified consumed objects to expand by "size"?
 * Return 0 if yes, else an error code.  "size" is the delta expand BY
 * size, not the resulting size.  Update the "size" if your container
 * would expand by a different delta size when your consumed object
 * expanded by the given size.
 *
 * Return:
 * - 0 for yes
 * - Non-zero error-code for no.
 **/
static int template_can_expand_container_by(storage_container_t *container,
					    storage_object_t *consumed_object,
					    sector_count_t *size)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_can_shrink_container
 *
 * Can you shrink this container?  If yes, build a shrink_object_info_t
 * and add it to the shrink_points list.  If you allow your consumed
 * objects to shrink, call can_shrink() on whichever consumed objects
 * you will allow to shrink.  If you can not handle shrinking below
 * you, do not pass the command down to your consumed objects.
 *
 * Return:
 * - Return 0, even if you can't shrink and didn't add anything to the list.
 *   The engine will determine success or failure based on the contents of the
 *   shrink_points list.  Return non-zero if something went wrong in the
 *   processing of the command, such as a failure to allocate memory.
 **/
static int template_can_shrink_container(storage_container_t *container,
					 list_anchor_t shrink_points);	/* of type shrink_object_info_t, */
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_can_shrink_container_by
 *
 * Can you allow the specified consumed objects to shrink by "size"?
 * Return 0 if yes, else an error code.  "size" is the delta shrink BY
 * size, not the resulting size.  Update the "size" if your object would
 * shrink by a different delta size when your consumed object shrunk by
 * the given size.
 *
 * Return:
 * - 0 for yes
 * - Non-zero error-code for no.
 **/
static int template_can_shrink_container_by(storage_container_t *container,
					    storage_object_t *consumed_object,
					    sector_count_t *size)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_create_container
 *
 * Create a new container and claim the objects (just as during discovery).
 * Mark the container dirty. Must use allocate_container engine API to
 * allocate the container structure.
 *
 * Return:
 * - 0 for success
 * - Non-zero error-code for failure.
 **/
static int template_create_container(list_anchor_t objects,
				     option_array_t *options,
				     storage_container_t **container)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_expand_container
 *
 * Return:
 * - 0 for success
 * - Non-zero error-code for failure.
 **/
static int template_expand_container(storage_container_t *container,
				     storage_object_t *consumed_object,
				     storage_object_t *expand_object,
				     list_anchor_t input_objects,
				     option_array_t *options)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_shrink_container
 *
 * Return:
 * - 0 for success
 * - Non-zero error-code for failure.
 **/
static int template_shrink_container(storage_container_t *container,
				     storage_object_t *consumed_object,
				     storage_object_t *shrink_object,
				     list_anchor_t input_objects,
				     option_array_t *options)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_discard_container
 *
 * Forget about this container. Don't delete it. Just clean up any data
 * structures you may have associated with it.
 *
 * Return:
 * - 0 for success
 * - Non-zero error-code for failure.
 **/
static int template_discard_container(storage_container_t *container)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_delete_container
 *
 * Destroy this container. Make sure there are no data objects being produced
 * by the container. Put your consumed objects from the container's
 * consumed_objects list onto the list provided in the second parameter. Free
 * any private data, then use the Engine's free_container() to deallocate the
 * container structure.
 *
 * Return:
 * - 0 for success
 * - Non-zero error-code for failure.
 **/
static int template_delete_container(storage_container_t *container,
				     list_anchor_t objects_consumed)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_commit_container_changes
 *
 * Write any container metadata to disk. Clear the SCFLAG_DIRTY in the
 * container.
 *
 * Committing changes in done in several phases. "phase" says which phase of
 * the commit is being performed. Write your first copy of metadata during the
 * FIRST_METADATA_WRITE phase. Write your second copy of metadata (if you have
 * one) during the SECOND_METADATA_WRITE phase.
 *
 * Return:
 * - 0 for success
 * - Non-zero error-code for failure.
 **/
static int template_commit_container_changes(storage_container_t *container,
					     commit_phase_t phase)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_get_container_info
 * Return any additional information that you wish to provide about the
 * container. The Engine provides an external API to get the information
 * stored in the storage_container_t. This call is to get any other
 * information about the container that is not specified in the
 * storage_container_t. Any piece of information you wish to provide must
 * be in an extended_info_t structure.  Use the Engine's engine_alloc() to
 * allocate the memory for the extended_info_t. Also use engine_alloc() to
 * allocate any strings that may go into the extended_info_t.  Then use
 *  engine_alloc() to allocate an extended_info_array_t with enough entries
 * for the number of extended_info_t structures you are returning. Fill in
 * the array and return it in *info.
 *
 * If you have extended_info_t descriptors that themselves may have more
 * extended information, set the EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE flag in
 * the extended_info_t flags field. If the caller wants more information
 * about a particular extended_info_t item, this API will be called with a
 * pointer to the storage_container_t and with a pointer to the name of the
 * extended_info_t item. In that case, return an extended_info_array_t with
 * further information about the item. Each of those items may have the
 * EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE flag set if you desire. It is your
 * responsibility to give the items unique names so that you know which item
 * the caller is asking additional information for. If info_name is NULL, the
 * caller just wants top level information about the object.
 *
 * Return:
 * - 0 for success
 * - Non-zero error-code for failure.
 **/
static int template_get_container_info(storage_container_t *container,
				       char *info_name,
				       extended_info_array_t **info)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_set_container_info
 *
 * Apply the settings of the options to the given container.
 *
 * Return:
 * - 0 for success
 * - Non-zero error-code for failure.
 **/
static int template_set_container_info(storage_container_t *container,
				       option_array_t *options)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_get_container_functions
 *
 * Return an array of plugin functions that you support for this container.
 *
 * Return:
 * - 0 for success
 * - Non-zero error-code for failure.
 *
 * OPTIONAL
 **/
static int template_get_container_functions(storage_container_t *container,
					    function_info_array_t **actions)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_container_function
 *
 * Execute the plug-in function on the container.
 *
 * Return:
 * - 0 for success
 * - Non-zero error-code for failure.
 *
 * OPTIONAL
 **/
static int template_container_function(storage_container_t *container,
				       task_action_t action,
				       list_anchor_t objects,
				       option_array_t *options)
{
	LOG_ENTRY();

	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * template_backup_container_metadata
 *
 * Save the metadata needed to build this container. The plug-in can call
 * the Engine's save_metadata() service for each contiguous chunk of
 * metadata that it writes to each child object.
 *
 * Return:
 * - 0 for success
 * - Non-zero error-code for failure.
 **/
static int template_backup_container_metadata(storage_container_t *container)
{
	LOG_ENTRY();

	LOG_EXIT_INT(0);
	return 0;
}

/*
 * Table of standard plugin APIs for the Template plugin.
 */
static plugin_functions_t template_functions = {
	.setup_evms_plugin		= template_setup_evms_plugin,
	.cleanup_evms_plugin		= template_cleanup_evms_plugin,
	.can_add_feature		= template_can_add_feature,
	.can_delete			= template_can_delete,
	.can_unassign			= template_can_unassign,
	.can_expand			= template_can_expand,
	.can_expand_by			= template_can_expand_by,
	.can_shrink			= template_can_shrink,
	.can_shrink_by			= template_can_shrink_by,
	.can_replace_child		= template_can_replace_child,
	.can_set_volume			= template_can_set_volume,
	.discover			= template_discover,
	.create				= template_create,
	.assign				= template_assign,
	.discard			= template_discard,
	.delete				= template_delete,
	.unassign			= template_unassign,
	.expand				= template_expand,
	.shrink				= template_shrink,
	.replace_child			= template_replace_child,
	.set_volume			= template_set_volume,
	.add_sectors_to_kill_list	= template_add_sectors_to_kill_list,
	.commit_changes			= template_commit_changes,
	.can_activate			= template_can_activate,
	.activate			= template_activate,
	.can_deactivate			= template_can_deactivate,
	.deactivate			= template_deactivate,
	.get_option_count		= template_get_option_count,
	.init_task			= template_init_task,
	.set_option			= template_set_option,
	.set_objects			= template_set_objects,
	.get_info			= template_get_info,
	.set_info			= template_set_info,
	.get_plugin_info		= template_get_plugin_info,
	.read				= template_read,
	.write				= template_write,
	.get_plugin_functions		= template_get_functions,
	.plugin_function		= template_function,
	.backup_metadata		= template_backup_metadata,
};

/*
 * Table of container APIs for the Template plugin. If the plugin does not use
 * containers, simply delete this table.
 */
static container_functions_t template_container_functions = {
	.can_delete_container		= template_can_delete_container,
	.can_expand_container		= template_can_expand_container,
	.can_expand_container_by	= template_can_expand_container_by,
	.can_shrink_container		= template_can_shrink_container,
	.can_shrink_container_by	= template_can_shrink_container_by,
	.create_container		= template_create_container,
	.expand_container		= template_expand_container,
	.shrink_container		= template_shrink_container,
	.discard_container		= template_discard_container,
	.delete_container		= template_delete_container,
	.commit_container_changes	= template_commit_container_changes,
	.get_container_info		= template_get_container_info,
	.set_container_info		= template_set_container_info,
	.get_plugin_functions		= template_get_container_functions,
	.plugin_function		= template_container_function,
	.backup_container_metadata	= template_backup_container_metadata,
};

/*
 * Plugin record for the Template plugin.
 */
plugin_record_t template_plugin = {
	/* Plugin Identifier. Choose the appropriate plugin type and a unique
	 * ID number. OEM ID may be the IBM value, or select your own. See the
	 * top-level PLUGIN.IDS file for more information.
	 */
	.id = SetPluginID(EVMS_OEM_IBM,
			  EVMS_SEGMENT_MANAGER,
			  10),

	/* MAJOR_VERSION, MINOR_VERSION, and PATCH_LEVEL are
	 * defined in the plugin's Makefile.in.
	 */
	.version = {
		.major		= MAJOR_VERSION,
		.minor		= MINOR_VERSION,
		.patchlevel	= PATCH_LEVEL
	},
	/* Don't change these API versions. */
	.required_engine_api_version = {
		.major		= 15,
		.minor		= 0,
		.patchlevel	= 0
	},
	.required_plugin_api_version = {
		.plugin	= {
			.major		= 13,
			.minor		= 1,
			.patchlevel	= 0
		}
	},
	/* Delete this field if the plugin doesn't use containers. */
	.required_container_api_version = {
		.major		= 10,
		.minor		= 1,
		.patchlevel	= 0
	},
	.short_name = "Template",
	.long_name = "Template Segment Manager",
	.oem_name = "IBM",
	.functions = {
		.plugin = &template_functions
	},
	/* Delete this field if the plugin doesn't use containers. */
	.container_functions = &template_container_functions
};

/*
 * Table of all plugin records for this shared object. A shared object can
 * contain multiple plugins (see MD), but most have a single one.
 * "evms_plugin_records" is the well-known symbol name that the engine looks
 * for when dynamically loading the plugins.
 */
plugin_record_t *evms_plugin_records[] = {
	&template_plugin,
	NULL
};

