/*
 *
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: ptables.h
 *
 */

#ifndef   PTABLES_HEADER
#define   PTABLES_HEADER 1

#include <common.h>


typedef storage_object_t LOGICALDISK;
typedef storage_object_t DISKSEG;

extern struct engine_functions_s       *SegEngFncs;
extern plugin_record_t                 *Seg_My_PluginRecord_Ptr;

// these logging macros expect args ....
#define LOG_WARNING(args...) SegEngFncs->write_log_entry(WARNING,Seg_My_PluginRecord_Ptr, ##args)
#define LOG_INFO(args...)    SegEngFncs->write_log_entry(DEFAULT,Seg_My_PluginRecord_Ptr, ##args)
#define LOG_ERROR(args...)   SegEngFncs->write_log_entry(ERROR,Seg_My_PluginRecord_Ptr, ##args)
#define LOG_DEBUG(args...)   SegEngFncs->write_log_entry(DEBUG,Seg_My_PluginRecord_Ptr, ##args)
#define LOGENTRY()           SegEngFncs->write_log_entry(ENTRY_EXIT,Seg_My_PluginRecord_Ptr,"%s: entry\n", __FUNCTION__ )
#define LOGEXIT()            SegEngFncs->write_log_entry(ENTRY_EXIT,Seg_My_PluginRecord_Ptr,"%s: exit\n", __FUNCTION__ )
#define LOGEXITRC()          SegEngFncs->write_log_entry(ENTRY_EXIT,Seg_My_PluginRecord_Ptr,"%s: exit, RC= %d\n", __FUNCTION__ , rc)



// used by option APIs to set string fields
#define SET_STRING_FIELD(a,b)\
a = SegEngFncs->engine_alloc( strlen(b)+1 );\
if (a ) {\
    strcpy(a, b);\
}\
else {\
    LOG_DEBUG("RC= ENOMEM\n");\
    LOGEXIT();\
    return ENOMEM;\
}


#define POPUP_MSG(answer,choices,args...) SegEngFncs->user_message(Seg_My_PluginRecord_Ptr,answer,choices, ##args)


/*    The maximum number of cylinders, heads, and sectors that a partition table entry can accomodate.
 *
 *    Cylinders are numbered 0 - 1023, for a maximum of 1024 cylinders.
 *    Heads are numbered 0 - 254, for a maximum of 255 heads.
 *    Sectors are numbered 1 - 63, for a maximum of 63 sectors per track.
 */
#define MAX_CYLINDERS 1023
#define MAX_HEADS     254
#define MAX_SECTORS   63

/* A DLA Table entry for an OS2 partition size is only 32 bits. So, there is a limit as to the size of an OS2 partition */
#define MAX_SECTORS_FOR_OS2_PARTITION 0xFFFFFFFF

/* must define the largest ebr number well accept. used by fixup_ebr_chain() */
#define MAX_EBR_NUMBER  0x7FFFFFFF

/* The following define is used to hold the value of the Boot_Indicator for active partitions. */
#define ACTIVE_PARTITION   0x80

/* Define the size of a Partition Name.  Partition Names are user defined names given to a partition. */
#define PARTITION_NAME_SIZE  20

/* Define the size of a volume name.  Volume Names are user defined names given to a volume. */
#define VOLUME_NAME_SIZE  20

/* Define the size of a disk name.  Disk Names are user defined names given to physical disk drives in the system. */
#define DISK_NAME_SIZE    20

/* Define the size of a disk drive letter.  Drive letters are the equivaltent of a mount cmd. */
#define DRIVE_LETTER_SIZE    1

/* Needed for CRC calculations */
#define  INITIAL_CRC 0xFFFFFFFFL


/*
 *    OS/2 LVM Drive Letter Assignment Tables (DLA Tables) appear on the
 *    last sector of each track containing a valid MBR or EBR.  Since
 *    partitions must be track aligned, any track containing an MBR or
 *    EBR will be almost all empty sectors.  We will grab the last
 *    of these empty sectors for our DLT_Tables.
 *
 */
#define DLA_TABLE_SIGNATURE1  0x424D5202L
#define DLA_TABLE_SIGNATURE2  0x44464D50L

typedef struct _DLA_Entry { /* DE */
u_int32_t   Volume_Serial_Number;                /* The serial number of the volume that this partition belongs to. */
u_int32_t   Partition_Serial_Number;             /* The serial number of this partition. */
u_int32_t   Partition_Size;                      /* The size of the partition, in sectors. */
u_int32_t   Partition_Start;                     /* The starting sector of the partition. */
BOOLEAN     On_Boot_Manager_Menu;                /* Set to TRUE if this volume/partition is on the Boot Manager Menu. */
BOOLEAN     Installable;                         /* Set to TRUE if this volume is the one to install the operating system on. */
char        Drive_Letter;                        /* The drive letter assigned to the partition. */
char        Reserved;
char        Volume_Name[VOLUME_NAME_SIZE];       /* The name assigned to the volume by the user. */
char        Partition_Name[PARTITION_NAME_SIZE]; /* The name assigned to the partition. */
} DLA_Entry;

typedef struct _DLA_Table_Sector { /* DTS */
u_int32_t      DLA_Signature1;             /* The magic signature (part 1) of a Drive Letter Assignment Table. */
u_int32_t      DLA_Signature2;             /* The magic signature (part 2) of a Drive Letter Assignment Table. */
u_int32_t      DLA_CRC;                    /* The 32 bit CRC for this sector.  Calculated assuming that this field and all unused space in the sector is 0. */
u_int32_t      Disk_Serial_Number;         /* The serial number assigned to this disk. */
u_int32_t      Boot_Disk_Serial_Number;    /* The serial number of the disk used to boot the system. Conflict Resolution */
u_int32_t      Install_Flags;              /* Used by the Install program. */
u_int32_t      Cylinders;
u_int32_t      Heads_Per_Cylinder;
u_int32_t      Sectors_Per_Track;
char           Disk_Name[DISK_NAME_SIZE];  /* The name assigned to the disk containing this sector. */
BOOLEAN        Reboot;                     /* For use by Install.  Used to keep track of reboots initiated by install. */
char           Reserved[3];                /* Alignment. */
DLA_Entry      DLA_Array[4];               /* These are the four entries which correspond to the entries in the partition table. */
} DLA_Table_Sector;


struct partition {
    unsigned char boot_ind;     /* 0x80 - active */
    unsigned char head;     /* starting head */
    unsigned char sector;       /* starting sector */
    unsigned char cyl;      /* starting cylinder */
    unsigned char sys_ind;      /* What partition type */
    unsigned char end_head;     /* end head */
    unsigned char end_sector;   /* end sector */
    unsigned char end_cyl;      /* end cylinder */
    unsigned int start_sect;    /* starting sector counting from 0 */
    unsigned int nr_sects;      /* nr of sectors in partition */
} __attribute__((packed));


typedef  struct partition  Partition_Record;

typedef struct _Master_Boot_Record {
char                Reserved[446];
Partition_Record    Partition_Table[4];
u_int16_t           Signature;
} Master_Boot_Record;

typedef Master_Boot_Record  Extended_Boot_Record;


/*    The following is the signature used for an Master Boot Record,
 *    an Extended Boot Record, and a Boot Sector.
 */
#define MSDOS_DISKMAGIC    0xAA55


/* Some defines for the sys_ind field of the partition record */
#define GPT_PARTITION                      0xee

#define LINUX_LVM_PARTITION                0x8e
#define LINUX_EXT2_PARTITION               0x83
#define LINUX_SWAP_PARTITION               0x82
#define SOLARIS_X86_PARTITION              0x82
#define LINUX_RAID_PARTITION               0xfd

#define LINUX_EXTENDED_PARTITION           0x85
#define DOS_EXTENDED_PARTITION             0x05
#define WIN95_EXTENDED_PARTITION           0x0f
#define WIN98_EXTENDED_PARTITION           0x0f

#define WIN95_FAT32_PARTITION              0x0b
#define HIDDEN_WIN95_FAT32_PARTITION       0x1b

#define WIN95_FAT16_PARTITION              0x0e
#define HIDDEN_WIN95_FAT16_PARTITION       0x1e

#define NT_LVM1_PARTITION                  0x86
#define NT_LVM2_PARTITION                  0x87

#define HPFSNTFS_PARTITION                 0x07
#define HIDDEN_HPFSNTFS_PARTITION          0x17

#define HIDDEN_OS2_PARTITION               0x84
#define OS2_BOOT_MGR_PARTITION             0x0a

#define OS2_LVM_SIGNATURE_PARTITION        0x35
#define HIDDEN_OS2_LVM_SIGNATURE_PARTITION 0x45

#define SMALL_FAT16_PARTITION              0x04
#define HIDDEN_SMALL_FAT16_PARTITION       0x14

#define LARGE_FAT16_PARTITION              0x06
#define HIDDEN_LARGE_FAT16_PARTITION       0x16

#define SMALL_FAT12_PARTITION              0x01
#define HIDDEN_SMALL_FAT12_PARTITION       0x11

#define FREEBSD_PARTITION                  0xa5
#define BSD_PARTITION                      0xa5
#define OPENBSD_PARTITION                  0xa6
#define NETBSD_PARTITION                   0xa9
#define BSDI_PARTITION                     0xb7
#define MINIX_PARTITION                    0x81
#define UNIXWARE_PARTITION                 0x63



#define MBR_PARTITION                      0xff

#define UNUSED_PARTITION_ENTRY             0x00

// Byte fields in a partition record
#define SYS_IND(p)     (p->sys_ind)
#define BOOT_IND(p)    (p->boot_ind)

#define START_SECT(p)  (p->sector)
#define START_CYL(p)   (p->cyl)
#define START_HEADS(p) (p->head)

#define END_SECT(p)    (p->end_sector)
#define END_CYL(p)     (p->end_cyl)
#define END_HEADS(p)   (p->end_head)

// 32bit signed integer fields in a partition record
#define NR_SECTS(p)    (p->nr_sects)
#define START_LBA(p)   (p->start_sect)


#define  DISK_HAS_SWAP_PARTITION                    0x00000001
#define  DISK_HAS_EXTENDED_PARTITION                0x00000002
#define  DISK_HAS_OS2_DLAT_TABLES                   0x00000004
#define  DISK_HAS_FORCED_LBA_ADDRESSING             0x00000008
#define  DISK_HAS_PHYS_ORDERED_LOGICAL_PARTITIONS   0x00000010

#define  DISK_USES_LBA_ADDRESSING                   0x10000000


#define  DEFAULT_SEG_MGR_PDATA_SIGNATURE     0x44736567   /* Dseg */

typedef struct disk_private_data_s {

     u_int32_t                   signature;                /* used to validate disk private data area      */

     void *                      key;                      /* search key, address of the LOGICALDISK       */

     u_int32_t                   flags;                    /* see DISK_HAS_xxxx flags above                */

     u_int32_t                   logical_drive_count;      /* count of logical partitions on the drive     */

     u_int32_t                   embedded_partition_count; /* count of embedded partitions on the drive    */

//     u_int32_t                   freespace_gap_count;      /* freespace gaps are numbered 1-n and this     */
                                                           /* count provides names for the freespace       */
                                                           /* storage objects that will be produced from   */
                                                           /* this disk.                                   */
     char                        disk_name[DISK_NAME_SIZE+1];  /* either from the OS/2 dlat or from the    */
                                                               /* disk storage object.                     */
     lba_t                       extd_partition_lba;       /* starting LBA of the extended partition       */
     lba_t                       extd_partition_end_lba;   /* ending   LBA of the extended partition       */
     u_int64_t                   extd_partition_size;      /* size of the extended partition (sectors)     */
     u_int32_t                   extd_partition_sys_ind;   /* system indicator field from extd part record */

     geometry_t                  geometry;                 /* partition geometry                           */

     dlist_t                     container_segs;           /* list of partitions holding embedded partition*/
                                                           /* schemes.                                     */
} DISK_PRIVATE_DATA;


#define  SEG_IS_PRIMARY_PARTITION               0x00000001
#define  SEG_IS_LOGICAL_PARTITION               0x00000002
#define  SEG_IS_EBR                             0x00000004
#define  SEG_IS_MBR                             0x00000008
#define  SEG_IS_FREE_SPACE_PARTITION            0x00000010
#define  SEG_IS_LINUX_SWAP_PARTITION            0x00000020
#define  SEG_IS_LINUX_RAID_PARTITION            0x00000040
#define  SEG_HAS_DLAT_SERIAL_NUMBERS_REGISTERED 0x00000080


#define  SEG_IS_CONSUMED                        0x00000100
#define  SEG_IS_EMBEDDED                        0x00000200
#define  SEG_IS_EMBEDDED_METADATA               0x00000400
#define  SEG_IS_BSD_PARTITION                   0x00000800
#define  SEG_IS_UNIXWARE_PARTITION              0x00001000
#define  SEG_IS_SOLARIS_X86_PARTITION           0x00002000
#define  SEG_IS_DIRTY_CONTAINER                 0x00004000

typedef struct seg_private_data_s {

    u_int32_t          signature;            /* used to validate our private data area                      */

    LOGICALDISK       *logical_disk;         /* ptr to the logical disk this segment resides on             */

    u_int32_t          sys_id;               /* partition record ... type field                             */

    u_int32_t          boot_ind;             /* partition record boot ind field                             */

    u_int32_t          flags;                /* see above definitions for flag bits                         */

    u_int32_t          tag;                  /* tag field from slice partition record                       */

    u_int32_t          permissions;          /* permissions bits from slice partition record                */

    u_int32_t          frag;                 /* bsd only ... fs frag field of partition record              */

    u_int32_t          cpg;                  /* bsd only ... cpg field of partition record                  */


    u_int32_t          ptable_index;         /* 0-3 ... index into EBR or MBR partition table where this    */
                                             /* segment is found.                                           */

    u_int32_t          part_number;          /* primary partitions are numbered 1-4, logical partitions are */
                                             /* numbered 5-n, embedded partitions are numbered n+1-?        */

    u_int32_t          ebr_number;           /* if EBR seg ... index of this ebr in the ebr chain           */

    u_int64_t          ebr_sector_count;     /* if EBR seg ... this is the total of EBR+all data segs       */

    DISKSEG           *ebr;                  /* every data seg has an EBR or MBR meta data seg associated   */
                                             /* it. If a primary seg then this points to the mbr. If this   */
                                             /* is a logical partition then this points to the ebr.         */
    DISKSEG           *prev_ebr;             /* if EBR seg then this is the back ptr in the EBR chain       */
    DISKSEG           *next_ebr;             /* if EBR seg then this is the forward ptr in the EBR chain    */


    DLA_Table_Sector  *dlat;                 /* pts to DLA table we belong to. DLAT is owned by MBR or EBR  */
                                             /* DISKSEG. it is only referred to by data segs.               */
    DLA_Entry         *dla_entry;            /* valid only if DISKSEG is data segment, otherwise NULL       */

} SEG_PRIVATE_DATA;


#endif
