/*
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: aixregmgr
 * File: aix_regions.c
 *
 * Description: This file contains functions related to region management
 *		in AIX.
 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include "aixregmgr.h"

/* aix_find_region_by_index
 *
 *	Search through the specified container for the given region index.
 */
storage_object_t * aix_find_region_by_index(	storage_container_t	* container,
						unsigned int		index )
{
	storage_object_t	* region;
	aix_region_data_t	* r_data;
	int			rc;

	LOG_ENTRY;

	FOR_EACH(region, container->objects_produced) {
		r_data = region->private_data;
		if ( r_data->lv->lvname == index ) {
			RETURN(region);
		}
	}

	RETURN(NULL);
}


/* aix_deallocate_region
 *
 *	Free up all memory used by this region.
 */
int aix_deallocate_region( storage_object_t * region )
{
	aix_region_data_t	* r_data = region->private_data;
	unsigned int		i;

	LOG_ENTRY;

	if ( r_data ) {
		// Free the LP maps.
		for ( i = 0; i < AIX_MAX_MIRRORS; i++ ) {
			if ( r_data->lp_map[i] ) {
				aix_engine->engine_free(r_data->lp_map[i]);
			}
		}

		if ( r_data->lv &&
		     region->data_type == FREE_SPACE_TYPE ) {
			aix_engine->engine_free(r_data->lv);
		}

		// Free the private data.
		aix_engine->engine_free(r_data);
	}

	// Free the region.
	aix_engine->free_region(region);

	RETURN(0);
}


/* aix_allocate_region
 *
 *	Allocate a new AIX region and all necessary private data fields.
 */
storage_object_t * aix_allocate_region( lv_entries		* lv,
					char			* lv_name,
					unsigned int		pp_size )
{
	storage_object_t	* region;
	aix_region_data_t	* r_data;
	char			name[EVMS_VOLUME_NAME_SIZE+1] = {0};
	unsigned int		i;
	int			rc;

	LOG_ENTRY;

	// Convert the AIX LV-name to an EVMS region-name.
	strncpy(name, AIX_NAME, EVMS_VOLUME_NAME_SIZE);
	strncat(name, "/", EVMS_VOLUME_NAME_SIZE - strlen(name));
	strncat(name, lv_name, EVMS_VOLUME_NAME_SIZE - strlen(name));

	// Create the region structure.
	rc = aix_engine->allocate_region(name, &region);
	if (rc) {
		LOG_CRITICAL("Memory error creating region %s\n", lv_name);
		RETURN(NULL);
	}

	// Create the region private data
	r_data = aix_engine->engine_alloc(sizeof(aix_region_data_t));
	if ( ! r_data ) {
		LOG_CRITICAL("Memory error creating private data for region %s\n", lv_name);
		aix_deallocate_region(region);
		RETURN(NULL);
	}
	region->private_data = r_data;

	// Allocate the LP-to-PP mapping tables.
	for ( i = 0; i < lv->mirror; i++ ) {
		r_data->lp_map[i] = aix_engine->engine_alloc(lv->num_lps * sizeof(aix_lp_map_entry_t));
		if ( ! r_data->lp_map[i] ) {
			LOG_CRITICAL("Memory error creating LP map %d for region %s\n", i, lv_name);
			aix_deallocate_region(region);
			RETURN(NULL);
		}
	}

	// Initialize remaining fields
	region->object_type	= REGION;
	region->data_type	= DATA_TYPE;
	region->plugin		= aix_plugin;
	region->size		= lv->num_lps * PP_SIZE_TO_VSECTORS(pp_size);
	r_data->lv		= lv;

	LOG_DETAILS("Created region %s\n", name);

	RETURN(region);
}


/* aix_calculate_freespace_lps
 *
 *	Go through all real regions in the container and calculate the number of
 *	LPs that should be allocated to the freespace region.
 */
static unsigned long aix_calculate_free_lps( storage_container_t * container )
{
	aix_container_data_t	* c_data = container->private_data;
	unsigned long		used_lps = 0;
	int			i;

	LOG_ENTRY;

	for ( i = 0; i < LVM_MAXLVS; i++ ) {
		if ( c_data->vgda->lv_array[i].lv_state ) {
			used_lps += c_data->vgda->lv_array[i].num_lps;
		}
	}

	RETURN(c_data->pp_count - used_lps);
}


/* aix_create_freespace_region
 *
 *	Create a region to represent the freespace in the specified container,
 *	and place it in the container's list.
 */
static int aix_create_freespace_region( storage_container_t * container )
{
	aix_container_data_t	* c_data = container->private_data;
	storage_object_t	* free_region;
	lv_entries		* free_lv;
	char			free_name[EVMS_VOLUME_NAME_SIZE+1] = {0};

	LOG_ENTRY;

	if ( c_data->freespace ) {
		// Freespace region already exists.
		RETURN(0);
	}

	// Create a name for this freespace region.
	strncpy(free_name, &container->name[strlen(AIX_NAME)+1], EVMS_VOLUME_NAME_SIZE);
	strncat(free_name, "/", EVMS_VOLUME_NAME_SIZE - strlen(free_name));
	strncat(free_name, "Freespace", EVMS_VOLUME_NAME_SIZE - strlen(free_name));

	// Create a dummy lv_entries structure and initialize.
	free_lv = aix_engine->engine_alloc(sizeof(lv_entries));
	if ( ! free_lv ) {
		LOG_CRITICAL("Memory error creating LV structure for Freespace region for container %s\n", container->name);
		RETURN(ENOMEM);
	}

	free_lv->lvname		= -1;
	free_lv->lv_state	= 1;
	free_lv->mirror		= 1;
	free_lv->num_lps	= aix_calculate_free_lps(container);

	// Create the region.
	free_region = aix_allocate_region(free_lv, free_name, c_data->vgda->vg_head->pp_size);
	if ( ! free_region ) {
		LOG_CRITICAL("Memory error creating Freespace region for container %s\n", container->name);
		RETURN(ENOMEM);
	}
	free_region->data_type = FREE_SPACE_TYPE;
	
	// Place the freespace region in the container.
	aix_add_object_to_list(free_region, container->objects_produced);
	free_region->producing_container = container;
	c_data->freespace = free_region;

	RETURN(0);
}


/* aix_build_lp_maps
 *
 *	For each region in this container, set up the logical-to-physical
 *	remappping table.
 */
static int aix_build_lp_maps( storage_container_t * container )
{
	aix_container_data_t	* c_data = container->private_data;
	aix_vgda_t		* vgda = c_data->vgda;
	storage_object_t	* object;
	storage_object_t	* region;
	aix_region_data_t	* r_data;
	pp_entries		* pp_map;
	pv_header		* pv_head;
	unsigned int		lv_index;
	unsigned int		lp_num;
	unsigned int		mirror_copy;
	unsigned int		current_freespace_lp = 0;
	unsigned int		i, j;

	LOG_ENTRY;

	// Search every PVs PP map
	for ( i = 0; i < LVM_MAXPVS; i++ ) {
		if ( ! vgda->pv_headers[i] ||
		     ! vgda->pp_array[i] ) {
			continue;
		}

		pv_head = vgda->pv_headers[i];
		pp_map = vgda->pp_array[i];
		object = aix_find_pv_by_number(container, pv_head->pv_num);

		// Check every PP entry on this PV.
		for ( j = 0; j < pv_head->pp_count; j++ ) {
			if ( pp_map[j].pp_state ) {
				// Valid PP. Add it to the correct LV's map.
				lv_index	= pp_map[j].lv_index - 1;
				lp_num		= pp_map[j].lp_num - 1;
				region		= c_data->regions[lv_index];

				// Determine the value for mirror_copy
				switch (pp_map[j].copy) {
				case 0:
				case 1:
				case 2:
					mirror_copy = 0;
					break;
				case 3:
				case 4:
					mirror_copy = 1;
					break;
				case 5:
					mirror_copy = 2;
					break;
				default:
					LOG_SERIOUS("ERROR: Invalid value for PP map copy field.\n");
					LOG_SERIOUS("       PV %d : PP entry %d : copy = %d\n",
						pv_head->pv_num, j, pp_map[j].copy);
					continue;
				}

				if ( region ) {
					r_data = region->private_data;
					r_data->lp_map[mirror_copy][lp_num].pp_number = j;

					if ( object ) {
						r_data->lp_map[mirror_copy][lp_num].object = object;
						aix_append_region_to_object(region, object);
					}
					else {
						r_data->lp_map[mirror_copy][lp_num].flags |= AIX_LP_MISSING;
					}
				}
				else {
					LOG_SERIOUS("Possible metadata inconsistency detected!\n");
					LOG_SERIOUS("PP %d on PV %s belongs to lv_index %d, lp_num %d, copy %d\n",
						j, object->name, lv_index, lp_num, mirror_copy);
					LOG_SERIOUS("but no region exists with lv_index %d.\n", lv_index);
					// Should we delete this PP entry?
				}
			}
			else {
				// Unassigned PP. Add it to the freespace.
				r_data = c_data->freespace->private_data;
				r_data->lp_map[0][current_freespace_lp].pp_number = j;
				if ( object ) {
					r_data->lp_map[0][current_freespace_lp].object = object;
					aix_append_region_to_object(c_data->freespace, object);
				}
				else {
					r_data->lp_map[0][current_freespace_lp].flags |= AIX_LP_MISSING;
				}
				current_freespace_lp++;
			}
		}
	}

	RETURN(0);
}

/* aix_discover_regions_in_container
 *
 *	Examine the LV array in the VG metadata, and create a new region
 *	for each valid LV entry.
 */
static int aix_discover_regions_in_container( storage_container_t * container )
{
	aix_container_data_t	* c_data = container->private_data;
	aix_vgda_t		* vgda = c_data->vgda;
	storage_object_t	* region;
	unsigned int		i;

	LOG_ENTRY;

	// Look for every valid entry in the LV array. Create a new region
	// for that entry if one does not yet exist.
	for ( i = 0; i < LVM_MAXLVS; i++ ) {
		if ( vgda->lv_array[i].lv_state &&
		     ! c_data->regions[i] ) {
			region = aix_allocate_region(&vgda->lv_array[i], 
							vgda->lv_names->name[i],
							vgda->vg_head->pp_size);
			if ( ! region ) {
				LOG_CRITICAL("Memory error creating region %s\n", vgda->lv_names->name[i]);
				continue;
			}

			aix_add_object_to_list(region, container->objects_produced);
			region->producing_container = container;
			c_data->regions[i] = region;
		}
	}

	RETURN(0);
}

/* aix_discover_regions
 *
 *	For each container, create a freespace region, create all the
 *	necessary real regions, and set up the LP maps.
 */
int aix_discover_regions( void )
{
	storage_container_t	* container;
	int			rc;

	LOG_ENTRY;

	FOR_EACH(container, aix_container_list) {

		LOG_DETAILS("Searching for regions in container %s\n", container->name);

		// Create the freespace region.
		aix_create_freespace_region(container);

		// Create all real regions
		aix_discover_regions_in_container(container);

		// Build the LP maps
		aix_build_lp_maps(container);
	}

	RETURN(0);
}


/* aix_export_regions
 *
 *	Go through all existing containers and export all regions.
 */
int aix_export_regions( dlist_t regions )
{
	storage_container_t	* container;
	aix_container_data_t	* c_data;
	storage_object_t	* region;
	aix_region_data_t	* r_data;
	unsigned int		i, count = 0;
	int			rc;

	LOG_ENTRY;

	FOR_EACH(container, aix_container_list) {
		c_data = container->private_data;
		for ( i = 0; i < LVM_MAXLVS; i++ ) {
			if ( c_data->regions[i] ) {
				region = c_data->regions[i];
				r_data = region->private_data;
				if ( ! (r_data->flags & AIX_REGION_EXPORTED) ) {
					aix_add_object_to_list(region, regions);
					r_data->flags |= AIX_REGION_EXPORTED;
					count++;
				}
			}
		}
	}



	RETURN(count);
}
