# -*- coding: utf-8 -*-

# Copyright (c) 2006 - 2008 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the VCS status monitor thread class for Subversion.
"""

import os

import pysvn

from PyQt4.QtCore import QRegExp, QStringList, QString

from VCS.StatusMonitorThread import VcsStatusMonitorThread

import Preferences

class SvnStatusMonitorThread(VcsStatusMonitorThread):
    """
    Class implementing the VCS status monitor thread class for Subversion.
    """
    def __init__(self, interval, projectDir, vcs, parent = None):
        """
        Constructor
        
        @param interval new interval in seconds (integer)
        @param projectDir project directory to monitor (string or QString)
        @param vcs reference to the version control object
        @param parent reference to the parent object (QObject)
        """
        VcsStatusMonitorThread.__init__(self, interval, projectDir, vcs, parent)
    
    def _performMonitor(self):
        """
        Protected method implementing the monitoring action.
        
        This method populates the statusList member variable
        with a list of strings giving the status in the first column and the
        path relative to the project directory starting with the third column.
        The allowed status flags are:
        <li>
            <ul>"A" path was added but not yet comitted</ul>
            <ul>"M" path has local changes</ul>
            <ul>"U" path needs an update</ul>
            <ul>"Z" path contains a conflict</ul>
            <ul>" " path is back at normal</ul>
        </li>
        
        @return tuple of flag indicating successful operation (boolean) and 
            a status message in case of non successful operation (QString)
        """
        client = pysvn.Client()
        client.exception_style = 1
        
        cwd = os.getcwd()
        os.chdir(unicode(self.projectDir))
        try:
            allFiles = client.status('.', recurse = True, get_all = True, 
                                          ignore = True, 
                                          update = \
                                            not Preferences.getVCS("MonitorLocalStatus"))
            states = {}
            for file in allFiles:
                uptodate = True
                if file.repos_text_status != pysvn.wc_status_kind.none:
                    uptodate = uptodate and file.repos_text_status == file.text_status
                if file.repos_prop_status != pysvn.wc_status_kind.none:
                    uptodate = uptodate and file.repos_prop_status == file.prop_status
                
                status = ""
                if not uptodate:
                    status = "U"
                elif file.text_status == pysvn.wc_status_kind.conflicted or \
                   file.prop_status == pysvn.wc_status_kind.conflicted:
                    status = "Z"
                elif file.text_status == pysvn.wc_status_kind.modified or \
                   file.prop_status == pysvn.wc_status_kind.modified:
                    status = "M"
                elif file.text_status == pysvn.wc_status_kind.added or \
                   file.prop_status == pysvn.wc_status_kind.added:
                    status = "A"
                if status:
                    states[file.path] = status
                    try:
                        if self.reportedStates[file.path] != status:
                            self.statusList.append("%s %s" % (status, file.path))
                    except KeyError:
                        self.statusList.append("%s %s" % (status, file.path))
            for name in self.reportedStates.keys():
                if name not in states:
                    self.statusList.append("  %s" % name)
            self.reportedStates = states
            res = True
            statusStr = \
                self.trUtf8("Subversion status checked successfully (using pysvn)")
        except pysvn.ClientError, e:
            res = False
            statusStr = QString(e.args[0])
        os.chdir(cwd)
        return res, statusStr
