#include <gtk/gtk.h>
#include <glib.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <signal.h>

#include "entity.h"
#include "modulegen.h"
#include "renderers.h"
#include "mainloop.h"

#ifdef USE_ENTITY_EXEC_CLASS
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#endif				/* USE_ENTITY_EXEC_CLASS */

#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>

/* Eprocfs has a nice ring to it, plus it explains what is going on. */
#define ENTITY_EXEC_CLASS_ROOT "/eprocfs"

/* Final path of the entry in the eprocfs. */
static gchar *socket_path = NULL;


void
entity_exec_class_cleanup (void)
{
    /* Try to make sure that the procfs is clean. We don't * need any stale
     * links running around confusing the user. */
    EDEBUG (("entity-exec-class", "cleaning up socket path = %s\n",
	     socket_path));

#ifdef USE_ENTITY_EXEC_CLASS
    unlink (socket_path);
#endif				/* USE_ENTITY_EXEC_CLASS */
}

/* Handle all termination signals. */
static void
signal_handle_end (int sig)
{
    EDEBUG (("entity-exec-class", "got sig=%i\n", sig));

    entity_exec_class_cleanup ();
    fprintf (stderr, "Exiting with sig%i\n", sig);

    exit (1);			/* For now we must exit... */
}

#ifdef USE_ENTITY_EXEC_CLASS

static gint
file_exists (gchar * path)
{
    struct stat s;

    return (stat (path, &s) == 0);
}

#endif				/* USE_ENTITY_EXEC_CLASS */

/* Another brick in the -Wall */
void perl_init (RendererFlags flags);
void tcl_init (RendererFlags flags);
void c_init (RendererFlags flags);
void python_init (RendererFlags flags);
void rendgtk_init (RendererFlags flags);
void mzscheme_init (RendererFlags flags);
void javascript_init (RendererFlags flags);

/* Another brick in the -Wall */
void rendgeneric_init (RendererFlags flags);
void entity_lang_init (RendererFlags flags);
void rendcsink_init (RendererFlags flags);
void rendresolver_init (RendererFlags flags);


void
entity_exec_initialize_modules (void)
{
    /* These are built in, and are always initialized */
    rendgeneric_init (RENDERER_REGISTER | RENDERER_INIT);
    entity_lang_init (RENDERER_REGISTER | RENDERER_INIT);
#ifdef USE_CSINK
    rendcsink_init (RENDERER_REGISTER | RENDERER_INIT);
    rendresolver_init (RENDERER_REGISTER | RENDERER_INIT);
#endif

#ifdef STATIC_PERL
    perl_init (RENDERER_REGISTER | RENDERER_INIT);
#endif

#ifdef STATIC_TCL
    tcl_init (RENDERER_REGISTER | RENDERER_INIT);
#endif

#ifdef STATIC_C
    c_init (RENDERER_REGISTER | RENDERER_INIT);
#endif

#ifdef STATIC_PYTHON
    python_init (RENDERER_REGISTER | RENDERER_INIT);
#endif

#ifdef STATIC_GTK
    rendgtk_init (RENDERER_REGISTER | RENDERER_INIT);
#endif

#ifdef USE_MZSCHEME
    mzscheme_init (RENDERER_REGISTER | RENDERER_INIT);
#endif

#ifdef STATIC_JAVASCRIPT
    javascript_init (RENDERER_REGISTER | RENDERER_INIT);
#endif
}

#ifdef USE_ENTITY_EXEC_CLASS


static gint
connect_to_exec_class (void)
{
    struct sockaddr_un addr;
    gint fd = -1;
    gint len;
    gint ret;

    EDEBUG (("main", "Attempting to connect to exec class server.."));
    if (file_exists (socket_path)) {
	/* We want to make sure that we can actually connect into it,
	 * and it's not a old dud lying around */
	EDEBUG (("main", "%s found, connecting..", socket_path));

	fd = socket (AF_UNIX, SOCK_STREAM, 0);
	if (fd < 0)
	    g_error
		("Error opening link to Master Entity: "
		 "creating AF_UNIX socket: '%s': %s",
		 socket_path, g_strerror (errno));

	memset (&addr, 0, sizeof (addr));
	addr.sun_family = AF_UNIX;
	strcpy (addr.sun_path, socket_path);
	len = sizeof (addr);

	ret = connect (fd, (struct sockaddr *) &addr, len);
	if (ret < 0) {
	    EDEBUG (("main",
		     "Master Entity doesn't want to talk to us: "
		     "%s - Running as Master", g_strerror (errno)));

	    /* In this case, we delete the old link and assume no server */
	    unlink (socket_path);
	    close (fd);
	    fd = -1;
       	}
    }

    return (fd);
}

static gint
setup_exec_class_server (void)
{
    struct sockaddr_un addr;
    gint fd;
    gint len;
    gint ret;

    EDEBUG (("main", "creating domain socket"));

    /* create link */
    fd = socket (AF_UNIX, SOCK_STREAM, 0);

    if (fd < 0)
	g_error
	    ("Error while trying to take over the universe: creating "
	     "AF_UNIX socket: '%s': %s", socket_path, g_strerror (errno));

    /* establish a unix domain socket */
    memset (&addr, 0, sizeof (addr));
    addr.sun_family = AF_UNIX;
    strcpy (addr.sun_path, socket_path);

    len = sizeof (addr);

    ret = bind (fd, (struct sockaddr *) &addr, len);
    if (ret < 0)
	g_error ("World takeover plan failed at stage 2: %s",
	    g_strerror (errno));

    ret = listen (fd, 5);

    EDEBUG (("main", "domain server socket opened, returning fd"));

    return (fd);
}


static void
link_data_ready (gint fd, EIOCond condition, gpointer data)
{
    gchar buf[PATH_MAX];
    gint fd2;
    struct sockaddr_un addr;
    gint len;

    len = sizeof (addr);
    fd2 = accept (fd, (struct sockaddr *) &addr, &len);

    memset (buf, 0, PATH_MAX);
    EDEBUG (("main", "going to read from buf - condition %d", condition));
    recv (fd2, buf, PATH_MAX, 0);

    EDEBUG (("main", "Loading file sent to exec class '%s'", buf));
    xml_parse_file (NULL, buf);
}

#endif				/* USE_ENTITY_EXEC_CLASS */

static void
entity_create_dotdirs (void)
{
    /* broken on win32 FIXME */
#ifndef WIN32
    gchar *path;

    /* Insure entity home is created */
    path = g_strconcat (ENTITY_HOME, NULL);

    if (-1 == mkdir (path, 0700)) {
	if (errno != EEXIST) {	/* Hope this is a dir. */
	    g_error ("creating directory %s: %s", path, g_strerror (errno));
	}
    }
    g_free (path);

    /* First need to have a path to the fs. */
    path = g_strconcat (ENTITY_HOME, ENTITY_EXEC_CLASS_ROOT, NULL);
    if (-1 == mkdir (path, 0700)) {
	if (errno != EEXIST) {	/* Hope this is a dir. */
	    g_error ("creating directory %s: %s", path, g_strerror (errno));
	}
    }
    g_free (path);
#endif
}


void
entity_exec_class_load (GSList *files)
{
    GSList *tmp;

#ifdef USE_ENTITY_EXEC_CLASS
    int client_fd = -1;
    int server_fd = -1;
    int ret;
    gchar *exec_class;
#endif				/* USE_ENTITY_EXEC_CLASS */

    /* Insure ~/.entity and ~/.entity/eprocfs/ are built */
    entity_create_dotdirs ();

#ifdef USE_ENTITY_EXEC_CLASS

    /* Derive an exec domain for this file to run in. */
    exec_class = econfig_get_attr ("exec-class");

    /* Set the exec_class to the pid if it wasn't specified on the cmdline. */
    if (!exec_class) {
	exec_class = g_new0 (char, 20);
	/* Getpid won't fail. */
	g_snprintf (exec_class, 20, "%i", getpid ());
    }

#endif				/* USE_ENTITY_EXEC_CLASS */

    EDEBUG (("main", "renderers init"));
    renderers_init ();

#ifdef HAVE_SIGNAL_H
    /* We are now a main app so we need to watch termination signals so
     * that we can clean up after ourselves. */
    signal (SIGTERM, signal_handle_end);
    signal (SIGINT, signal_handle_end);
    signal (SIGQUIT, signal_handle_end);
    signal (SIGILL, signal_handle_end);
    signal (SIGSEGV, signal_handle_end);
#endif				/* HAVE_SIGNAL_H */

    /* Initialized built in renderers */
    entity_exec_initialize_modules ();

#ifdef USE_ENTITY_EXEC_CLASS
    /* Setup socket path */
    socket_path = g_strconcat (ENTITY_HOME, ENTITY_EXEC_CLASS_ROOT, "/",
			       exec_class, NULL);

    client_fd = connect_to_exec_class ();
    
    /* If the exec class failed, we want to set it up so we
     * can accept connects from others */
    if (client_fd < 0) {
	EDEBUG (("main", "Unable to connect to server exec class, becoming server.."));
	server_fd = setup_exec_class_server ();
    }

#endif /* USE_ENTITY_EXEC_CLASS */

    tmp = files;
    while (tmp) {
	gchar *filename = tmp->data;

	EDEBUG (("main", "loading file '%s'", filename));

#ifdef USE_ENTITY_EXEC_CLASS
	if (client_fd > 0) {
	    ret = write (client_fd, filename, strlen (filename));
	    if (ret != strlen (filename)) {
		g_error ("Error writing filename to server through exec class: %s",
			 g_strerror (errno));
	    }

	} else {
	    xml_parse_file (NULL, filename);
	}

#else
	xml_parse_file (NULL, filename);
#endif /* USE_ENTITY_EXEC_CLASS */

	tmp = tmp->next;
    }
#ifdef USE_ENTITY_EXEC_CLASS
    if (server_fd > 0)
	entity_mainloop_io_add (server_fd, EIO_READ, (gpointer) link_data_ready, NULL);
    else
	exit (0);
#endif

}


