/* This file is part of EdiTeX, an editor of mathematical
 * expressions based on TeX syntax.
 * 
 * Copyright (C) 2002-2003 Luca Padovani <lpadovan@cs.unibo.it>,
 *                    2003 Paolo Marinelli <pmarinel@cs.unibo.it>.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * For more information, please visit the project's home page
 * http://helm.cs.unibo.it/editex/
 * or send an email to <lpadovan@cs.unibo.it>
 */

#include <string>
#include <cctype>
#include <cassert>

#include "ALogger.hh"
#include "TToken.hh"
#include "LPushLexer.hh"
#include "APushParser.hh"

LPushLexer::LPushLexer(ALogger& l, APushParser& p) : APushLexer(l, p)
{
  state = ACCEPT;
}

void
LPushLexer::reset()
{
  buffer.erase();
  state = ACCEPT;

  displayCursor();
}

void
LPushLexer::flush()
{
  push(-1);
}

void
LPushLexer::transaction(char ch, State newState)
{
  switch (ch)
    {
    case '{': parser.push(TToken(TToken::BEGIN)); break;
    case '}': parser.push(TToken(TToken::END)); break;
    case '$': parser.push(TToken(TToken::SHIFT)); break;
    case '&': parser.push(TToken(TToken::ALIGN)); break;
    case '\n':
    case '\r': parser.push(TToken(TToken::EOL, ch)); break;
    case '^': parser.push(TToken(TToken::SUPERSCRIPT)); break;
    case '_': parser.push(TToken(TToken::SUBSCRIPT)); break;
    case '\t': parser.push(TToken(TToken::IGNORABLE_SPACE, ch)); break;
    case ' ': parser.push(TToken(TToken::SPACE, ch)); break;
    case '~': parser.push(TToken(TToken::ACTIVE, ch)); break;
    case '%': parser.push(TToken(TToken::COMMENT)); break;     
    default: parser.push(TToken(TToken::OTHER, ch)); break;
    }
  state = newState;
}

void
LPushLexer::push(char ch)
{
  switch (state)
    {
    case ACCEPT:
      if (ch == '\\') state = ESCAPE;
      else if (ch == '#') state = PARAMETER;
      else if (ch == -1) ;
      else if (isalpha(ch))
	{
	  buffer.push_back(ch);
	  state = IDENTIFIER;
	}
      else if (isdigit(ch))
        {
	  buffer.push_back(ch);
	  state = NUMBER;
	}
      else transaction(ch, ACCEPT);
      break;
    case ESCAPE:
      if (isalpha(ch))
	{
	  buffer.push_back(ch);
	  state = MACRO;
	}
      else if (ch == -1) error();
      else if (isdigit(ch))
        {
	  // in this case, the previous '\' is ignored
	  buffer.push_back(ch);
	  state = NUMBER;
	}
      else
	{
	  parser.push(TToken(TToken::CONTROL, ch));
	  state = ACCEPT;
	}
      break;
    case MACRO:
      if (ch == '\\')
	{
	  parser.push(TToken(TToken::CONTROL, buffer));
	  buffer.erase();
	  state = ESCAPE;
	}
      else if (ch == '#')
	{
	  parser.push(TToken(TToken::CONTROL, buffer));
	  buffer.erase();
	  state = PARAMETER;
	}
      else if (isalpha(ch))
	buffer.push_back(ch);
      else if (ch == -1)
        {
	  parser.push(TToken(TToken::CONTROL, buffer));
	  buffer.erase();
	  state = ACCEPT;
	}
      else if (isspace(ch))
          {
	    // we don't call transaction, because a white space is useful to exit from the macro,
	    // without "side effect". It's the TeX syntax.
	    parser.push(TToken(TToken::CONTROL, buffer));
	    buffer.erase();
	    state = ACCEPT;
	}
      else if (isdigit(ch))
        {
	  parser.push(TToken(TToken::CONTROL, buffer));
	  buffer.erase();
	  buffer.push_back(ch);
	  state = NUMBER;
	}
      else
        {
	  parser.push(TToken(TToken::CONTROL, buffer));
	  buffer.erase();
          transaction(ch, ACCEPT);
        }
      break;
    case PARAMETER:
      if (ch == -1) error();
      else
        {
	  parser.push(TToken(TToken::PARAMETER, ch));
	  state = ACCEPT;
	}
      break;
    case IDENTIFIER:
      if (ch == -1)
        {
	  parser.push(TToken(TToken::LETTER, buffer));
	  buffer.erase();
	  state = ACCEPT;
	}
      else if (isalpha(ch) || isdigit(ch))
        {
	  buffer.push_back(ch);
	}
      else if (ch == '\\') state = ESCAPED_CHARACTER;
      else if (ch == '#')
        {
	  parser.push(TToken(TToken::LETTER, buffer));
	  buffer.erase();
	  state = PARAMETER;
	}
      else
        {
	  parser.push(TToken(TToken::LETTER, buffer));
	  buffer.erase();
	  transaction(ch, ACCEPT);
	}
      break;
    case ESCAPED_CHARACTER:
      if ((ch == '-') || (ch == '_') || (ch == '/'))
        {
	  buffer.push_back(ch);
	  state = IDENTIFIER;
	}
      else if (isalpha(ch))
	{
	  parser.push(TToken(TToken::LETTER, buffer));
	  buffer.erase();
	  buffer.push_back(ch);
	  state = MACRO;
	}
      else if (ch == -1) error();
      else if (isdigit(ch))
        {
	  parser.push(TToken(TToken::LETTER, buffer));
	  buffer.erase();
	  buffer.push_back(ch);
	  state = NUMBER;
	}
      else
	{
	  parser.push(TToken(TToken::LETTER, buffer));
	  buffer.erase();
	  parser.push(TToken(TToken::CONTROL, ch));
	  state = ACCEPT;
	}
      break;
    case NUMBER:
      if (isdigit(ch)) buffer.push_back(ch);
      else if (isalpha(ch))
        {
	  parser.push(TToken(TToken::DIGIT, buffer));
	  buffer.erase();
	  buffer.push_back(ch);
	  state = IDENTIFIER;
	}
      else if (ch == -1)
        {
	  parser.push(TToken(TToken::DIGIT, buffer));
	  buffer.erase();
	  state = ACCEPT;
	}
      else if (ch == '\\')
        {
	  parser.push(TToken(TToken::DIGIT, buffer));
	  buffer.erase();
	  state = ESCAPE;
	}
      else if (ch == '#')
        {
	  parser.push(TToken(TToken::DIGIT, buffer));
	  buffer.erase();
	  state = PARAMETER;
	}
      else
        {
	  parser.push(TToken(TToken::DIGIT, buffer));
	  buffer.erase();
	  transaction(ch, ACCEPT);
	}
      break;
    default:
      assert(0);
      break;
    }

  displayCursor();

}

void
LPushLexer::drop(bool alt)
{
  std::string restore = "";

  switch (state)
    {
    case ACCEPT:
      {
        restore = parser.drop(alt);
        long bs_pos = restore.find('\\');
        if ((restore.length() > 0) && (bs_pos != std::string::npos))
          {
	    // in this case we have to control the blackslash's position
	    if (bs_pos == 0)
	      {
	        //logger.debug(restore);
	        buffer = std::string(restore, 1, restore.length() - 1);
	        state = (buffer.length() > 0) ? MACRO : ESCAPE;
	      }
	    else
	      {
	        assert(bs_pos == restore.length() - 1);
	        buffer = std::string(restore, 0, bs_pos);
	        state = ESCAPED_CHARACTER;
	      }
	  }
        else if (restore.length() > 0 && isdigit(restore[0]))
          {
	    buffer = restore;
	    state = NUMBER;
	  }
        else if (restore.length() > 0 && isalpha(restore[0]))
          {
	    buffer = restore;
	    state = IDENTIFIER;
	  }
      }
      break;
/*      if (restore.length() > 0 && restore[0] == '\\')
	{
	  logger.debug(restore);
	  buffer = std::string(restore, 1, restore.length() - 1);
	  state = (buffer.length() > 0) ? MACRO : ESCAPE;
	}
      else if (restore.length() > 0 && isdigit(restore[0]))
        {
	  buffer = restore;
	  state = NUMBER;
	}
      else if (restore.length() > 0 && isalpha(restore[0]))
        {
	  buffer = restore;
	  state = IDENTIFIER;
	}
      break;*/
    case ESCAPED_CHARACTER:
      state = IDENTIFIER;
      break;
    case ESCAPE:
      state = ACCEPT;
      break;
    case MACRO:
      if (alt) buffer.erase();
      else buffer.erase(buffer.length() - 1, 1);
      if (buffer.length() == 0) state = ESCAPE;
      break;
    case IDENTIFIER:
      switch (buffer[buffer.length() - 1])
	{
	case '-':
	case '_':
	  buffer.erase(buffer.length() - 1, 1);
	  if (alt) state = ESCAPED_CHARACTER;
	  break;
	default:
          if (alt) buffer.erase(); 
          else buffer.erase(buffer.length() - 1, 1);
          if (buffer.length() == 0) state = ACCEPT;
          break;
	}
      break;
    case NUMBER:
      if (alt) buffer.erase();
      else buffer.erase(buffer.length() - 1, 1);
      if (buffer.length() == 0) state = ACCEPT;
      break;
    case PARAMETER:
    default:
      //assert(0);
      error();
      break;
    }

  displayCursor();

}

void
LPushLexer::displayCursor()
{
  switch (state)
    {
    case ESCAPE: parser.setCursorHint("\\"); break;
    case ESCAPED_CHARACTER: parser.setCursorHint(buffer + "\\"); break;
    case MACRO: parser.setCursorHint("\\" + buffer); break;
    case PARAMETER: parser.setCursorHint("#"); break;
    case IDENTIFIER: parser.setCursorHint(buffer); break;
    case NUMBER: parser.setCursorHint(buffer); break;
    default: parser.setCursorHint(""); break;
    }
}

bool
LPushLexer::error() const
{
  return false;
}
