/*
***************************************************************************
*
* Author: Teunis van Beelen
*
* Copyright (C) 2010 Teunis van Beelen
*
* teuniz@gmail.com
*
***************************************************************************
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation version 2 of the License.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License along
* with this program; if not, write to the Free Software Foundation, Inc.,
* 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*
***************************************************************************
*
* This version of GPL is at http://www.gnu.org/licenses/old-licenses/gpl-2.0.txt
*
***************************************************************************
*/



#include "signalcurve.h"


SignalCurve::SignalCurve(QWidget *parent) : QWidget(parent)
{
  setAttribute(Qt::WA_OpaquePaintEvent);

  SignalColor = Qt::blue;
  tracewidth = 0;
  BackgroundColor = Qt::gray;
  RasterColor = Qt::darkGray;
  BorderColor = Qt::lightGray;
  RulerColor = Qt::black;
  TextColor = Qt::black;

  dbuf = NULL;
  fbuf = NULL;
  h_rastersize = 10;
  v_rastersize = 10;
  bordersize = 60;
  drawHruler = 1;
  drawVruler = 1;
  h_ruler_startvalue = 0.0;
  h_ruler_endvalue = 100.0;
  h_ruler_precision = 2;
  drawcurve_before_raster = 0;
  h_label[0] = 0;
  v_label[0] = 0;
  upperlabel1[0] = 0;
  upperlabel2[0] = 0;
  lowerlabel[0] = 0;
  max_value = 100.0;
  min_value = -100.0;
  extra_button = 0;
  extra_button_txt[0] = 0;
  use_move_events = 0;
  crosshair_1_active = 0;
  crosshair_1_moving = 0;
  crosshair_1_value = 0.0;
  crosshair_1_value_2 = 0.0;
  crosshair_1_x_position = 0;

  crosshair_1_color = Qt::red;

  cursorEnabled = TRUE;
  printEnabled = TRUE;
  dashBoardEnabled = TRUE;
  updates_enabled = TRUE;

  spectrum_color = NULL;

  old_w = 10000;
}


void SignalCurve::clear()
{
  dbuf = NULL;
  fbuf = NULL;

  bufsize = 0;

  use_move_events = 0;
  crosshair_1_active = 0;
  crosshair_1_moving = 0;
  crosshair_1_value = 0.0;
  crosshair_1_value_2 = 0.0;
  crosshair_1_x_position = 0;

  update();
}


void SignalCurve::mousePressEvent(QMouseEvent *event)
{
  int m_x,
      m_y;

  setFocus(Qt::MouseFocusReason);

  w = width();
  h = height();

  m_x = event->x() - bordersize;
  m_y = event->y() - bordersize;

  if(m_x < 0 ||
     m_x > (w - (bordersize * 2)) ||
     m_y < 0 ||
     m_y > (h - (bordersize * 2)))
  {
    return;
  }

  if(event->button()==Qt::LeftButton)
  {
    if(printEnabled == TRUE)
    {
      if((m_y<21)&&(m_y>3)&&(m_x>((w - (bordersize * 2)) - 43))&&(m_x<((w - (bordersize * 2)) - 3)))
      {
        exec_sidemenu();

        return;
      }
    }

    if(dashBoardEnabled == TRUE)
    {
      if((m_y<61)&&(m_y>43)&&(m_x>((w - (bordersize * 2)) - 43))&&(m_x<((w - (bordersize * 2)) - 3)))
      {
        emit dashBoardClicked();

        return;
      }
    }

    if(cursorEnabled == TRUE)
    {
      if((m_y<41)&&(m_y>23)&&(m_x>((w - (bordersize * 2)) - 43))&&(m_x<((w - (bordersize * 2)) - 3)))
      {
        if(crosshair_1_active)
        {
          crosshair_1_active = 0;
          crosshair_1_moving = 0;
          use_move_events = 0;
          setMouseTracking(FALSE);
        }
        else
        {
          crosshair_1_active = 1;
          if(!crosshair_1_x_position)
          {
            crosshair_1_value = 0.0;
            crosshair_1_x_position = (w - (bordersize * 2)) / 2;
            crosshair_1_y_position = (h - (bordersize * 2)) / 2;
            mouse_old_x = crosshair_1_x_position;
            mouse_old_y = crosshair_1_y_position;
          }
        }

        update();

        return;
      }
    }

    if(crosshair_1_active)
    {
      if((m_y<(crosshair_1_y_position + 15))&&(m_y>(crosshair_1_y_position - 25))&&(m_x>crosshair_1_x_position)&&(m_x<(crosshair_1_x_position + 65)))
      {
        crosshair_1_moving = 1;
        use_move_events = 1;
        setMouseTracking(TRUE);
        mouse_old_x = m_x;
        mouse_old_y = m_y;
      }

      if(m_x>(crosshair_1_x_position-10)&&(m_x<(crosshair_1_x_position + 10)))
      {
        crosshair_1_moving = 1;
        use_move_events = 1;
        setMouseTracking(TRUE);
        mouse_old_x = m_x;
        mouse_old_y = m_y;
      }
    }
  }
}


void SignalCurve::mouseReleaseEvent(QMouseEvent *)
{
  crosshair_1_moving = 0;
  use_move_events = 0;
  setMouseTracking(FALSE);
}


void SignalCurve::mouseMoveEvent(QMouseEvent *event)
{
  if(!use_move_events)
  {
    return;
  }

  mouse_x = event->x() - bordersize;
  mouse_y = event->y() - bordersize;

  if(crosshair_1_moving)
  {
    crosshair_1_x_position += (mouse_x - mouse_old_x);
    mouse_old_x = mouse_x;
    if(crosshair_1_x_position<2)
    {
      crosshair_1_x_position = 2;
    }
    if(crosshair_1_x_position>(w-(bordersize * 2) -40))
    {
      crosshair_1_x_position = w-(bordersize * 2) -40;
    }

    crosshair_1_y_position += (mouse_y - mouse_old_y);
    mouse_old_y = mouse_y;
    if(crosshair_1_y_position<25)
    {
      crosshair_1_y_position = 25;
    }
    if(crosshair_1_y_position>(h-(bordersize * 2) -25))
    {
      crosshair_1_y_position = h-(bordersize * 2) -25;
    }
  }

  update();
}


void SignalCurve::resizeEvent(QResizeEvent *event)
{
  if(crosshair_1_active)
  {
    crosshair_1_x_position *= ((double)(width() - (bordersize * 2)) / (double)(old_w - (bordersize * 2)));
  }

  QWidget::resizeEvent(event);
}


void SignalCurve::exec_sidemenu()
{
  sidemenu = new QDialog(this);

  if(extra_button)
  {
    sidemenu->setMinimumSize(QSize(120, 160));
    sidemenu->setMaximumSize(QSize(120, 160));
  }
  else
  {
    sidemenu->setMinimumSize(QSize(120, 130));
    sidemenu->setMaximumSize(QSize(120, 130));
  }
  sidemenu->setWindowTitle("Print");
  sidemenu->setModal(TRUE);
  sidemenu->setAttribute(Qt::WA_DeleteOnClose, TRUE);

  sidemenuButton1 = new QPushButton(sidemenu);
  sidemenuButton1->setGeometry(10, 10, 100, 20);
  sidemenuButton1->setText("to printer");

  sidemenuButton2 = new QPushButton(sidemenu);
  sidemenuButton2->setGeometry(10, 40, 100, 20);
  sidemenuButton2->setText("to Postscript");

  sidemenuButton3 = new QPushButton(sidemenu);
  sidemenuButton3->setGeometry(10, 70, 100, 20);
  sidemenuButton3->setText("to PDF");

  sidemenuButton4 = new QPushButton(sidemenu);
  sidemenuButton4->setGeometry(10, 100, 100, 20);
  sidemenuButton4->setText("to Image");

  if(extra_button)
  {
    sidemenuButton5 = new QPushButton(sidemenu);
    sidemenuButton5->setGeometry(10, 130, 100, 20);
    sidemenuButton5->setText(extra_button_txt);
  }

  QObject::connect(sidemenuButton1, SIGNAL(clicked()), this, SLOT(print_to_printer()));
  QObject::connect(sidemenuButton2, SIGNAL(clicked()), this, SLOT(print_to_postscript()));
  QObject::connect(sidemenuButton3, SIGNAL(clicked()), this, SLOT(print_to_pdf()));
  QObject::connect(sidemenuButton4, SIGNAL(clicked()), this, SLOT(print_to_image()));
  if(extra_button)
  {
    QObject::connect(sidemenuButton5, SIGNAL(clicked()), this, SLOT(send_button_event()));
  }

  sidemenu->exec();
}


void SignalCurve::send_button_event()
{
  emit extra_button_clicked();

  sidemenu->close();
}


void SignalCurve::create_button(const char *txt)
{
  extra_button = 1;
  strncpy(extra_button_txt, txt, 16);
  extra_button_txt[15] = 0;
}


void SignalCurve::backup_colors_for_printing(void)
{
  backup_color_1 = SignalColor;
  SignalColor = Qt::black;
  backup_color_2 = RasterColor;
  RasterColor = Qt::darkGray;
  backup_color_3 = BorderColor;
  BorderColor = Qt::lightGray;
  backup_color_4 = RulerColor;
  RulerColor = Qt::black;
  backup_color_5 = TextColor;
  TextColor = Qt::black;
}


void SignalCurve::restore_colors_after_printing(void)
{
  SignalColor = backup_color_1;
  RasterColor = backup_color_2;
  BorderColor = backup_color_3;
  RulerColor = backup_color_4;
  TextColor = backup_color_5;
}


void SignalCurve::print_to_postscript()
{
  QFileDialog fchooser;

  QStringList fileNames;

  fchooser.setViewMode(QFileDialog::Detail);
  fchooser.setConfirmOverwrite(1);
  fchooser.setFileMode(QFileDialog::AnyFile);
  fchooser.setAcceptMode(QFileDialog::AcceptSave);
  fchooser.setWindowTitle("Print to PostScript");
  fchooser.setLabelText(QFileDialog::FileName, "File");
  fchooser.setDefaultSuffix("ps");
  fchooser.setFilter("PostScript files (*.ps *.PS)");

  if(!(fchooser.exec() == QDialog::Accepted))
  {
    sidemenu->close();

    return;
  }

  fileNames = fchooser.selectedFiles();

  QPrinter printer(QPrinter::HighResolution);

  printer.setOutputFormat(QPrinter::PostScriptFormat);
  printer.setOutputFileName(fileNames.at(0));
  printer.setPageSize(QPrinter::A4);
  printer.setOrientation(QPrinter::Landscape);

  backup_colors_for_printing();

  QPainter paint(&printer);

  drawWidget_to_printer(&paint, printer.pageRect().width(), printer.pageRect().height());

  restore_colors_after_printing();

  sidemenu->close();
}


void SignalCurve::print_to_pdf()
{
  QFileDialog fchooser;

  QStringList fileNames;

  fchooser.setViewMode(QFileDialog::Detail);
  fchooser.setConfirmOverwrite(1);
  fchooser.setFileMode(QFileDialog::AnyFile);
  fchooser.setAcceptMode(QFileDialog::AcceptSave);
  fchooser.setWindowTitle("Print to PDF");
  fchooser.setLabelText(QFileDialog::FileName, "File");
  fchooser.setDefaultSuffix("pdf");
  fchooser.setFilter("PDF files (*.pdf *.PDF)");

  if(!(fchooser.exec() == QDialog::Accepted))
  {
    sidemenu->close();

    return;
  }

  fileNames = fchooser.selectedFiles();

  QPrinter printer(QPrinter::HighResolution);

  printer.setOutputFormat(QPrinter::PdfFormat);
  printer.setOutputFileName(fileNames.at(0));
  printer.setPageSize(QPrinter::A4);
  printer.setOrientation(QPrinter::Landscape);

  backup_colors_for_printing();

  QPainter paint(&printer);

  drawWidget_to_printer(&paint, printer.pageRect().width(), printer.pageRect().height());

  restore_colors_after_printing();

  sidemenu->close();
}


void SignalCurve::print_to_image()
{
  QFileDialog fchooser;

  QStringList fileNames;

  fchooser.setViewMode(QFileDialog::Detail);
  fchooser.setConfirmOverwrite(1);
  fchooser.setFileMode(QFileDialog::AnyFile);
  fchooser.setAcceptMode(QFileDialog::AcceptSave);
  fchooser.setWindowTitle("Print to Image");
  fchooser.setLabelText(QFileDialog::FileName, "File");
  fchooser.setDefaultSuffix("png");
  fchooser.setFilter("PNG files (*.png *.PNG)");

  if(!(fchooser.exec() == QDialog::Accepted))
  {
    sidemenu->close();

    return;
  }

  fileNames = fchooser.selectedFiles();

  QPixmap pixmap(width(), height());

  QPainter paint(&pixmap);

  drawWidget(&paint, width(), height());

  pixmap.save(fileNames.at(0), "PNG", 90);

  sidemenu->close();
}


void SignalCurve::print_to_printer()
{
  QPrinter printer(QPrinter::HighResolution);

  printer.setOutputFormat(QPrinter::NativeFormat);
  printer.setPageSize(QPrinter::A4);
  printer.setOrientation(QPrinter::Landscape);

  QPrintDialog printerdialog(&printer, this);
  printerdialog.setWindowTitle("Print");

  if(!(printerdialog.exec()==QDialog::Accepted))
  {
    sidemenu->close();

    return;
  }

  backup_colors_for_printing();

  QPainter paint(&printer);

  drawWidget_to_printer(&paint, printer.pageRect().width(), printer.pageRect().height());

  restore_colors_after_printing();

  sidemenu->close();
}


void SignalCurve::drawWidget_to_printer(QPainter *painter, int w, int h)
{
  int i,
      ruler_divisor,
      value_skip,
      precision,
      bordersize_backup=0;

  char str[128];

  double v_sens,
         offset,
         h_step,
         rasterstep,
         value,
         p_factor;

  QString q_str;

  QFont font;


  p_factor = (double)w / width();

  bordersize_backup = bordersize;
  bordersize *= p_factor;

  font.setFamily("Arial");
  font.setPixelSize((int)((double)w / 104.0));
  painter->setFont(font);

  if((w < ((bordersize * 2) + 5)) || (h < ((bordersize * 2) + 5)))
  {
    bordersize = bordersize_backup;

    return;
  }

/////////////////////////////////// draw the rulers ///////////////////////////////////////////

  if((drawHruler) && (h_rastersize > 1) && (bordersize > (19 * p_factor)))
  {
    painter->setPen(RulerColor);

    painter->drawLine(bordersize, h - bordersize + (5 * p_factor), w - bordersize, h - bordersize + (5 * p_factor));

    ruler_divisor = 10;

    rasterstep = (double)(w - (bordersize * 2)) / (h_rastersize * ruler_divisor);

    if(rasterstep < (7.0 * p_factor))
    {
      ruler_divisor = 5;

      rasterstep = (double)(w - (bordersize * 2)) / (h_rastersize * ruler_divisor);
    }

    if(rasterstep < (4.0 * p_factor))
    {
      ruler_divisor = 2;

      rasterstep = (double)(w - (bordersize * 2)) / (h_rastersize * ruler_divisor);
    }

    for(i = 0; i < ((h_rastersize * ruler_divisor) + 1); i++)
    {
      if(i % ruler_divisor)  painter->drawLine((i * rasterstep) + bordersize , h - bordersize + (5 * p_factor), (i * rasterstep) + bordersize, h - bordersize + ((5 + 4) * p_factor));
    }

    rasterstep = (double)(w - (bordersize * 2)) / h_rastersize;

    value_skip = 1;

    if(rasterstep < (60 * p_factor))  value_skip = 2;
    if(rasterstep < (30 * p_factor))  value_skip = 4;
    if(rasterstep < (15 * p_factor))  value_skip = 8;
    if((w - (bordersize * 2)) < (120 * p_factor))  value_skip = h_rastersize;

    for(i = 0; i < (h_rastersize + 1); i++)
    {
      painter->drawLine((i * rasterstep) + bordersize , h - bordersize + (5 * p_factor), (i * rasterstep) + bordersize, h - bordersize + ((5 + 10) * p_factor));

      if(!(i % value_skip))
      {
        value = h_ruler_startvalue - (((h_ruler_startvalue - h_ruler_endvalue) / h_rastersize) * i);

        q_str.setNum(value, 'f', h_ruler_precision);

        if(i == h_rastersize)
        {
          if(h_label[0] != 0)
          {
            q_str.append(" ");
            q_str.append(h_label);
          }
        }

        painter->drawText(bordersize + (i * rasterstep) - (30 * p_factor),  h - bordersize + (18 * p_factor), 60 * p_factor, 16 * p_factor, Qt::AlignCenter | Qt::TextSingleLine, q_str);
      }
    }
  }

  if((drawVruler) && (v_rastersize > 1) && (bordersize > (29 * p_factor)))
  {
    painter->setPen(RulerColor);

    painter->drawLine(bordersize - (5 * p_factor), bordersize, bordersize - (5 * p_factor), h - bordersize);

    ruler_divisor = 10;

    rasterstep = (double)(h - (bordersize * 2)) / (v_rastersize * ruler_divisor);

    if(rasterstep < (7.0 * p_factor))
    {
      ruler_divisor = 5;

      rasterstep = (double)(h - (bordersize * 2)) / (v_rastersize * ruler_divisor);
    }

    if(rasterstep < (4.0 * p_factor))
    {
      ruler_divisor = 2;

      rasterstep = (double)(h - (bordersize * 2)) / (v_rastersize * ruler_divisor);
    }

    for(i = 0; i < ((v_rastersize * ruler_divisor) + 1); i++)
    {
      if(i % ruler_divisor)  painter->drawLine(bordersize - (5 * p_factor), bordersize + (i * rasterstep), bordersize - ((5 + 4) * p_factor), bordersize + (i * rasterstep));
    }

    rasterstep = (double)(h - (bordersize * 2)) / v_rastersize;

    value_skip = 1;

    if(rasterstep < (40 * p_factor))  value_skip = 2;
    if(rasterstep < (20 * p_factor))  value_skip = 4;
    if(rasterstep < (10 * p_factor))  value_skip = 8;
    if((h - (bordersize * 2)) < (80 * p_factor))  value_skip = v_rastersize;

    for(i = 0; i < (v_rastersize + 1); i++)
    {
      painter->drawLine(bordersize - (5 * p_factor), bordersize + (i * rasterstep), bordersize - ((5 + 10) * p_factor), bordersize + (i * rasterstep));

      if(!(i % value_skip))
      {
        value = max_value - (((max_value - min_value) / v_rastersize) * i);
        precision = 0;
        if(max_value < 1000.0)
        {
          precision = 1;
        }
        if(max_value < 100.0)
        {
          precision = 2;
        }
        if(max_value < 10.0)
        {
          precision = 3;
        }
        if(max_value < 1.0)
        {
          precision = 4;
        }
        q_str.setNum(value, 'f', precision);
        painter->drawText(3 * p_factor, bordersize + (i * rasterstep) - (8 * p_factor), 40 * p_factor, 16 * p_factor, Qt::AlignRight | Qt::AlignVCenter | Qt::TextSingleLine, q_str);
      }
    }
  }

/////////////////////////////////// draw the labels ///////////////////////////////////////////

  painter->setPen(TextColor);
  painter->setFont(font);

  if(v_label[0] != 0)
  {
    painter->drawText(15 * p_factor, 30 * p_factor, 80 * p_factor, 16 * p_factor, Qt::AlignCenter | Qt::TextSingleLine, v_label);
  }

  if(upperlabel1[0] != 0)
  {
    painter->drawText(w / 2 - (150 * p_factor), 10 * p_factor, 300 * p_factor, 16 * p_factor, Qt::AlignCenter | Qt::TextSingleLine, upperlabel1);
  }

  if(upperlabel2[0] != 0)
  {
    painter->drawText(w / 2 - (150 * p_factor), 30 * p_factor, 300 * p_factor, 16 * p_factor, Qt::AlignCenter | Qt::TextSingleLine, upperlabel2);
  }

  if(lowerlabel[0] != 0)
  {
    painter->drawText(w / 2 - (80 * p_factor), h - (20 * p_factor), 160 * p_factor, 16 * p_factor, Qt::AlignCenter | Qt::TextSingleLine, lowerlabel);
  }

/////////////////////////////////// translate coordinates, draw and fill a rectangle ///////////////////////////////////////////

  painter->translate(QPoint(bordersize, bordersize));

  w -= (bordersize * 2);

  h -= (bordersize * 2);

  painter->setClipping(TRUE);
  painter->setClipRegion(QRegion(0, 0, w, h), Qt::ReplaceClip);

/////////////////////////////////// draw the rasters ///////////////////////////////////////////

  painter->setPen(RasterColor);

  painter->drawRect (0, 0, w - 1, h - 1);

  if(h_rastersize > 1)
  {
    rasterstep = (double)w / h_rastersize;

    for(i = 1; i < h_rastersize; i++)
    {
      painter->drawLine(i * rasterstep, 0, i * rasterstep, h);
    }
  }

  if(v_rastersize > 1)
  {
    rasterstep = (double)h / v_rastersize;

    for(i = 1; i < v_rastersize; i++)
    {
      painter->drawLine(0, i * rasterstep, w, i * rasterstep);
    }
  }

/////////////////////////////////// draw the curve ///////////////////////////////////////////

  if((dbuf == NULL) && (fbuf == NULL)) return;

  if(max_value == min_value)  return;

  if(bufsize < 2)  return;

  offset = (-(max_value));

  v_sens = (-(h / (max_value - min_value)));

  h_step = (double)w / (double)bufsize;

  painter->setPen(QPen(QBrush(SignalColor, Qt::SolidPattern), tracewidth, Qt::SolidLine, Qt::SquareCap, Qt::BevelJoin));

  if(dbuf)
  {
    for(i = 0; i < bufsize; i++)
    {
      if(bufsize < (w / 2))
      {
        painter->drawLine(i * h_step, (dbuf[i] + offset) * v_sens, (i + 1) * h_step, (dbuf[i] + offset) * v_sens);
        if(i)
        {
          painter->drawLine(i * h_step, (dbuf[i - 1] + offset) * v_sens, i * h_step, (dbuf[i] + offset) * v_sens);
        }
      }
      else
      {
        if(i < (bufsize - 1))
        {
          {
            painter->drawLine(i * h_step, (dbuf[i] + offset) * v_sens, (i + 1) * h_step, (dbuf[i + 1] + offset) * v_sens);
          }
        }
      }

      if(crosshair_1_active)
      {
        if(i==((int)(((double)crosshair_1_x_position * p_factor) / h_step)))
        {
          crosshair_1_y_value = (dbuf[i] + offset) * v_sens;
          crosshair_1_value = dbuf[i];
          value = (h_ruler_endvalue - h_ruler_startvalue) / bufsize;
          crosshair_1_value_2 = (i * value) + (0.5 * value) + h_ruler_startvalue;
        }
      }
    }
  }

  if(fbuf)
  {
    for(i = 0; i < bufsize; i++)
    {
      if(bufsize < (w / 2))
      {
        painter->drawLine(i * h_step, (fbuf[i] + offset) * v_sens, (i + 1) * h_step, (fbuf[i] + offset) * v_sens);
        if(i)
        {
          painter->drawLine(i * h_step, (fbuf[i - 1] + offset) * v_sens, i * h_step, (fbuf[i] + offset) * v_sens);
        }
      }
      else
      {
        if(i < (bufsize - 1))
        {
          {
            painter->drawLine(i * h_step, (fbuf[i] + offset) * v_sens, (i + 1) * h_step, (fbuf[i + 1] + offset) * v_sens);
          }
        }
      }

      if(crosshair_1_active)
      {
        if(i==((int)(((double)crosshair_1_x_position * p_factor) / h_step)))
        {
          crosshair_1_y_value = (fbuf[i] + offset) * v_sens;
          crosshair_1_value = fbuf[i];
          value = (h_ruler_endvalue - h_ruler_startvalue) / bufsize;
          crosshair_1_value_2 = (i * value) + (0.5 * value) + h_ruler_startvalue;
        }
      }
    }
  }

/////////////////////////////////// draw the cursor ///////////////////////////////////////////

  if(crosshair_1_active)
  {
    QPainterPath path;
    path.moveTo(crosshair_1_x_position * p_factor, crosshair_1_y_value);
    path.lineTo((crosshair_1_x_position - 4) * p_factor, crosshair_1_y_value - (9 * p_factor));
    path.lineTo((crosshair_1_x_position + 4) * p_factor, crosshair_1_y_value - (9 * p_factor));
    path.lineTo(crosshair_1_x_position * p_factor, crosshair_1_y_value);
    painter->fillPath(path, QBrush(Qt::black));

    snprintf(str, 128, "%f", crosshair_1_value);
    painter->drawText((crosshair_1_x_position + 8) * p_factor, (crosshair_1_y_position - 10) * p_factor, str);
    snprintf(str, 128, "%f %s", crosshair_1_value_2, h_label);
    painter->drawText((crosshair_1_x_position + 8) * p_factor, (crosshair_1_y_position + 10) * p_factor, str);
  }

  bordersize = bordersize_backup;
}


void SignalCurve::setUpdatesEnabled(bool enabled)
{
  updates_enabled = enabled;
}


void SignalCurve::paintEvent(QPaintEvent *)
{
  if(updates_enabled == TRUE)
  {
    QPainter paint(this);

    drawWidget(&paint, width(), height());

    old_w = width();
  }
}


void SignalCurve::drawWidget(QPainter *painter, int w, int h)
{
  int i,
      ruler_divisor,
      value_skip,
      precision;

  double v_sens,
         offset,
         h_step,
         rasterstep,
         value,
         pixelsPerHz,
         max_colorbar_value;

  char str[128];

  QString q_str;


  painter->setFont(QFont("Arial", 8));

  painter->fillRect(0, 0, w, h, BorderColor);

  if((w < ((bordersize * 2) + 5)) || (h < ((bordersize * 2) + 5)))
  {
    return;
  }

/////////////////////////////////// draw the rulers ///////////////////////////////////////////

  if((drawHruler) && (h_rastersize > 1) && (bordersize > 19))
  {
    painter->setPen(RulerColor);

    painter->drawLine(bordersize, h - bordersize + 5, w - bordersize, h - bordersize + 5);

    ruler_divisor = 10;

    rasterstep = (double)(w - (bordersize * 2)) / (h_rastersize * ruler_divisor);

    if(rasterstep < 7.0)
    {
      ruler_divisor = 5;

      rasterstep = (double)(w - (bordersize * 2)) / (h_rastersize * ruler_divisor);
    }

    if(rasterstep < 4.0)
    {
      ruler_divisor = 2;

      rasterstep = (double)(w - (bordersize * 2)) / (h_rastersize * ruler_divisor);
    }

    for(i = 0; i < ((h_rastersize * ruler_divisor) + 1); i++)
    {
      if(i % ruler_divisor)  painter->drawLine((i * rasterstep) + bordersize , h - bordersize + 5, (i * rasterstep) + bordersize, h - bordersize + 5 + 4);
    }

    rasterstep = (double)(w - (bordersize * 2)) / h_rastersize;

    value_skip = 1;

    if(rasterstep < 60)  value_skip = 2;
    if(rasterstep < 30)  value_skip = 4;
    if(rasterstep < 15)  value_skip = 8;
    if((w - (bordersize * 2)) < 120)  value_skip = h_rastersize;

    for(i = 0; i < (h_rastersize + 1); i++)
    {
      painter->drawLine((i * rasterstep) + bordersize , h - bordersize + 5, (i * rasterstep) + bordersize, h - bordersize + 5 + 10);

      if(!(i % value_skip))
      {
        value = h_ruler_startvalue - (((h_ruler_startvalue - h_ruler_endvalue) / h_rastersize) * i);

        q_str.setNum(value, 'f', h_ruler_precision);

        if(i == h_rastersize)
        {
          if(h_label[0] != 0)
          {
            q_str.append(" ");
            q_str.append(h_label);
          }
        }

        painter->drawText(bordersize + (i * rasterstep) - 30,  h - bordersize + 18, 60, 16, Qt::AlignCenter | Qt::TextSingleLine, q_str);
      }
    }
  }

  if((drawVruler) && (v_rastersize > 1) && (bordersize > 29))
  {
    painter->setPen(RulerColor);

    painter->drawLine(bordersize - 5, bordersize, bordersize - 5, h - bordersize);

    ruler_divisor = 10;

    rasterstep = (double)(h - (bordersize * 2)) / (v_rastersize * ruler_divisor);

    if(rasterstep < 7.0)
    {
      ruler_divisor = 5;

      rasterstep = (double)(h - (bordersize * 2)) / (v_rastersize * ruler_divisor);
    }

    if(rasterstep < 4.0)
    {
      ruler_divisor = 2;

      rasterstep = (double)(h - (bordersize * 2)) / (v_rastersize * ruler_divisor);
    }

    for(i = 0; i < ((v_rastersize * ruler_divisor) + 1); i++)
    {
      if(i % ruler_divisor)  painter->drawLine(bordersize - 5, bordersize + (i * rasterstep), bordersize - 5 - 4, bordersize + (i * rasterstep));
    }

    rasterstep = (double)(h - (bordersize * 2)) / v_rastersize;

    value_skip = 1;

    if(rasterstep < 40)  value_skip = 2;
    if(rasterstep < 20)  value_skip = 4;
    if(rasterstep < 10)  value_skip = 8;
    if((h - (bordersize * 2)) < 80)  value_skip = v_rastersize;

    for(i = 0; i < (v_rastersize + 1); i++)
    {
      painter->drawLine(bordersize - 5, bordersize + (i * rasterstep), bordersize - 5 - 10, bordersize + (i * rasterstep));

      if(!(i % value_skip))
      {
        value = max_value - (((max_value - min_value) / v_rastersize) * i);
        precision = 0;
        if(max_value < 1000.0)
        {
          precision = 1;
        }
        if(max_value < 100.0)
        {
          precision = 2;
        }
        if(max_value < 10.0)
        {
          precision = 3;
        }
        if(max_value < 1.0)
        {
          precision = 4;
        }
        q_str.setNum(value, 'f', precision);
        painter->drawText(3, bordersize + (i * rasterstep) - 8, 40, 16, Qt::AlignRight | Qt::AlignVCenter | Qt::TextSingleLine, q_str);
      }
    }
  }

/////////////////////////////////// draw the labels ///////////////////////////////////////////

  painter->setPen(TextColor);
  painter->setFont(QFont("Arial", 8));

  if(v_label[0] != 0)
  {
    painter->drawText(15, 30, 80, 16, Qt::AlignCenter | Qt::TextSingleLine, v_label);
  }

  if(upperlabel1[0] != 0)
  {
    painter->drawText(w / 2 - 150, 20, 300, 16, Qt::AlignCenter | Qt::TextSingleLine, upperlabel1);
  }

  if(lowerlabel[0] != 0)
  {
    painter->drawText(w / 2 - 80, h - 20, 160, 16, Qt::AlignCenter | Qt::TextSingleLine, lowerlabel);
  }

/////////////////////////////////// translate coordinates, draw and fill a rectangle ///////////////////////////////////////////

  painter->translate(QPoint(bordersize, bordersize));

  w -= (bordersize * 2);

  h -= (bordersize * 2);

  painter->fillRect(0, 0, w, h, BackgroundColor);

  painter->setClipping(TRUE);
  painter->setClipRegion(QRegion(0, 0, w, h), Qt::ReplaceClip);

/////////////////////////////////// draw the colorbars /////////////////////////////////////////

  if(spectrum_color != NULL)
  {
    max_colorbar_value = 0.0;

    for(i=0; i < spectrum_color->items; i++)
    {
      if(spectrum_color->value[i] > max_colorbar_value)
      {
        max_colorbar_value = spectrum_color->value[i];
      }
    }

    max_colorbar_value *= 1.05;

    pixelsPerHz = (double)w / (h_ruler_endvalue - h_ruler_startvalue);

    if((spectrum_color->freq[0] > h_ruler_startvalue) && (spectrum_color->items > 1))
    {
      painter->fillRect(0,
                        (max_colorbar_value - spectrum_color->value[0]) * ((double)h / max_colorbar_value),
                        (spectrum_color->freq[0] - h_ruler_startvalue) * pixelsPerHz,
                        h,
                        (Qt::GlobalColor)spectrum_color->color[0]);

      painter->setPen(Qt::lightGray);
      painter->drawText(10, 20, spectrum_color->label[0]);
    }

    for(i=1; i < spectrum_color->items; i++)
    {
      if(spectrum_color->freq[i] > h_ruler_startvalue)
      {
        painter->fillRect((spectrum_color->freq[i-1] - h_ruler_startvalue) * pixelsPerHz,
                          (max_colorbar_value - spectrum_color->value[i]) * ((double)h / max_colorbar_value),
                          (spectrum_color->freq[i] - spectrum_color->freq[i-1]) * pixelsPerHz,
                          h,
                          (Qt::GlobalColor)spectrum_color->color[i]);

        painter->setPen(Qt::lightGray);
        painter->drawText((spectrum_color->freq[i-1] - h_ruler_startvalue) * pixelsPerHz + 10, 20, spectrum_color->label[i]);
      }
    }
  }

/////////////////////////////////// draw the curve ///////////////////////////////////////////

  if(drawcurve_before_raster)
  {
    if((dbuf == NULL) && (fbuf == NULL)) return;

    if(max_value == min_value)  return;

    if(bufsize < 2)  return;

    offset = (-(max_value));

    v_sens = (-(h / (max_value - min_value)));

    h_step = (double)w / (double)bufsize;

    painter->setPen(QPen(QBrush(SignalColor, Qt::SolidPattern), tracewidth, Qt::SolidLine, Qt::SquareCap, Qt::BevelJoin));

    if(dbuf)
    {
      for(i = 0; i < bufsize; i++)
      {
        if(bufsize < (w / 2))
        {
          painter->drawLine(i * h_step, (dbuf[i] + offset) * v_sens, (i + 1) * h_step, (dbuf[i] + offset) * v_sens);
          if(i)
          {
            painter->drawLine(i * h_step, (dbuf[i - 1] + offset) * v_sens, i * h_step, (dbuf[i] + offset) * v_sens);
          }
        }
        else
        {
          if(i < (bufsize - 1))
          {
            {
              painter->drawLine(i * h_step, (dbuf[i] + offset) * v_sens, (i + 1) * h_step, (dbuf[i + 1] + offset) * v_sens);
            }
          }
        }

        if(crosshair_1_active)
        {
          if(i==((int)(((double)crosshair_1_x_position) / h_step)))
          {
            crosshair_1_y_value = (dbuf[i] + offset) * v_sens;
            crosshair_1_value = dbuf[i];
            value = (h_ruler_endvalue - h_ruler_startvalue) / bufsize;
            crosshair_1_value_2 = (i * value) + h_ruler_startvalue;
          }
        }
      }
    }

    if(fbuf)
    {
      for(i = 0; i < bufsize; i++)
      {
        if(bufsize < (w / 2))
        {
          painter->drawLine(i * h_step, (fbuf[i] + offset) * v_sens, (i + 1) * h_step, (fbuf[i] + offset) * v_sens);
          if(i)
          {
            painter->drawLine(i * h_step, (fbuf[i - 1] + offset) * v_sens, i * h_step, (fbuf[i] + offset) * v_sens);
          }
        }
        else
        {
          if(i < (bufsize - 1))
          {
            {
              painter->drawLine(i * h_step, (fbuf[i] + offset) * v_sens, (i + 1) * h_step, (fbuf[i + 1] + offset) * v_sens);
            }
          }
        }

        if(crosshair_1_active)
        {
          if(i==((int)(((double)crosshair_1_x_position) / h_step)))
          {
            crosshair_1_y_value = (fbuf[i] + offset) * v_sens;
            crosshair_1_value = fbuf[i];
            value = (h_ruler_endvalue - h_ruler_startvalue) / bufsize;
            crosshair_1_value_2 = (i * value) + h_ruler_startvalue;
          }
        }
      }
    }
  }

/////////////////////////////////// draw the rasters ///////////////////////////////////////////

  painter->setPen(RasterColor);

  painter->drawRect (0, 0, w - 1, h - 1);

  if(h_rastersize > 1)
  {
    rasterstep = (double)w / h_rastersize;

    for(i = 1; i < h_rastersize; i++)
    {
      painter->drawLine(i * rasterstep, 0, i * rasterstep, h);
    }
  }

  if(v_rastersize > 1)
  {
    rasterstep = (double)h / v_rastersize;

    for(i = 1; i < v_rastersize; i++)
    {
      painter->drawLine(0, i * rasterstep, w, i * rasterstep);
    }
  }

/////////////////////////////////// draw the curve ///////////////////////////////////////////

  if(!drawcurve_before_raster)
  {
    if((dbuf == NULL) && (fbuf == NULL)) return;

    if(max_value == min_value)  return;

    if(bufsize < 2)  return;

    offset = (-(max_value));

    v_sens = (-(h / (max_value - min_value)));

    h_step = (double)w / (double)bufsize;

    painter->setPen(QPen(QBrush(SignalColor, Qt::SolidPattern), tracewidth, Qt::SolidLine, Qt::SquareCap, Qt::BevelJoin));

    if(dbuf)
    {
      for(i = 0; i < bufsize; i++)
      {
        if(bufsize < (w / 2))
        {
          painter->drawLine(i * h_step, (dbuf[i] + offset) * v_sens, (i + 1) * h_step, (dbuf[i] + offset) * v_sens);
          if(i)
          {
            painter->drawLine(i * h_step, (dbuf[i - 1] + offset) * v_sens, i * h_step, (dbuf[i] + offset) * v_sens);
          }
        }
        else
        {
          if(i < (bufsize - 1))
          {
            {
              painter->drawLine(i * h_step, (dbuf[i] + offset) * v_sens, (i + 1) * h_step, (dbuf[i + 1] + offset) * v_sens);
            }
          }
        }

        if(crosshair_1_active)
        {
          if(i==((int)(((double)crosshair_1_x_position) / h_step)))
          {
            crosshair_1_y_value = (dbuf[i] + offset) * v_sens;
            crosshair_1_value = dbuf[i];
            value = (h_ruler_endvalue - h_ruler_startvalue) / bufsize;
            crosshair_1_value_2 = (i * value) + h_ruler_startvalue;
          }
        }
      }
    }

    if(fbuf)
    {
      for(i = 0; i < bufsize; i++)
      {
        if(bufsize < (w / 2))
        {
          painter->drawLine(i * h_step, (fbuf[i] + offset) * v_sens, (i + 1) * h_step, (fbuf[i] + offset) * v_sens);
          if(i)
          {
            painter->drawLine(i * h_step, (fbuf[i - 1] + offset) * v_sens, i * h_step, (fbuf[i] + offset) * v_sens);
          }
        }
        else
        {
          if(i < (bufsize - 1))
          {
            {
              painter->drawLine(i * h_step, (fbuf[i] + offset) * v_sens, (i + 1) * h_step, (fbuf[i + 1] + offset) * v_sens);
            }
          }
        }

        if(crosshair_1_active)
        {
          if(i==((int)(((double)crosshair_1_x_position) / h_step)))
          {
            crosshair_1_y_value = (fbuf[i] + offset) * v_sens;
            crosshair_1_value = fbuf[i];
            value = (h_ruler_endvalue - h_ruler_startvalue) / bufsize;
            crosshair_1_value_2 = (i * value) + h_ruler_startvalue;
          }
        }
      }
    }
  }

/////////////////////////////////// draw the cursor ///////////////////////////////////////////

  if(crosshair_1_active)
  {
    painter->setPen((Qt::GlobalColor)crosshair_1_color);

    QPainterPath path;
    path.moveTo(crosshair_1_x_position, crosshair_1_y_value);
    path.lineTo(crosshair_1_x_position - 4, crosshair_1_y_value - 10);
    path.lineTo(crosshair_1_x_position + 5, crosshair_1_y_value - 10);
    path.lineTo(crosshair_1_x_position, crosshair_1_y_value);
    painter->fillPath(path, QBrush((Qt::GlobalColor)crosshair_1_color));

    painter->setFont(QFont("Arial", 10));
    painter->fillRect(crosshair_1_x_position + 6, crosshair_1_y_position - 23, 60, 16, BackgroundColor);
    snprintf(str, 128, "%f", crosshair_1_value);
    painter->drawText(crosshair_1_x_position + 8, crosshair_1_y_position - 10, str);
    painter->fillRect(crosshair_1_x_position + 6, crosshair_1_y_position - 3, 60, 16, BackgroundColor);
    snprintf(str, 128, "%f %s", crosshair_1_value_2, h_label);
    painter->drawText(crosshair_1_x_position + 8, crosshair_1_y_position + 10, str);
  }

/////////////////////////////////// draw the buttons ///////////////////////////////////////////

  painter->setPen(Qt::black);
  painter->setFont(QFont("Arial", 8));
  if(printEnabled == TRUE)
  {
    painter->fillRect(w - 43, 3, 40, 18, Qt::gray);
    painter->drawText(w - 38, 16, "print");
  }
  if(cursorEnabled == TRUE)
  {
    painter->fillRect(w - 43, 23, 40, 18, Qt::gray);
    painter->drawText(w - 38, 36, "cursor");
  }
  if(dashBoardEnabled == TRUE)
  {
    painter->fillRect(w - 43, 43, 40, 18, Qt::gray);
    painter->drawText(w - 38, 56, "ctls");
  }
}


void SignalCurve::drawCurve(double *samplebuf, int size, double max_val, double min_val)
{
  dbuf = samplebuf;

  fbuf = NULL;

  bufsize = size;

  max_value = max_val;

  min_value = min_val;

  update();
}


void SignalCurve::drawCurve(float *samplebuf, int size, double max_val, double min_val)
{
  fbuf = samplebuf;

  dbuf = NULL;

  bufsize = size;

  max_value = max_val;

  min_value = min_val;

  update();
}


void SignalCurve::setH_RulerValues(double start, double end, int precision)
{
  h_ruler_startvalue = start;
  h_ruler_endvalue = end;
  h_ruler_precision = precision;

  update();
}


void SignalCurve::setSignalColor(QColor newColor)
{
  SignalColor = newColor;
  update();
}


void SignalCurve::setTraceWidth(int width)
{
  tracewidth = width;
  if(tracewidth < 0)  tracewidth = 0;
  update();
}


void SignalCurve::setBackgroundColor(QColor newColor)
{
  BackgroundColor = newColor;
  update();
}


void SignalCurve::setRasterColor(QColor newColor)
{
  RasterColor = newColor;
  update();
}


void SignalCurve::setBorderColor(QColor newColor)
{
  BorderColor = newColor;
  update();
}


void SignalCurve::setTextColor(QColor newColor)
{
  TextColor = newColor;
  update();
}


void SignalCurve::setH_RasterSize(int newsize)
{
  h_rastersize = newsize;
  update();
}


void SignalCurve::setV_RasterSize(int newsize)
{
  v_rastersize = newsize;
  update();
}


void SignalCurve::setBorderSize(int newsize)
{
  bordersize = newsize;
  if(bordersize < 0)  bordersize = 0;
  update();
}


void SignalCurve::setH_label(const char *str)
{
  strncpy(h_label, str, 32);
  h_label[31] = 0;
  update();
}


void SignalCurve::setV_label(const char *str)
{
  strncpy(v_label, str, 8);
  v_label[7] = 0;
  update();
}


void SignalCurve::setUpperLabel1(const char *str)
{
  strncpy(upperlabel1, str, 64);
  upperlabel1[63] = 0;
  update();
}


void SignalCurve::setUpperLabel2(const char *str)
{
  strncpy(upperlabel2, str, 64);
  upperlabel2[63] = 0;
  update();
}


void SignalCurve::setLowerLabel(const char *str)
{
  strncpy(lowerlabel, str, 64);
  lowerlabel[63] = 0;
  update();
}


void SignalCurve::setCursorEnabled(bool value)
{
  cursorEnabled = value;
}


void SignalCurve::setPrintEnabled(bool value)
{
  printEnabled = value;
}


void SignalCurve::setDashBoardEnabled(bool value)
{
  dashBoardEnabled = value;
}


void SignalCurve::enableSpectrumColors(struct spectrum_markersblock *spectr_col)
{
  spectrum_color = spectr_col;
  update();
}


void SignalCurve::disableSpectrumColors()
{
  spectrum_color = NULL;
  update();
}








