#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>

#include <iostream>

#include "v4l.h"

using namespace std;


V4L::V4L() :
  v4lfd(-1),
  current_frame(0),
  vgrab(NULL),
  capture_memory(NULL),
  current_video_channel(0),
  current_audio_channel(0),
  video_preview_enabled(false)
{
	sem_id=semget(IPC_PRIVATE, 1, IPC_CREAT | 0666 );
	semctl(sem_id, 0, SETVAL, 1);
}

V4L::~V4L() {
  _close();
	semctl(sem_id, 0, IPC_RMID, 0);
}

void V4L::lock() {
	struct sembuf op;
	op.sem_flg=0;
	op.sem_num=0;
  op.sem_op=-1;
  semop(sem_id, &op, 1);
}

void V4L::unlock() {
	struct sembuf op;
	op.sem_flg=0;
	op.sem_num=0;
  op.sem_op=1;
  semop(sem_id, &op, 1);
}

void V4L::open(const char *file_name) {
  lock();
  int res;
  
  v4l_file_name=file_name;
  v4lfd=::open(file_name, O_RDWR | O_EXCL);
  if(v4lfd==-1) {
    cerr<<"Can't open video device '"<<file_name<<"' : "<<strerror(errno)<<endl;
    unlock();
    return;
  }
 
  memset(&vcap, 0, sizeof(vcap));
  res=ioctl(v4lfd, VIDIOCGCAP, &vcap);
  if(res==-1) {
    cerr<<"Can't obtain capabilities of the video device '"<<file_name<<"' : "<<strerror(errno)<<endl;
    _close();
    unlock();
    return;
  }

  for(int i=0; i<vcap.channels; i++) {
    video_channel vc;
    memset(&vc, 0, sizeof(vc));
    vc.channel=i;
    res=ioctl(v4lfd, VIDIOCGCHAN, &vc);
    if(res==-1) {
      cerr<<"Can't obtain the description of video channel "<<vc.channel<<" : "<<strerror(errno)<<endl;
      _close();
      unlock();
      return;
      /* this code is for test purpose with the vloopback device
      memset(&vc, 0, sizeof(vc));
      vchan.push_back(vc);*/
    } else {
      vchan.push_back(vc);
    }
  }
   
  for(int i=0; i<vcap.audios; i++) {
    video_audio va;
    va.audio=i;
    res=ioctl(v4lfd, VIDIOCGAUDIO, &va);
    if(res==-1) {
      cerr<<"Can't obtain the description of audio channel "<<i<<" : "<<strerror(errno)<<endl;
      _close();
      unlock();
      return;
    } else {
      achan.push_back(va);
    }
  }

  res=ioctl(v4lfd, VIDIOCGMBUF, &vmbuf);
  if(res==-1) {
    cerr<<"Can't obtain the description of the video buffer mapping : "<<strerror(errno)<<endl;
    _close();
    unlock();
    return;
  }

  res=ioctl(v4lfd, VIDIOCGWIN, &vwind);
  if(res==-1) {
    cerr<<"Can't obtain the description of the video window : "<<strerror(errno)<<endl;
    _close();
    unlock();
    return;
  }

  res=ioctl(v4lfd, VIDIOCGPICT, &vpic);
  if(res==-1) {
    cerr<<"Can't obtain the description of the video picture : "<<strerror(errno)<<endl;
    _close();
    unlock();
    return;
  }

  if(vmbuf.size < vmbuf.frames*vcap.maxwidth*vcap.maxheight*3) {
    capture_memory_size=vmbuf.frames*vcap.maxwidth*vcap.maxheight*3;
    cerr<<"*** WARNING *** : the driver doesn't provide a correct size for memory mapping. DVR tries to correct this error, but some strange things may happend, you are warned."<<endl;
  } else {
    capture_memory_size=vmbuf.size;
  }
  
  capture_memory=(unsigned char*)mmap(0, capture_memory_size, PROT_READ|PROT_WRITE, MAP_SHARED, v4lfd, 0);
  if(capture_memory==MAP_FAILED) {
    capture_memory=NULL;
    cerr<<"Can't map memory for capture : "<<strerror(errno)<<endl;
    _close();
    unlock();
    return;
  }

  vgrab=new video_mmap[vmbuf.frames];
  for(int i=0; i<vmbuf.frames; i++) {
    vgrab[i].frame=i;
    vgrab[i].format=vpic.palette;
    vgrab[i].width=vwind.width;
    vgrab[i].height=vwind.height;
  }

  _setCaptureSize(vwind.width, vwind.height);

  // try to set the default pixel format, prefered is rgb24, then yuv420p and worst is rgb565
  if(!_setPixelFormat(PX_RGB24)) {
    if(!_setPixelFormat(PX_YUV420P)) {
      _setPixelFormat(PX_RGB565);
    }
  }
    
  switch(vpic.palette) {
    case VIDEO_PALETTE_RGB565:  pixel_format=PX_RGB565; break;
    case VIDEO_PALETTE_RGB24:   pixel_format=PX_RGB24; break;
    case VIDEO_PALETTE_YUV420P: pixel_format=PX_YUV420P; break;
      default:
      cerr<<"Unsupported pixel format ("<<vpic.palette<<")! Please contact the author, in order to handle this case."<<endl;
      _close();
      unlock();
      return;
  }

  unlock();
  return;
}

bool V4L::isGood() {
  return (v4lfd!=-1);
}

void V4L::close() {
  lock();
  _close();
  unlock();
}

void V4L::_close() {
  lock();
  if(v4lfd!=-1) {
    if(hasAudio()) {
      _enableAudio(false);
    }
    
    if(hasVideoPreview()) {
      _enableVideoPreview(false);
    }
    
    vchan.clear();
    achan.clear();
    
    if(vgrab) {
      delete []vgrab;
    }

    if(capture_memory) {
      munmap(capture_memory, capture_memory_size);
      capture_memory=NULL;
    }

    current_video_channel=0;
    current_audio_channel=0;
    current_frame=0;
    
    ::close(v4lfd);
    v4lfd=-1;
  }
  unlock();
}

string V4L::deviceFileName() {
  return v4l_file_name;
}

string V4L::deviceName() {
  return string(vcap.name);
}

unsigned char *V4L::captureFrame() {
  lock();
  unsigned char *f;
  int res;
  static int last_sync_frame=-vmbuf.frames+1;

  res=ioctl(v4lfd, VIDIOCMCAPTURE, &(vgrab[current_frame]));
  if(last_sync_frame>=0) {
    int res2=ioctl(v4lfd, VIDIOCSYNC, &last_sync_frame);
    DEBUG_OUTPUT(res2, "Can't sync");
  }
  if(res==-1) {
    DEBUG_OUTPUT(res, "Can't capture frame");
    unlock();
    return NULL;
  }

  f=(unsigned char*)(capture_memory)+vmbuf.offsets[current_frame];
  
  last_sync_frame++;
  if(last_sync_frame==vmbuf.frames) {
    last_sync_frame=0;
  }

  
  current_frame++;
  if(current_frame==vmbuf.frames) {
    current_frame=0;
  }

  unlock();
  return f;
}

bool V4L::hasVideoPreview() {
  return vcap.type & VID_TYPE_OVERLAY;
}

void V4L::enableVideoPreview(bool on) {
  lock();
  _enableVideoPreview(on);
  unlock();
}

void V4L::_enableVideoPreview(bool on) {
  if(video_preview_enabled==on) {
    return;
  }

  int res, a;
  a=(on?1:0);
  res=ioctl(v4lfd, VIDIOCCAPTURE, &a);
  DEBUG_OUTPUT(res, NULL);
  
  video_preview_enabled=on;
}

bool V4L::isVideoPreviewEnabled() {
  return video_preview_enabled;
}

vector<video_channel> &V4L::videoChannels() {
  return vchan;
}

vector<video_audio> &V4L::audioChannels() {
  return achan;
}

unsigned int V4L::currentVideoChannel() {
  return current_video_channel;
}

unsigned int V4L::currentAudioChannel() {
  return current_audio_channel;
}

void V4L::getMinSize(int &width, int &height) {
  width=vcap.minwidth;
  height=vcap.minheight;
}

void V4L::getMaxSize(int &width, int &height) {
  width=vcap.maxwidth;
  height=vcap.maxheight;
}

void V4L::setCurrentVideoChannel(unsigned int c) {
  lock();
  if(vchan.size()<=c) {
    unlock();
    return;
  }
    
  current_video_channel=c;
  int res=ioctl(v4lfd, VIDIOCSCHAN, &vchan[current_video_channel]);
  if(res==-1) {
    cerr<<"Can't set the current video channel : "<<strerror(errno)<<endl;
    _close();
  }
  unlock();
}

void V4L::setCurrentAudioChannel(unsigned int c) {
  lock();
  _setCurrentAudioChannel(c);
  unlock();
}

void V4L::_setCurrentAudioChannel(unsigned int c) {
  if(achan.size()<=c) {
    return;
  }

  current_audio_channel=c;
  int res=ioctl(v4lfd, VIDIOCSAUDIO, &achan[current_audio_channel]);
  if(res==-1) {
    cerr<<"Can't set the current audio channel : "<<strerror(errno)<<endl;
    _close();
    return;
  }
}

void V4L::getWindowPosition(unsigned int &x, unsigned int &y) {
  x=vwind.x;
  y=vwind.y;
}

void V4L::getWindowSize(unsigned int &width, unsigned int &height) {
  width=vwind.width;
  height=vwind.height;
}

void V4L::setWindowPositionSize(unsigned int x, unsigned int y, unsigned int width, unsigned int height) {
  lock();
  int res;

  if(vwind.x==x && vwind.y==y && vwind.width==width && vwind.height==height) {
    unlock();
    return;
  }

  vwind.clipcount=0;
  vwind.flags=0;
  vwind.x=x;
  vwind.y=y;
  vwind.width=width;
  vwind.height=height;
 
  res=ioctl(v4lfd, VIDIOCSWIN, &vwind);
  DEBUG_OUTPUT(res, "Can't set the window size and position");

  res=ioctl(v4lfd, VIDIOCGWIN, &vwind);
  DEBUG_OUTPUT(res, "Can't get the window size and position");

  unlock();
}

void V4L::setCaptureSize(unsigned int width, unsigned int height) {
  lock();
  _setCaptureSize(width, height);
  unlock();
}

void V4L::_setCaptureSize(unsigned int width, unsigned int height) {
  for(int i=0; i<vmbuf.frames; i++) {
    vgrab[i].width=width;
    vgrab[i].height=height;
  }
}

void V4L::getCaptureSize(unsigned int &width, unsigned int &height, int frame) {
  width=vgrab[frame].width;
  height=vgrab[frame].height;
}

/** return true if ok */
bool V4L::setPixelFormat(PixelFormat pxfmt) {
  lock();
  bool res=_setPixelFormat(pxfmt);
  unlock();
  return res;
}

bool V4L::_setPixelFormat(PixelFormat pxfmt) {
  unsigned int palette, old_palette=vpic.palette;
  int old_depth=vpic.depth;

  switch(pxfmt) {
    case PX_RGB565:  palette=VIDEO_PALETTE_RGB565; vpic.depth=16; break;
    case PX_RGB24:   palette=VIDEO_PALETTE_RGB24; vpic.depth=24; break;
    case PX_YUV420P: palette=VIDEO_PALETTE_YUV420P; vpic.depth=24; break;
    default: return false;
  };
  

  vpic.palette=palette;

  int res=ioctl(v4lfd, VIDIOCSPICT, &vpic);
  if(res==-1) {
    vpic.depth=old_depth;
    vpic.palette=old_palette;
    return false;
  } else {
    for(int i=0; i<vmbuf.frames; i++) {
      vgrab[i].format=palette;
    }
    pixel_format=pxfmt;
    return true;
  }
}

V4L::PixelFormat V4L::pixelFormat() {
  return pixel_format;
}

bool V4L::hasAudio() {
  return achan.size()!=0;
}

bool V4L::isAudioEnabled() {
  if(achan.size()==0) {
    return false;
  }

  return !(achan[current_audio_channel].flags & VIDEO_AUDIO_MUTE);
}

void V4L::enableAudio(bool on) {
  lock();
  _enableAudio(on);
  unlock();
}

void V4L::_enableAudio(bool on) {
  if(on) {
    achan[current_audio_channel].flags &= ~VIDEO_AUDIO_MUTE;
  } else {
    achan[current_audio_channel].flags |= VIDEO_AUDIO_MUTE;
  }
  _setCurrentAudioChannel(current_audio_channel);
}

bool V4L::hasTuner() {
  return vcap.type & VID_TYPE_OVERLAY;
}

void V4L::setTunerFrequency(unsigned long frequency) {
  lock();
  int res=ioctl(v4lfd, VIDIOCSFREQ, &frequency);
  if(res==-1) {
   cerr<<"Can't set the tuner frequency : "<<strerror(errno)<<endl;
   _close();
  }
  unlock();
}

unsigned long V4L::tunerFrequency() {
  lock();
  unsigned long frequency;
  int res=ioctl(v4lfd, VIDIOCGFREQ, &frequency);
  if(res==-1) {
    cerr<<"Can't get the tuner frequency : "<<strerror(errno)<<endl;
    unlock();
    return 0;
  } else {
    unlock();
    return frequency;
  }
}

vector<V4LDevice> V4L::availableDevicesList() {
  int res;
  V4LDevice device;
  vector<string> filenames;
  vector<V4LDevice> devices;
 
  lock();
  
  filenames.push_back("/dev/video");

  for(int i=0; i<7; i++) {
    filenames.push_back(string("/dev/video")+char('0'+i));
  }
  for(int i=0; i<7; i++) {
    filenames.push_back(string("/dev/video/video")+char('0'+i));
  }
  for(int i=0; i<7; i++) {
    filenames.push_back(string("/dev/v4l/video")+char('0'+i));
  }


  for(unsigned long d=0; d<filenames.size(); d++) {
    int f=::open(filenames[d].c_str(), O_RDWR | O_EXCL);
    if(f!=-1) {
      struct video_capability vcap;
      memset(&vcap, 0, sizeof(vcap));
      res=ioctl(f, VIDIOCGCAP, &vcap);
      if(res!=-1) {
        device.file_name=filenames[d].c_str();
        device.name=vcap.name;
        devices.push_back(device);
      }
      ::close(f);
    }
  }

  unlock();

  if(devices.size()==0) {
    cerr<<"No device found"<<endl;
  }
  
  return devices;
}
