/* 
    STV0299 - QPSK demodulator

    Copyright (C) 2001 Convergence Integrated Media GmbH <ralph@convergence.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/    

#include <linux/module.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/poll.h>
#include <asm/io.h>
#include <linux/i2c.h>

#include "dvb_frontend.h"

#ifdef MODULE
MODULE_PARM(debug,"i");
MODULE_DESCRIPTION("STV0299 DVB demodulator driver");
MODULE_AUTHOR("Ralph Metzler");
#ifdef MODULE_LICENSE
MODULE_LICENSE("GPL");
#endif
#endif

static int debug = 0;
#define dprintk	if (debug) printk

static struct i2c_driver dvbt_driver;
static struct i2c_client client_template;


#define M_CLK (88000000UL)

/* M=21, K=0, P=0, f_VCO = 4MHz*4*(M+1)/(K+1) = 352 MHz */

static u8
init_tab[] = {
	0x01, 0x15,   // M: 0x15 DIRCLK: 0 K:0
	0x02, 0x30,   // P: 0  SERCLK: 0 VCO:ON  STDBY:0
 
	0x03, 0x00,
	0x04, 0x88,   // F22FR, F22=
	0x05, 0x0f,   // SDAT:0 SCLT:0 I2CT:0
	0x06, 0x00,   // DAC mode and MSB 
	0x07, 0x00,   // DAC LSB
	0x08, 0x40,   // DiSEqC
	0x09, 0x00,
	0x0a, 0x42, 
	0x0c, 0x51,   // QPSK reverse:1  Nyquist:0 OP0 val:1 OP0 con:1 OP1 val:1 OP1 con:1 
	0x0d, 0x82,
	0x0e, 0x23,
	0x0f, 0x52,

	0x10, 0x3d,   // AGC2
	0x11, 0x84,   
	0x12, 0xb5,   // Lock detect: -64  Carrier freq detect:on
	0x13, 0xb6,   // alpha_car b:4 a:0  noise est:256ks  derot:on 
	0x14, 0x93,   // beat carc:0 d:0 e:0xf  phase detect algo: 1
	0x15, 0xc9,   // lock detector threshold
	
	0x16, 0x1d,
	0x17, 0x0,
	0x18, 0x14,
	0x19, 0xf2,

	0x1a, 0x11,

	0x1b, 0x9c,
	0x1c, 0x0,
	0x1d, 0x0,
	0x1e, 0xb,

	0x22, 0x00,
	0x23, 0x00,
	0x24, 0xff,
	0x25, 0xff,
	0x26, 0xff,

	0x28, 0x00,  // out imp: normal  out type: parallel FEC mode:0 
	0x29, 0x1e,  // 1/2 threshold
	0x2a, 0x14,  // 2/3 threshold
	0x2b, 0x0f,  // 3/4 threshold
	0x2c, 0x09,  // 5/6 threshold
	0x2d, 0x05,  // 7/8 threshold
	0x2e, 0x01,

	0x31, 0x1f,  // test all FECs 

	0x32, 0x19,  // viterbi and synchro search
	0x33, 0xfc,  // rs control
	0x34, 0x11,  // error control

	
	0x0b, 0x00, 
	0x27, 0x00, 0x2f, 0x00, 0x30, 0x00,
	0x35, 0x00, 0x36, 0x00, 0x37, 0x00, 
	0x38, 0x00, 0x39, 0x00, 0x3a, 0x00, 0x3b, 0x00, 
	0x3c, 0x00, 0x3d, 0x00, 0x3e, 0x00, 0x3f, 0x00,
	0x40, 0x00, 0x41, 0x00, 0x42, 0x00, 0x43, 0x00,
	0x44, 0x00, 0x45, 0x00, 0x46, 0x00, 0x47, 0x00, 
	0x48, 0x00, 0x49, 0x00, 0x4a, 0x00, 0x4b, 0x00, 
	0x4c, 0x00, 0x4d, 0x00, 0x4e, 0x00, 0x4f, 0x00, 
	0xff
};

struct stv0299 {
        u32 srate;
        u8 fec;
        u8 inv;
	u8 aclk;
	u8 bclk;
};

static int 
writereg(struct i2c_client *client, int reg, int data)
{
        int ret;
        unsigned char msg[] = {0x00, 0x00};
        
        msg[0]=reg; msg[1]=data;
        ret=i2c_master_send(client, msg, 2);
        if (ret!=2) 
	        printk("stv0299: writereg error\n");
        return ret;
}

static u8 
readreg(struct i2c_client *client, u8 reg)
{
        struct i2c_adapter *adap=client->adapter;
        unsigned char mm1[] = {0x1e};
        unsigned char mm2[] = {0x00};
        struct i2c_msg msgs[2];
        
        msgs[0].flags=0;
        msgs[1].flags=I2C_M_RD;
        msgs[0].addr=msgs[1].addr=client->addr;
        mm1[0]=reg;
        msgs[0].len=1; msgs[1].len=1;
        msgs[0].buf=mm1; msgs[1].buf=mm2;
        i2c_transfer(adap, msgs, 2);
        
        return mm2[0];
}

static int 
init(struct i2c_client *client)
{
        struct stv0299 *stv=(struct stv0299 *) client->data;
        u8 *it=init_tab;
	
        dprintk("stv0299: init chip\n");

	while (it[0]!=0xff) {
		writereg(client, it[0], it[1]);
		it+=2;
	}

        stv->srate=0;
        stv->fec=9;
        stv->inv=0;

        return 0;
}

static inline 
void ddelay(int i) 
{
        current->state=TASK_INTERRUPTIBLE;
        schedule_timeout((HZ*i)/100);
}

static int SetInversion(struct i2c_client *client, int inversion)
{
        struct stv0299 *stv=(struct stv0299 *) client->data;
	u8 val;

	if (inversion == stv->inv) 
		return 0;
	stv->inv=inversion;
	switch (inversion) {
	case INVERSION_AUTO: // Hmm, return EINVAL or leave it like this?
	default:
	case INVERSION_OFF:
		val=0x01;
		break;
	case INVERSION_ON:
		val=0x00;
		break;
	}
	writereg(client, 0x0c, 
		 (readreg(client, 0x0c)&0xfe)|val);
	return 0;
}

static int 
SetFEC(struct i2c_client *client, u8 fec)
{
        struct stv0299 *stv=(struct stv0299 *) client->data;
        u8 val=0x1f;

        if (stv->fec==fec)
                return 0;
        stv->fec=fec;
	
	switch (fec) {
	default:
	case FEC_AUTO:
		val=0x1f;
		break;
	case FEC_1_2:
		val=0x01;
		break;
	case FEC_2_3:
		val=0x02;
		break;
	case FEC_3_4:
		val=0x04;
		break;
	case FEC_5_6:
		val=0x08;
		break;
	case FEC_7_8:
		val=0x10;
		break;
	}
	writereg(client, 0x31, val);
        return 0;
}

static int 
SetSymbolrate(struct i2c_client *client, u32 srate)
{
        struct stv0299 *stv=(struct stv0299 *) client->data;
        u32 ratio;
	u32 tmp;
	u8 aclk=0xb4, bclk=0x51;

	if (srate>M_CLK)
                srate=M_CLK;
        if (srate<500000)
                srate=500000;
        stv->srate=srate;

	if (srate<30000000) { aclk=0xb6; bclk=0x53; }
	if (srate<14000000) { aclk=0xb7; bclk=0x53; }
	if (srate< 7000000) { aclk=0xb7; bclk=0x4f; }
	if (srate< 3000000) { aclk=0xb7; bclk=0x4b; }
	if (srate< 1500000) { aclk=0xb7; bclk=0x47; }

#define FIN (M_CLK>>4)
        tmp=srate<<4;
	ratio=tmp/FIN;
        
	tmp=(tmp%FIN)<<8;
	ratio=(ratio<<8)+tmp/FIN;
        
	tmp=(tmp%FIN)<<8;
	ratio=(ratio<<8)+tmp/FIN;
  
	ratio&=0xfffff0;

	writereg(client, 0x13, aclk);
	writereg(client, 0x14, bclk);
	writereg(client, 0x1f, (ratio>>16)&0xff);
	writereg(client, 0x20, (ratio>> 8)&0xff);
	writereg(client, 0x21, (ratio    )&0xff);

	stv->aclk=aclk;
	stv->bclk=bclk&0x3f;
	
	return 1;
}

static int attach_adapter(struct i2c_adapter *adap)
{
        struct stv0299 *stv;
        struct i2c_client *client;
        
        client_template.adapter=adap;
        
        if (i2c_master_send(&client_template,NULL,0))
                return -1;
        
        client_template.adapter=adap;
        
        if ((readreg(&client_template, 0x00))!=0xa1)
                return -1;
        
	  
        if (NULL == (client = kmalloc(sizeof(struct i2c_client), GFP_KERNEL)))
                return -ENOMEM;
        memcpy(client, &client_template, sizeof(struct i2c_client));
        
        client->data=stv=kmalloc(sizeof(struct stv0299),GFP_KERNEL);
        if (stv==NULL) {
                kfree(client);
                return -ENOMEM;
        }
       
        i2c_attach_client(client);
        init(client);

        dprintk("stv0299: attaching stv0299 at 0x%02x ", (client->addr)<<1);
        dprintk("to adapter %s\n", adap->name);
        writereg(client, 0x05, 0x80);
	MOD_INC_USE_COUNT;
        return 0;
}

static int detach_client(struct i2c_client *client)
{
        dprintk("stv0299: detach_client\n");
        i2c_detach_client(client);
        kfree(client->data);
        kfree(client);
	MOD_DEC_USE_COUNT;
        return 0;
}


static int dvb_command(struct i2c_client *client, unsigned int cmd, void *arg)
{
        //struct stv0299 *stv=(struct stv0299 *) client->data;
                
        switch (cmd) {
        case FE_READ_STATUS:
	{
		FrontendStatus *status=(FrontendStatus *) arg;
		int sync;

		*status=0;

                sync=readreg(client,0x1b);
		if (sync&0x80) // FIXME: find criteria for having signal 
			*status|=FE_HAS_SIGNAL;
		if (sync&0x80)
			*status|=FE_HAS_CARRIER;
		if (sync&0x10)
			*status|=FE_HAS_VITERBI;
		if (sync&8)
			*status|=FE_HAS_SYNC;
		if ((sync&0x98)==0x98)
			*status|=FE_HAS_LOCK;

                sync=readreg(client,0x0c);
		if (!(sync&1))
			*status|=FE_SPECTRUM_INV;
		break;
	}
        case FE_READ_BER:
	{
		u32 *ber=(u32 *) arg;
		*ber=0;
		break;
	}
        case FE_READ_SIGNAL_STRENGTH:
	{
		s32 *signal=(s32 *) arg;
		*signal=0;
		break;
	}
        case FE_READ_SNR:
	{
		s32 *snr=(s32 *) arg;
		*snr=0;
		break;
	}
	case FE_READ_UNCORRECTED_BLOCKS: 
	{
		u32 *ublocks=(u32 *) arg;
		*ublocks=0;
		break;
	}
        case FE_READ_AFC:
	{
		s32 *afc=(s32 *) arg;
		*afc=0;
		break;
	}
        case FE_WRITEREG:
        {
                u8 *msg = (u8 *) arg;
                writereg(client, msg[0], msg[1]);

                break;
        }
        case FE_READREG:
        {
                u8 *msg = (u8 *) arg;
                msg[1]=readreg(client, msg[0]);
                break;
        }
        case FE_INIT:
        {
                init(client);
                break;
        }

        case FE_SET_FRONTEND:
        {
		FrontendParameters *param = (FrontendParameters *) arg;

		SetInversion(client, param->Inversion);
                SetFEC(client, param->u.qpsk.FEC_inner);
                SetSymbolrate(client, param->u.qpsk.SymbolRate);
                break;
        }

        case FE_RESET:
        {
		writereg(client, 0x22, 0x00);
		writereg(client, 0x23, 0x00);
		readreg(client, 0x23);
		writereg(client, 0x12, 0xb9);
                break;
        }

#if 0
        case DVB_GET_FRONTEND:
        {
                struct frontend *front = (struct frontend *)arg;
		u8 vstatus=readreg(client, 0x1b);

                front->afc=0;//((int)((char)(readreg(client,0x0a)<<1)))/2;
                front->afc=0;//(front->afc*(int)(front->srate/8))/16;
                front->agc=0;//(readreg(client,0x0b)<<8);
                front->nest=(readreg(client,0x24)<<8)|readreg(client,0x25);

		if (front->sync) {
			writereg(client, 0x14, 0x80 | stv->bclk); 
			writereg(client, 0x12, 0x39 ); 
		}
		
		/*
		if (vstatus&0x90)
		  printk("vstatus=0x%02x  cldi=0x%02x  nest=0x%04x\n", 
			vstatus, readreg(client, 0x1c), 
			front->nest);
		*/
                front->vber = 0;//readreg(client,0x15);
                break;
        }
#endif
        default:
                return -1;
        }
        
        return 0;
} 

static void inc_use (struct i2c_client *client)
{
#ifdef MODULE
        MOD_INC_USE_COUNT;
#endif
}

static void dec_use (struct i2c_client *client)
{
#ifdef MODULE
        MOD_DEC_USE_COUNT;
#endif
}

static struct i2c_driver dvbt_driver = {
        "stv0299 DVB demodulator",
        I2C_DRIVERID_STV0299,
        I2C_DF_NOTIFY,
        attach_adapter,
        detach_client,
        dvb_command,
        inc_use,
        dec_use,
};

static struct i2c_client client_template = {
        name:    "stv0299",
        id:      I2C_DRIVERID_STV0299,
        flags:   0,
        addr:    (0xd0 >> 1),
        adapter: NULL,
        driver:  &dvbt_driver,
};


static __init int
init_stv0299(void) {
        int res;
        
        if ((res = i2c_add_driver(&dvbt_driver))) 
        {
                printk("stv0299: I2C driver registration failed\n");
                return res;
        }
        
        dprintk("stv0299: init done\n");
        return 0;
}

static __exit void
exit_stv0299(void)
{
        int res;
        
        if ((res = i2c_del_driver(&dvbt_driver))) 
        {
                printk("dvb-tuner: Driver deregistration failed, "
                       "module not removed.\n");
        }
        dprintk("stv0299: cleanup\n");
}

module_init(init_stv0299);
module_exit(exit_stv0299);
