#ifndef DUNE_GEOGRID_CORNERSTORAGE_HH
#define DUNE_GEOGRID_CORNERSTORAGE_HH

#include <dune/grid/geometrygrid/coordfunctioncaller.hh>

namespace Dune
{

  namespace GeoGrid
  {

    // CoordVector
    // -----------

    template< int mydim, class Grid, bool fake >
    class CoordVector;


    template< int mydim, class Grid >
    class CoordVector< mydim, Grid, false >
    {
      typedef typename remove_const< Grid >::type::Traits Traits;

      typedef typename Traits::ctype ctype;

      static const int dimension = Traits::dimension;
      static const int mydimension = mydim;
      static const int codimension = dimension - mydimension;
      static const int dimensionworld = Traits::dimensionworld;

      typedef FieldVector< ctype, dimensionworld > Coordinate;

      typedef typename Traits::HostGrid HostGrid;
      typedef typename Traits::CoordFunction CoordFunction;

      typedef typename HostGrid::template Codim< codimension >::Entity HostEntity;

      typedef GeoGrid :: CoordFunctionCaller< HostEntity, typename CoordFunction::Interface >
        CoordFunctionCaller;

    public:
      CoordVector ( const HostEntity &hostEntity,
                    const CoordFunction &coordFunction )
      : coordFunctionCaller_( hostEntity, coordFunction )
      {}

      template< unsigned int numCorners >
      void calculate ( Coordinate (&corners)[ numCorners ] ) const
      {
        assert( numCorners == coordFunctionCaller_.numCorners() );
        for( unsigned int i = 0; i < numCorners; ++i )
          coordFunctionCaller_.evaluate( i, corners[ i ] );
      }

    private:
      const CoordFunctionCaller coordFunctionCaller_;
    };


    template< int mydim, class Grid >
    class CoordVector< mydim, Grid, true >
    {
      typedef typename remove_const< Grid > :: type :: Traits Traits;

      typedef typename Traits::ctype ctype;

      static const int dimension = Traits::dimension;
      static const int mydimension = mydim;
      static const int codimension = dimension - mydimension;
      static const int dimensionworld = Traits::dimensionworld;

      typedef FieldVector< ctype, dimensionworld > Coordinate;

      typedef typename Traits::HostGrid HostGrid;
      typedef typename Traits::CoordFunction CoordFunction;

      typedef typename HostGrid::template Codim< 0 >::Entity HostElement;

      typedef GeoGrid::CoordFunctionCaller< HostElement, typename CoordFunction::Interface >
        CoordFunctionCaller;

    public:
      CoordVector ( const HostElement &hostElement,
                    const unsigned int subEntity,
                    const CoordFunction &coordFunction )
      : coordFunctionCaller_( hostElement, coordFunction ),
        subEntity_( subEntity )
      {}

      template< unsigned int numCorners >
      void calculate ( Coordinate (&corners)[ numCorners ] ) const
      {
        const GeometryType type = coordFunctionCaller_.type();
        const GenericReferenceElement< ctype, dimension > &refElement
          = GenericReferenceElements< ctype, dimension >::general( type );
        assert( numCorners == refElement.size( subEntity_, codimension, dimension ) );

        for( unsigned int i = 0; i < numCorners; ++i )
        {
          const unsigned int j = refElement.subEntity( subEntity_, codimension, i, dimension );
          coordFunctionCaller_.evaluate( j, corners[ i ] );
        }
      }

    private:
      const CoordFunctionCaller coordFunctionCaller_;
      const unsigned int subEntity_;
    };



    // IntersectionCoordVector
    // -----------------------

    template< class Grid >
    class IntersectionCoordVector
    {
      typedef typename remove_const< Grid >::type::Traits Traits;

      typedef typename Traits::ctype ctype;

      static const int dimension = Traits::dimension;
      static const int codimension = 1;
      static const int mydimension = dimension-codimension;
      static const int dimensionworld = Traits::dimensionworld;

      typedef FieldVector< ctype, dimensionworld > Coordinate;

      typedef typename Traits::HostGrid HostGrid;

      typedef typename Traits::template Codim< 0 >::GeometryImpl ElementGeometryImpl;
      typedef typename Traits::template Codim< codimension >::LocalGeometry HostLocalGeometry;

    public:
      IntersectionCoordVector ( const ElementGeometryImpl &elementGeometry,
                                const HostLocalGeometry &hostLocalGeometry )
      : elementGeometry_( elementGeometry ),
        hostLocalGeometry_( hostLocalGeometry )
      {}

      template< unsigned int numCorners >
      void calculate ( Coordinate (&corners)[ numCorners ] ) const
      {
        assert( numCorners == hostLocalGeometry_.corners() );
        for( unsigned int i = 0; i < numCorners; ++i )
          corners[ i ] = elementGeometry_.global( hostLocalGeometry_.corner( i ) );
      }

    private:
      const ElementGeometryImpl &elementGeometry_;
      HostLocalGeometry hostLocalGeometry_;
    };




    // CornerStorage
    // -------------

    template< class Topology, class Grid >
    class CornerStorage
    {
      typedef typename remove_const< Grid >::type::Traits Traits;

      typedef typename Traits::ctype ctype;

      static const int dimension = Traits::dimension;
      static const int mydimension = Topology::dimension;
      static const int codimension = dimension - mydimension;
      static const int dimensionworld = Traits::dimensionworld;

      typedef FieldVector< ctype, dimensionworld > Coordinate;

    public:
      static const unsigned int size = Topology::numCorners;

      template< class SubTopology >
      struct SubStorage
      {
        typedef CornerStorage< SubTopology, Grid > type;
      };

      template< bool fake >
      explicit
      CornerStorage ( const CoordVector< mydimension, Grid, fake > &coords )
      {
        coords.calculate( coords_ );
      }

      explicit CornerStorage ( const IntersectionCoordVector< Grid > &coords )
      {
        coords.calculate( coords_ );
      }

      template< class Mapping, unsigned int codim >
      explicit
      CornerStorage ( const GenericGeometry::SubMappingCoords< Mapping, codim > &coords )
      {
        for( unsigned int i = 0; i < size; ++i )
          coords_[ i ] = coords[ i ];
      }

      const Coordinate &operator[] ( unsigned int i ) const
      {
        return coords_[ i ];
      }

    private:
      Coordinate coords_[ size ];
    };

  } // namespace GeoGrid

} // namespace Dune

#endif // #ifndef DUNE_GEOGRID_CORNERSTORAGE_HH
