/* Drip - a transcoder for Unix
 * Copyright (C) 2001-2003 Jarl van Katwijk
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * cdinfo.c - reads a cd\dvd for volume title, publisher and preparer
 *
 * work by Steffen Solyga <solyga@tetibm3.ee.tu-berlin.de>
 * 
 */

#include "cdinfo.h"

ssize_t
my_read( int fd, void* buf, size_t nbtr ) {
/* returns number of bytes read or -1 on error */
/* like read(2) but nbr<nbtr only if eof reached */
  ssize_t nbr;
  ssize_t tnbr= 0;
  size_t rem= nbtr;
  unsigned char *p= (unsigned char*)buf;
  do {
    if( (nbr=read(fd,p+tnbr,rem)) == -1 ) return( -1 );
    tnbr+= nbr;
    rem= nbtr - tnbr;
  }
  while( nbr>0 && rem>0 );
  return( tnbr );
}


long
iso_num_733( char* p ) {
  return(   (p[0] & 0xffL)
          | (p[1] & 0xffL) <<  8
          | (p[2] & 0xffL) << 16
          | (p[3] & 0xffL) << 24 );
}


char*
iso_str( char* p, int n ) {
  static char string[256];
  int i = 0;
  while( i < n ) {
     string[i] = p[i];
     i++;
  }
  string[i--]= '\0';
  while( string[i] == ' ' ) string[i--]= '\0';
  return( string );
}


char*
iso_date( char* p ) { /* 14 chars          "1999015220021" */
  static char date[] = "1999-01-15 22:00:19";
  int i= 0;
  int j= 0;
  int k;
  while( i < 4 ) date[j++] = p[i++];
  for( k=6; k<16; k+=2 ) {
    j++;
    while( i < k ) date[j++] = p[i++];
  }
  return( date );
}


int
fra2min( int f ) {
/* convert frames to minutes of msf */
  f= f < 0 ? -f : f;
  return( f/CD_FRAMES/CD_SECS );
}


int
fra2sec( int f ) {
/* convert frames to seconds of msf */
  f= f < 0 ? -f : f;
  return( (f - fra2min(f)*CD_SECS*CD_FRAMES)/CD_FRAMES );
}


int
fra2fra( int f ) {
/* convert frames to frames of msf */
  f= f < 0 ? -f : f;
  return( f - (fra2min(f)*CD_SECS+fra2sec(f))*CD_FRAMES );
}


char *
basename( const char* filename ) {
  /* from glibc-2.1 */
  char *p = strrchr (filename, '/');
  return p ? p + 1 : (char *) filename;
}


int
cdinfo(drip_config *Config) {
  //char* fpn;//= *argv;
  char* pn = NULL;//= basename( fpn );		/* prg_name is Makefile constant */
  int retval= RETVAL_OK;		/* return value */
  //int c;				/* option char */
  int in_fd;				/* input fd */
  //char stdin_fn[]= "stdin";
  char default_fn[]= DEFAULT_INPUT_FILE;
  char* in_fn= default_fn;		/* input file name */
  //int verbose= 0;			/* verbosity level */
  //int of_bloc= 0;			/* opt-flag: size in blocks */
  //int of_size= 0;			/* opt-flag: size in bytes */
  //int of_Size= 0;			/* opt-flag: size in Mbytes */
  //int of_SIZE= 0;			/* opt-flag: size in msf */
  //int of_sysi= 0;			/* opt-flag: system */
  //int of_voli= 0;			/* opt-flag: volume */
  //int of_vols= 0;			/* opt-flag: volume set */
  //int of_publ= 0;			/* opt-flag: publisher */
  //int of_prep= 0;			/* opt-flag: preparer */
  //int of_appi= 0;			/* opt-flag: application */
  //int of_copy= 0;			/* opt-flag: copyright */
  //int of_abst= 0;			/* opt-flag: abstract */
  //int of_bibl= 0;			/* opt-flag: bibliography */
  int file= 1;				/* flag: .mkisofsrc format */
  ssize_t nbr;
  struct iso_primary_descriptor ipd;

/* open input file */

  in_fd= -1;
  in_fn= Config->DVDdevice->str;
  if( in_fd != STDIN_FILENO ) {
    if( (in_fd=open(in_fn,O_RDONLY)) == -1 ) {
      fprintf( ERROR_CHANNEL, "%s: Cannot open `%s'. %s.\n",pn, in_fn, strerror(errno) );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
  }

/* lseek input file */

  if( in_fd != STDIN_FILENO ) {
    if( lseek(in_fd,ISO_PD_OFFSET,SEEK_SET) == -1 ) {
      fprintf( ERROR_CHANNEL, "%s: Cannot lseek `%s'. %s.\n",pn, in_fn, strerror(errno) );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
  }
  else { /* stdin, read() instead of lseek() */
    unsigned char buf[ISO_PD_OFFSET];
    if( (nbr=my_read(in_fd,buf,ISO_PD_OFFSET)) != ISO_PD_OFFSET ) {
      fprintf( ERROR_CHANNEL, "%s: Unexpected EOF on stdin.\n", pn );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
  }

/* read primary descriptor */

  if( (nbr=my_read(in_fd,&ipd,sizeof(ipd))) != sizeof(ipd) ) {
    fprintf( ERROR_CHANNEL,"%s: Cannot read primary descriptor block of `%s'. %s.\n",pn, in_fn, nbr==-1?strerror(errno):"File too small" );
    retval= RETVAL_ERROR; goto DIE_NOW;
  }

/* list info */

  if( file ) {
    /* list all info in .mkisofsrc format */
    g_string_sprintf(Config->DVDvolumeid,"%s",ISO_STR(ipd.volume_id));
      if (Config->DVDvolumeid->len == 0 ) { g_string_sprintf(Config->DVDvolumeid,"%s","DVD_Volume");}
    g_string_sprintf(Config->DVDpublisher,"%s",ISO_STR(ipd.publisher_id) );
    g_string_sprintf(Config->DVDpreparer,"%s",ISO_STR(ipd.preparer_id) );
    g_string_sprintf(Config->DVDcreationdate,"%s",iso_date(ipd.creation_date) );

    Config->DVDvolumesize_MB = iso_num_733(ipd.volume_space_size)*ISO_BLOCK_SIZE/1024/1024;
  }

/* close input */

  if( close(in_fd) == -1 ) {
    fprintf( ERROR_CHANNEL, "%s: Cannot close `%s'. %s.\n",pn, in_fn, strerror(errno) );
    retval= RETVAL_ERROR; goto DIE_NOW;
  }
DIE_NOW:
  close( in_fd );
  return( retval );
}


