/* Drip - a transcoder for Unix
 * Copyright (C) 2001-2003 Jarl van Katwijk
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

// ----------------------------------------------------------------------------------
//
// echo3d, audio echo filter
//
// 17-08-2002 - Jarl van Katwijk, ported to Drip plugin system
//
//  Crystality Plugin - A plugin for remastering mp3 sound in realtime
//  Copyright (C) 2001 Rafal Bosak <gyver@fanthom.irc.pl>
//
// ----------------------------------------------------------------------------------

#include "../config.h"
#include "../src/drip.h"
#include "../encoder/plugin-loader.hh"
#include <gmodule.h>
// Custom includes:
#include <math.h>

#undef DEBUG


static gboolean init_done = FALSE;                 // Plugin initialised flag
static gint width,height;                          // Input Width\Height sizes
static gint xoff,yoff,xlen,ylen;                   // Offset and lenght values
static gdouble framerate;                           // Framerate (25.00, 29.97, etc)
static gdouble value1; // echo_level
static gdouble value2; // steroe_level
static gdouble value3; // feedback_level
static gdouble value4; // bext_level;
static gdouble value5; // filter_level
static gdouble value6; // harmonics_level
static guint8 *src_planes[3];                      // Input data planes (I420 colour space)
static guint8 *dst_planes[3];                      // Output data planes

// Custom vari's
struct Interpolation{
    int acount;         // counter
    int lval, rval;     // value
    int sal, sar;       // sum
    int al, ar;
    int a1l, a1r;
};
static struct Interpolation bandext_energy;
static struct Interpolation bandext_amplitude;
static const char about_text[] = "Crystality Plugin v0.92";
static GtkWidget *conf_dialog = NULL;
gint stereo_sfactor;
gint echo_sfactor;
gint harmonics_sfactor;
gint feedback_sfactor;
gint filter_sfactor;
gfloat bext_sfactor;
gint lsine[65536];
gint rsine[65536];
GtkObject *bext_adj;
GtkObject *filter_adj;
GtkObject *echo_adj;
GtkObject *stereo_adj;
GtkObject *feedback_adj;
GtkObject *harmonics_adj;
GtkWidget *enable_plugin_button;


#define DELAY2 21000
#define DELAY1 35000
#define DELAY3 14000
#define DELAY4 5

#define BUF_SIZE DELAY1+DELAY2+DELAY3+DELAY4



extern "C" {



/* Return type (SPU,AUDIO,VIDEO) of filter */
G_MODULE_EXPORT module_type_e crystalityfilter_type(void) {
    return AUDIO;
}


/* Return phase (PRE,NORMAL,POST) of filter */
G_MODULE_EXPORT module_phase_e crystalityfilter_phase(void) {
    return NORMAL;
}

/* ------------------------------------------------------------------------------------- */
/* Plugin functions                                                                      */

void bext_level_cb(void) {
    bext_sfactor = (gfloat)(((gfloat)16384 * 10) / (gfloat)(value4 + 1)) + (gfloat)(102 - value4) * 128;
}

void echo_level_cb(void) {
    echo_sfactor = (gint)value1;
}

void stereo_level_cb(void) {
    stereo_sfactor = (gint)value2;
}

void feedback_level_cb(void) {
    feedback_sfactor = (gint)((value3 * 3) / 2);
}

void harmonics_level_cb(void) {
    harmonics_sfactor = (gint)value6;//harmonics_level;
}

void enable_plugin_cb() {
}


// *******************************************************

static void conf_bext_chg_cb(GtkWidget *w, gpointer data) {
    value4 = GTK_ADJUSTMENT(bext_adj)->value; // bext_level
    bext_level_cb();
}

static void conf_echo_chg_cb(GtkWidget *w, gpointer data) {
    value1 = GTK_ADJUSTMENT(echo_adj)->value;
    echo_level_cb();
}

static void conf_stereo_chg_cb(GtkWidget *w, gpointer data) {
    value2 = GTK_ADJUSTMENT(stereo_adj)->value;
    stereo_level_cb();
}

static void conf_feedback_chg_cb(GtkWidget *w, gpointer data) {
    value3 = GTK_ADJUSTMENT(feedback_adj)->value;
    feedback_level_cb();
}

static void conf_harmonics_chg_cb(GtkWidget *w, gpointer data) {
    value6 = GTK_ADJUSTMENT(harmonics_adj)->value; // harmonics_level
    harmonics_level_cb();
}

static void conf_filter_chg_cb(GtkWidget *w, gpointer data) {
    value5 = GTK_ADJUSTMENT(filter_adj)->value * 10; // filter_level
g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Plugin: crystalityfilter value5 = %g",value5);
}

static void conf_enable_plugin_cb(GtkWidget *w, gpointer data) {
    enable_plugin_cb();
}

/* Function which implements a GtkPlug to let user configure this plugin */
G_MODULE_EXPORT GtkWidget* crystalityfilter_gui(void) {
    static GtkWidget *button, *label, *hscale;
    #ifdef DEBUG
    g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Plugin: crystalityfilter gui");
    #endif

    /* Build the gtk user interface for this plugins here. Also implement the callbacks */
    bext_adj = gtk_adjustment_new(value4, 0.0, 100.0, 1, 10, 0);
    echo_adj = gtk_adjustment_new(value1, 0.0, 100.0, 1, 10, 0);
    stereo_adj = gtk_adjustment_new(value2, 0.0, 100.0, 1, 10, 0);
    filter_adj = gtk_adjustment_new(value5/10, 0.1, 0.3, 0.1, 0.1, 0);
    feedback_adj = gtk_adjustment_new(value3, 0.0, 100.0, 1, 10, 0);
    harmonics_adj = gtk_adjustment_new(value6, 0.0, 100.0, 1, 10, 0);


    conf_dialog = gtk_table_new(2, 6, TRUE);
    gtk_container_set_border_width(GTK_CONTAINER(conf_dialog), 1);

    label = gtk_label_new("Bandwidth extender: ");
    gtk_table_attach_defaults(GTK_TABLE(conf_dialog), label, 0, 1, 0, 1);
    gtk_widget_show(label);

    hscale = gtk_hscale_new(GTK_ADJUSTMENT(bext_adj));
    gtk_widget_set_usize(hscale, 310, 35);
    gtk_table_attach_defaults(GTK_TABLE(conf_dialog), hscale, 1, 2, 0, 1);
    gtk_signal_connect(GTK_OBJECT(hscale), "motion_notify_event", GTK_SIGNAL_FUNC(conf_bext_chg_cb), NULL);
    gtk_widget_show(hscale);

    label = gtk_label_new("Extender's\n highpass filter: ");
    gtk_table_attach_defaults(GTK_TABLE(conf_dialog), label, 0, 1, 1, 2);
    gtk_widget_show(label);

    hscale = gtk_hscale_new(GTK_ADJUSTMENT(filter_adj));
    gtk_widget_set_usize(hscale, 310, 35);
    gtk_table_attach_defaults(GTK_TABLE(conf_dialog), hscale, 1, 2, 1, 2);
    gtk_signal_connect(GTK_OBJECT(hscale), "motion_notify_event", GTK_SIGNAL_FUNC(conf_filter_chg_cb), NULL);
    gtk_widget_show(hscale);

    label = gtk_label_new("Harmonic booster: ");
    gtk_table_attach_defaults(GTK_TABLE(conf_dialog), label, 0, 1, 2, 3);
    gtk_widget_show(label);

    hscale = gtk_hscale_new(GTK_ADJUSTMENT(harmonics_adj));
    gtk_widget_set_usize(hscale, 310, 35);
    gtk_table_attach_defaults(GTK_TABLE(conf_dialog), hscale, 1, 2, 2, 3);
    gtk_signal_connect(GTK_OBJECT(hscale), "motion_notify_event", GTK_SIGNAL_FUNC(conf_harmonics_chg_cb), NULL);
    gtk_widget_show(hscale);

    label = gtk_label_new("Echo volume: ");
    gtk_table_attach_defaults(GTK_TABLE(conf_dialog), label, 0, 1, 3, 4);
    gtk_widget_show(label);

    hscale = gtk_hscale_new(GTK_ADJUSTMENT(echo_adj));
    gtk_widget_set_usize(hscale, 310, 35);
    gtk_table_attach_defaults(GTK_TABLE(conf_dialog), hscale, 1, 2, 3, 4);
    gtk_signal_connect(GTK_OBJECT(hscale), "motion_notify_event", GTK_SIGNAL_FUNC(conf_echo_chg_cb), NULL);
    gtk_widget_show(hscale);

    label = gtk_label_new("Echo feedback: ");
    gtk_table_attach_defaults(GTK_TABLE(conf_dialog), label, 0, 1, 4, 5);
    gtk_widget_show(label);

    hscale = gtk_hscale_new(GTK_ADJUSTMENT(feedback_adj));
    gtk_widget_set_usize(hscale, 310, 35);
    gtk_table_attach_defaults(GTK_TABLE(conf_dialog), hscale, 1, 2, 4, 5);
    gtk_signal_connect(GTK_OBJECT(hscale), "motion_notify_event", GTK_SIGNAL_FUNC(conf_feedback_chg_cb), NULL);
    gtk_widget_show(hscale);

    label = gtk_label_new("Extra stereo: ");
    gtk_table_attach_defaults(GTK_TABLE(conf_dialog), label, 0, 1, 5, 6);
    gtk_widget_show(label);

    hscale = gtk_hscale_new(GTK_ADJUSTMENT(stereo_adj));
    gtk_widget_set_usize(hscale, 310, 25);
    gtk_table_attach_defaults(GTK_TABLE(conf_dialog), hscale, 1, 2, 5, 6);
    gtk_signal_connect(GTK_OBJECT(hscale), "motion_notify_event", GTK_SIGNAL_FUNC(conf_stereo_chg_cb), NULL);
    gtk_widget_show(hscale);

    /* Clean & Exit */
    return conf_dialog;
}

/*
 * interpolation routine for ampliude and "energy"
 */
static inline void interpolate(struct Interpolation *s, gint l, gint r){
#define AMPL_COUNT 64
    int a0l, a0r, dal = 0, dar = 0;

    if (l < 0) l = -l;
    if (r < 0) r = -r;

    s->lval += l / 8;
    s->rval += r / 8;

    s->lval = (s->lval * 120) / 128;
    s->rval = (s->rval * 120) / 128;

    s->sal += s->lval;
    s->sar += s->rval;

    s->acount++;
    if (s->acount == AMPL_COUNT){
        s->acount = 0;
        a0l = s->a1l;
        a0r = s->a1r;
        s->a1l = s->sal / AMPL_COUNT;
        s->a1r = s->sar / AMPL_COUNT;
        s->sal = 0;
        s->sar = 0;
        dal = s->a1l - a0l;
        dar = s->a1r - a0r;
        s->al = a0l * AMPL_COUNT;
        s->ar = a0r * AMPL_COUNT;
    }

    s->al += dal;
    s->ar += dar;
}


/*
 * simple pith shifter, plays short fragments at 1.5 speed
 */
void pitchShifter(const gint lin, const gint rin, gint *lout, gint *rout){
#define SH_BUF_SIZE 100 * 3
    static gint16 shBuf[SH_BUF_SIZE];
    static gint shBufPos = SH_BUF_SIZE - 6;
    static gint shBufPos1 = SH_BUF_SIZE - 6;
    static gint cond;

    shBuf[shBufPos++] = lin;
    shBuf[shBufPos++] = rin;

    if (shBufPos == SH_BUF_SIZE) shBufPos = 0;

    switch (cond){
        case 1:
            *lout = (shBuf[shBufPos1 + 0] * 2 + shBuf[shBufPos1 + 2])/4;
            *rout = (shBuf[shBufPos1 + 1] * 2 + shBuf[shBufPos1 + 3])/4;
            break;
        case 0:
            *lout = (shBuf[shBufPos1 + 4] * 2 + shBuf[shBufPos1 + 2])/4;
            *rout = (shBuf[shBufPos1 + 5] * 2 + shBuf[shBufPos1 + 3])/4;
            cond = 2;
            shBufPos1 += 6;
            if (shBufPos1 == SH_BUF_SIZE) {
                shBufPos1 = 0;
            }
            break;
    }
    cond--;

    return;
}

/*
 * calculate scalefactor for mixer
 */
inline gint calc_scalefactor(gint a, gint e){
    gint x;

    if (a > 8192) a = 8192;
    else if (a < 0) a = 0;

    if (e > 8192) e = 8192;
    else if (e < 0) e = 0;

    x = ((e+500) * 4096 )/ (a + 300) + e;

    if (x > 16384) x = 16384;
    else if (x < 0) x = 0;
    return x;
}

/* This function is called for every (audio\video) frame.
   Here the actual filtering is done. */
G_MODULE_EXPORT guint8** crystalityfilter_apply(guint8** src, glong size, gulong SCR, gulong PTS) {
    gint x;
    gint left, right, dif, left0, right0, left1, right1, left2, right2, left3, right3, left4, right4, leftc, rightc, lsf, rsf;
    static gint left0p = 0, right0p = 0;
    static gint rangeErrorsUp = 0;
    static gint rangeErrorsDown = 0;
    static gint l0, l1, l2, r0, r1, r2, ls, rs, ls1, rs1;
    static gint ll0, ll1, ll2, rr0, rr1, rr2;
    static gint lharmb = 0, rharmb = 0, lhfb = 0, rhfb = 0;
    gint lharm0, rharm0;
    static gint16 buf[BUF_SIZE];
    static gint bufPos = BUF_SIZE - 1;
    static gint bufPos1 = 1 + BUF_SIZE - DELAY1;
    static gint bufPos2 = 1 + BUF_SIZE - DELAY1 - DELAY2;
    static gint bufPos3 = 1 + BUF_SIZE - DELAY1 - DELAY2 - DELAY3;
    static gint bufPos4 = 1 + BUF_SIZE - DELAY1 - DELAY2 - DELAY3 - DELAY4;
    static gint16* dataptr = (gint16*)src[0];
    static gpointer burps;

    #ifdef DEBUG
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Plugin: crystalityfilter apply (PTS=%lu,SCR=%lu)",PTS,SCR);
    #endif
if (burps==NULL) {
    burps=(gpointer)1;
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Plugin: crystalityfilter apply (value1=%g,value2=%g,value3=%g,value4=%g,value5=%g,value6=%g)",value1,value2,value3,value4,value5,value6);
}

    /* Parse values */
    bext_level_cb();
    echo_level_cb();
    stereo_level_cb();
    feedback_level_cb();
    harmonics_level_cb();

    for (x = 0; x < size/2; x += 2) {

        // ************ load sample **********
        left0 = dataptr[x];
        right0 = dataptr[x+1];

        // ************ slightly expand stereo for direct input **********
        ll0=left0;rr0=right0;
        dif = (ll0+ll1+ll2 - rr0-rr1-rr2) * stereo_sfactor / 256;
        left0 += dif;
        right0 -= dif;
        ll2= ll1; ll1= ll0;
        rr2= rr1; rr1= rr0;

        // ************ echo from buffer - first echo **********
        // ************  **********
        left1 = buf[bufPos1++];
        if (bufPos1 == BUF_SIZE)
          bufPos1 = 0;
        right1 = buf[bufPos1++];
        if (bufPos1 == BUF_SIZE)
          bufPos1 = 0;

        // ************ highly expand stereo for first echo **********
        dif = (left1 - right1);
        left1 = left1 + dif;
        right1 = right1 - dif;

        // ************ second echo  **********
        left2 = buf[bufPos2++];
        if (bufPos2 == BUF_SIZE)
          bufPos2 = 0;
        right2 = buf[bufPos2++];
        if (bufPos2 == BUF_SIZE)
          bufPos2 = 0;

    // ************ expand stereo for second echo **********
    dif = (left2 - right2);
    left2 = left2 - dif;
    right2 = right2 - dif;

    // ************ third echo  **********
    left3 = buf[bufPos3++];
    if (bufPos3 == BUF_SIZE)
      bufPos3 = 0;
    right3 = buf[bufPos3++];
    if (bufPos3 == BUF_SIZE)
      bufPos3 = 0;

    // ************ fourth echo  **********
    left4 = buf[bufPos4++];
    if (bufPos4 == BUF_SIZE)
      bufPos4 = 0;
    right4 = buf[bufPos4++];
    if (bufPos4 == BUF_SIZE)
      bufPos4 = 0;

    left3 = (left4+left3) / 2;
    right3 = (right4+right3) / 2;

    // ************ expand stereo for second echo **********
    dif = (left4 - right4);
    left3 = left3 - dif;
    right3 = right3 - dif;

    // ************ a weighted sum taken from reverb buffer **********
    leftc = left1 / 9 + right2 /8  + left3 / 8;
    rightc = right1 / 11 + left2 / 9 + right3 / 10;

    // ************ mix reverb with (near to) direct input **********
    left = left0p + leftc * echo_sfactor / 16;
    right = right0p + rightc * echo_sfactor / 16;

    l0 = leftc + left0 / 2;
    r0 = rightc + right0 / 2;

    ls = l0 + l1 + l2;  // do not reverb high frequencies (filter)
    rs = r0 + r1 + r2;  //

    buf[bufPos++] = ls * feedback_sfactor / 256;
    if (bufPos == BUF_SIZE)
      bufPos = 0;
    buf[bufPos++] = rs * feedback_sfactor / 256;
    if (bufPos == BUF_SIZE)
      bufPos = 0;

    // ************ add some extra even harmonics **********
    // ************ or rather specific nonlinearity
    lhfb = lhfb + (ls * 32768 - lhfb) / 32;
    rhfb = rhfb + (rs * 32768 - rhfb) / 32;

    lsf = ls - lhfb / 32768;
    rsf = rs - rhfb / 32768;

    lharm0 = 0
        + ((lsf + 10000) * ((((lsine[((lsf/4) + 32768 + 65536) % 65536] * harmonics_sfactor)) / 64))) / 32768
        - ((lsine[((lsf/4) + 32768 +65536) % 65536]) * harmonics_sfactor) / 128
        ;
    rharm0 =
        + ((rsf + 10000) * ((((lsine[((rsf/4) + 32768 + 65536) % 65536] * harmonics_sfactor)) / 64))) / 32768
        - ((rsine[((rsf/4) + 32768 +65536) % 65536]) * harmonics_sfactor) / 128
        ;

    lharmb = lharmb + (lharm0 * 32768 - lharmb) / 16384;
    rharmb = rharmb + (rharm0 * 32768 - rharmb) / 16384;

    // ************ for convolution filters **********
    l2= l1; r2= r1;
    l1 = l0; r1 = r0;
    ls1 = ls; rs1 = rs;

    left  = 0
        + lharm0 - lharmb / 32768
        + left
        ;
    right = 0
        + rharm0 - rharmb / 32768
        + right
        ;

    left0p = left0;
    right0p = right0;


    // ************ limiter **********
    if (left < -32768) {
        left = -32768;  // limit
        rangeErrorsDown++;
    }
    else if (left > 32767) {
        left = 32767;
        rangeErrorsUp++;
    }
    if (right < -32768) {
        right = -32768;
        rangeErrorsDown++;
    }
    else if (right > 32767) {
        right = 32767;
        rangeErrorsUp++;
    }
    // ************ store sample **********
    dataptr[x  ] = left;
    dataptr[x+1] = right;
   }

   /* -------------------------------------------------------------------------------- */

    static gint saw;             // test stuff
    static gint lprev0, rprev0, lprev1, rprev1, lprev2, rprev2;
    //gint left0, right0, left1, right1, left2, right2, left3, right3;
    static gint lamplUp, lamplDown;
    static gint ramplUp, ramplDown;
    gint lampl, rampl;
    gint tmp;


    for (x = 0; x < size/2; x += 2) {
        // ************ load sample **********
        left0 = dataptr[x];
        right0 = dataptr[x+1];

        // ************ highpass filter part 1 **********
        left1  = (left0  - lprev0) * 56880 / 65536;
        right1 = (right0 - rprev0) * 56880 / 65536;

        left2  = (left1  - lprev1) * 56880 / 65536;
        right2 = (right1 - rprev1) * 56880 / 65536;

        left3  = (left2  - lprev2) * 56880 / 65536;
        right3 = (right2 - rprev2) * 56880 / 65536;

        switch ((gint)value5){
            case 1:
                pitchShifter(left1, right1, &left, &right);
                break;
            case 2:
                pitchShifter(left2, right2, &left, &right);
                break;
            case 3:
                pitchShifter(left3, right3, &left, &right);
                break;
        }

        // ************ amplitude detector ************
        tmp = left1 + lprev1;
        if      (tmp * 16 > lamplUp  ) lamplUp   += (tmp - lamplUp  );
        else if (tmp * 16 < lamplDown) lamplDown += (tmp - lamplDown);
        lamplUp   = (lamplUp   * 1000) /1024;
        lamplDown = (lamplDown * 1000) /1024;
        lampl = lamplUp - lamplDown;

        tmp = right1 + rprev1;
        if      (tmp * 16 > ramplUp  ) ramplUp   += (tmp - ramplUp  );
        else if (tmp * 16 < ramplDown) ramplDown += (tmp - ramplDown);
        ramplUp   = (ramplUp   * 1000) /1024;
        ramplDown = (ramplDown * 1000) /1024;
        rampl = ramplUp - ramplDown;
        interpolate(&bandext_amplitude, lampl, rampl);

        // ************ "sound energy" detector (approx. spectrum complexity) ***********
        interpolate(&bandext_energy, left0  - lprev0, right0 - rprev0);

        // ************ mixer ***********
        left   = (gint)(left0 + left  * calc_scalefactor(bandext_amplitude.lval, bandext_energy.lval) / bext_sfactor);
        right  = (gint)(right0 + right * calc_scalefactor(bandext_amplitude.rval, bandext_energy.rval) / bext_sfactor); //16384
        saw = (saw + 2048) & 0x7fff;

        // ************ highpass filter part 2 **********
        // ************ save previous values for filter
        lprev0 = left0;
        rprev0 = right0;
        lprev1 = left1;
        rprev1 = right1;
        lprev2 = left2;
        rprev2 = right2;
        // ************ END highpass filter part 2 **********


        // ************ limiter **********
        if (left < -32768) left = -32768;
        else if (left > 32767) left = 32767;
        if (right < -32768) right = -32768;
        else if (right > 32767) right = 32767;

        // ************ store sample **********
        dataptr[x  ] = left;
        dataptr[x+1] = right;
    }

    /* Clean & Exit */
    return src;             // returning input data, here dst=src, but dst!=src is OK too
}


G_MODULE_EXPORT void crystalityfilter_init(gint w, gint h, 
                                     gint xo, gint yo, gint xl, gint yl,
                                     gdouble fr) {
    static gint i,x;
    gdouble lsum;
    gdouble rsum;
#define COND 0


    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Plugin: crystalityfilter init");
    /* Copy values */
    width = w; height = h;
    xoff = xo; yoff = yo; xlen = xl; ylen = yl;
    framerate = fr;
    /* Filter specific initialisation */
    for (i = 0; i < 32768; ++i) {
        lsum = rsum = 0;
        if (COND || i < 32768 )lsum+=  ((cos((double)i * 3.141592535 / 32768/2  )) + 0) / 2;
        if (COND || i < 16384 )rsum-=  ((cos((double)i * 3.141592535 / 16384/2  )) + 0) / 4;
        rsum = lsum;

        if (COND || i < 8192 ) lsum += ((cos((double)i * 3.141592535 / 8192/2    )) + 0) /  8;
        if (COND || i < 5641 ) rsum += ((cos((double)i * 3.141592535 / 5641.333333/2)) + 0) /  8;

 //g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Plugin: crystalityfilter i=%i",i);
        lsine[32768 + i] = (gint)((gdouble)(lsum - 0.5) * 32768 * 1.45);
        lsine[32768 - i] = lsine[32768 + i];
        rsine[32768 + i] = (gint)((gdouble)(rsum - 0.5) * 32768 * 1.45);
        rsine[32768 - i] = rsine[32768 + i];

        x = i;
    }
    bext_sfactor = value4;
    echo_sfactor = (gint)value1;
    stereo_sfactor = (gint)value2;
    filter_sfactor = (gint)value5;
    feedback_sfactor = (gint)value3;
    harmonics_sfactor = (gint)value6;

    /* Set init flag */
    init_done = TRUE;

    /* Clean & Exit */
    return;
}


/* Called when parameters need tobe reset to default, auto called when plugin used for 1st time */
G_MODULE_EXPORT void crystalityfilter_default(void) {
    g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Plugin: crystalityfilter default");
    /* Set value's to default */
    value1 = 11;
    value2 = 11;
    value3 = 20;
    value4 = 20;
    value5 = 20;
    value6 = 30;

    /* Clean & Exit */
    return;
}


/* Called after encoding has ended. Variable can be reset etc. */
G_MODULE_EXPORT void crystalityfilter_destruct(void) {
    g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Plugin: crystalityfilter destruct");
    /* Clean & Exit */
    return;
}






/* ------------------------------------------------------------------------ */
/* FIXED FUNCTIONS AFTER THIS POINT, DONT EDIT, ONLY CHANGE FUNCTION NAMES! */
/* ------------------------------------------------------------------------ */


/* Return values of plugin (value1,value2,... value5) */
G_MODULE_EXPORT void crystalityfilter_values(gdouble* values) {
    values[0] = value1;
    values[1] = value2;
    values[2] = value3;
    values[3] = value4;
    values[4] = value5;
    values[5] = value6;
    return;
}

/* Set plugins values */
G_MODULE_EXPORT void crystalityfilter_values_function(gdouble *v) {
    value1 = v[0];
    value2 = v[1];
    value3 = v[2];
    value4 = v[3];
    value5 = v[4];
    value6 = v[5];
    return;
}

/* g_module_check_init is automatically executed upon loading */
G_MODULE_EXPORT const gchar* g_module_check_init(GModule *module) {
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Plugin: crystalityfilter selftest passed");
    return NULL;
}

} //extern "C"


