/* Drip - a transcoder for Unix
 * Copyright (C) 2001-2003 Jarl van Katwijk
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*

    Cleaner filter plugin for Drip - 14-0-2002, Jarl van Katwijk
    ported from M$ C++ RGB VirtualDub filter by Jim Casaburi


    --------------------------------------------------------------------

    2d cleaner filter for VirtualDub -- blends a pixel with pixels 
	surrounding it as long as those pixels are simular to the source
	pixel

    Copyright (C) 2000 Jim Casaburi
		Based on code by Avery Lee
		Useful suggestions and much help from Donald A. Graft

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

	The author can be contacted at:
	Jim Casaburi
	casaburi@earthlink.net

	New versions of the source code and the compiled filter can be 
	found at http://home.earthlink.net/~casaburi/download/
*/

#include "../config.h"
#include "../src/drip.h"
#include "../encoder/plugin-loader.hh"
#include <gmodule.h>

// Custom includes:
#include "../encoder/fast_memcpy.hh"

#undef DEBUG


static gboolean init_done = FALSE;                 // Plugin initialised flag
static gint width,height;                          // Input Width\Height sizes
static gint xoff,yoff,xlen,ylen;                   // Offset and lenght values
static gdouble framerate;                           // Framerate (25.00, 29.97, etc)
static gdouble value1,value2,value3,value4,value5;  // Plugin defined values - these are loaded\saved by Drip
//static guint8 *src_planes[3];                      // Input data planes (I420 colour space)
//static guint8 *dst_planes[3];                      // Output data planes
// Custom vari's
static guint8** _dst;

extern "C" {


//extern HINSTANCE g_hInst;

typedef struct MyFilterData {
	bool fDebugNoise;
	bool fInterlace;
	int fThreshold;
	int fRadius;
} MyFilterData;

//unsigned long ppixels(unsigned long *src, int lowx, int lowy, int highx, int highy);
//static int twodclean_run(const FilterActivation *fa, const FilterFunctions *ff);
//BOOL CALLBACK twodclean_ConfigDlgProc(HWND hdlg, UINT msg, WPARAM wParam, LPARAM lParam);
//int twodclean_ConfigProc(FilterActivation *fa, const FilterFunctions *ff, HWND hwnd);
//void twodclean_StringProc(const FilterActivation *fa, const FilterFunctions *ff, char *str);
//void twodcleanScriptConfig(IScriptInterpreter *isi, void *lpVoid, CScriptValue *argv, int argc);

#define READSRC(byteoff, lineoff) (unsigned long *)((char *)src + pitch*(lineoff) + (byteoff*4))

//ScriptFunctionDef twodclean_func_defs[]={
//    { (ScriptFunctionPtr)twodcleanScriptConfig, "Config", "0iii" },
//    { NULL },
//};

//CScriptObject twodclean_obj={
//    NULL, twodclean_func_defs
//};

static int fThreshold;
static bool fDebugNoise;
static int pitch = 1;


/* Return type (SPU,AUDIO,VIDEO) of filter */
G_MODULE_EXPORT module_type_e cleanfilter_type(void) {
    return VIDEO;
}


/* Return phase (PRE,NORMAL,POST) of filter */
G_MODULE_EXPORT module_phase_e cleanfilter_phase(void) {
    return NORMAL;
}


/* Function which implements a GtkPlug to let user configure this plugin */
G_MODULE_EXPORT GtkWidget* cleanfilter_gui(void) {
    static GtkWidget *gui = NULL;
    #ifdef DEBUG
    g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Plugin: testfilter gui");
    #endif


    /* Build the gtk user interface for this plugins here. Also implement the callbacks */
    gui = gtk_label_new("\ncleanfilter\nUNDER DEVELLOPMENT!\n");

    /* Clean & Exit */
    return gui;
}

guint8 max255(gint value) {
    if (value>255) return 255;
    if (value<0)   return 0;
    return value;
}
      

/* This function is called for every (audio\video) frame. 
   Here the actual filtering is done. */
G_MODULE_EXPORT guint8** cleanfilter_apply(guint8** _src, glong size, gulong SCR, gulong PTS) {
	register gulong w,h;
	gint x, y;
	gint lowx, lowy;
	gint highx, highy;
	static guint8* src = _src[0];
	guchar red, blue, green;
	guchar fred, fblue, fgreen;
        guchar _red, _blue, _green;
        guchar Y, Cr, Cb;
	gint reddif, bluedif, greendif;
	gint a, b;
	gint redcount;
	gint bluecount;
	gint greencount;
	gint totred = 0;
	gint totblue = 0;
	gint totgreen = 0;
	gint pix = 1;
        gint radius = 2; // TODO: make configurable
        gulong _pixel;
	gulong *pixel;
        glong offset = 0;
        glong offset2 = 0;
        guint8** dst;


    #ifdef DEBUG
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Plugin: cleanfilter apply (PTS=%lu,SCR=%lu)",PTS,SCR);
    #endif
        // Alloc dst planes
        dst = (guint8**)malloc(3*sizeof(guint8*));
        dst[0] = (guint8*)malloc(width*height);
        dst[1] = (guint8*)malloc(width*height/4);
        dst[2] = (guint8*)malloc(width*height/4);
 
	src -= pitch>>2;
	h = height;
	y = 0;
	do { 
		w = width;
		x = 0;
		do {
			// set the boundaries of the averaging region
			lowx = -radius;
			lowy = 1 - radius;
			highx = radius;
			highy = 1 + radius;
			// make sure we aren't exceeding frame boundaries
			if (x + lowx < 0) lowx = 0;
			if (y + lowy < 1) lowy = 1;
			if (x + highx > w-1) highx = (w - 1)-x;
			if (y + highy > h) highy = h - y;

			totred = 0;
			totgreen = 0;
			totblue = 0;
			pix = 1;
			// read in the source pixel
                        Y  = *(guchar*)((gulong)_src[0] + offset   + pitch);
                        Cr = *(guchar*)((gulong)_src[1] + offset/4 + pitch);
                        Cb = *(guchar*)((gulong)_src[2] + offset/4 + pitch);

                        fred   = redcount   = (guchar)(1.164 * (Y-16) + 1.596 * (Cr-128));
                        fgreen = greencount = (guchar)(1.164 * (Y-16) - 0.813 * (Cr-128) - 0.391 * (Cb-128));
                        fblue  = bluecount  = (guchar)(1.164 * (Y-16) + 2.018 * (Cb-128));
                        _pixel = (gulong)fred>>16 + (gulong)fgreen>>8 + (gulong)fblue;
                        pixel  = &_pixel;
			// get the other pixels
			for (b = lowy; b < highy; b += 1) {
                                offset2 = offset  + pitch * b + lowx * 4;
//g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Plugin: cleanfilter apply _src[0]=%p, offset = %lu, offset2 = %lu, b=%i, lowx=%i",_src[0],offset,offset2,b,lowx);
                                Y  = *(guchar*)((gulong)_src[0] + offset2);
                                Cr = *(guchar*)((gulong)_src[1] + offset2/4);
                                Cb = *(guchar*)((gulong)_src[2] + offset2/4);
//g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Plugin: cleanfilter apply Y=%i, Cr=%i, Cb=%i",Y,Cr,Cb);
                                fred   = redcount   = (guchar)(1.164 * (Y-16) + 1.596 * (Cr-128));
                                fgreen = greencount = (guchar)(1.164 * (Y-16) - 0.813 * (Cr-128) - 0.391 * (Cb-128));
                                fblue  = bluecount  = (guchar)(1.164 * (Y-16) + 2.018 * (Cb-128));
                                _pixel = (gulong)fred>>16 + (gulong)fgreen>>8 + (gulong)fblue;
                                pixel  = &_pixel;
				for (a = lowx; a < highx; a++) {
					if (!((a == 0) && (b == 1))) { // as long as it isn't the source pixel
						// extract the r,g,b values
                                                red   = fred;
                                                green = fgreen;
                                                blue  = fblue;
						// if the abs(red difference) is greater than the threshold
						// then throw out the value otherwise add it to the running 
						// total
						reddif = red - fred;
						if ((reddif > fThreshold) || (reddif < -fThreshold)) 
							totred--;
						else 
							redcount += red;
						// if the abs(blue difference) is greater than the threshold
						// then throw out the value otherwise add it to the running 
						// total
						bluedif = blue - fblue;
						if ((bluedif > fThreshold) || (bluedif < -fThreshold)) 
							totblue--;
							else 
							bluecount += blue;
						// if the abs(green difference) is greater than the threshold
						// then throw out the value otherwise add it to the running 
						// total
						greendif = green - fgreen;
						if ((greendif > fThreshold) || (greendif < -fThreshold)) 
							totgreen--;
							else 
							greencount += green;
						pix++;
						// move to the pixel to the right
                                                offset2++;
                                                Y  = *(guchar*)((gulong)_src[0] + offset2);
                                                Cr = *(guchar*)((gulong)_src[1] + offset2/4);
                                                Cb = *(guchar*)((gulong)_src[2] + offset2/4);

                                                fred   = redcount   = (guchar)(1.164 * (Y-16) + 1.596 * (Cr-128));
                                                fgreen = greencount = (guchar)(1.164 * (Y-16) - 0.813 * (Cr-128) - 0.391 * (Cb-128));
                                                fblue  = bluecount  = (guchar)(1.164 * (Y-16) + 2.018 * (Cb-128));
                                                _pixel = (gulong)fred>>16 + (gulong)fgreen>>8 + (gulong)fblue;
                                                pixel  = &_pixel;
					}
				}
			}
			// now tot* is the number of values that will be averaged
			totred += pix; 
			totblue += pix; 
			totgreen += pix;
//			// if we are to show edges and less than 1/2 of the rgb values are to be
//			// averaged together, show a black pixel instead
//			if (fDebugNoise && (totblue + totgreen + totred < ((pix * 3)/ 2))) {
//				*dst = 0;
//			}
//			else 
			// otherwise average the rgb values and construct a pixel around them
                        _red   = fred;//redcount/totred;
                        _green = fgreen;//greencount/totgreen;
                        _blue  = fblue;//bluecount/totblue;
                        Y =(guchar)( (0.257*_red)+(0.504*_green)+(0.098*_blue)+16);
                        Cr=(guchar)( (0.439*_red)-(0.368*_green)-(0.071*_blue)+128);
                        Cb=(guchar)(-(0.148*_red)-(0.291*_green)+(0.439*_blue)+128);
                        dst[0][        offset   ] = max255(Y);
                        dst[1][(guint)(offset/4)] = max255(Cr);
                        dst[2][(guint)(offset/4)] = max255(Cb);
			
			++offset;
			x++;
		} while (--w);
		y++;
	} while (--h);


    /* Clean & Exit */
    fast_memcpy(_src[0],dst[0],width*height);
    fast_memcpy(_src[1],dst[1],width*height/4);
    fast_memcpy(_src[2],dst[2],width*height/4);
    free(dst[2]);
    free(dst[1]);
    free(dst[0]);
    free(dst);
    return _src;
}


G_MODULE_EXPORT void cleanfilter_init(gint w, gint h,
                                     gint xo, gint yo, gint xl, gint yl,
                                     gdouble fr) {
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Plugin: cleanfilter init");
    /* Copy values */
    width = w; height = h;
    xoff = xo; yoff = yo; xlen = xl; ylen = yl;
    framerate = fr;
    /* Filter specific initialisation */
    _dst = (guint8**)malloc(3*sizeof(guint8*));
    /* Set init flag */
    init_done = TRUE;
    /* Clean & Exit */
    return;
}


/* Called when parameters need tobe reset to default, auto called when plugin used for 1st time */
G_MODULE_EXPORT void cleanfilter_default(void) {                                             /* DEFINE! */
    g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Plugin: cleanfilter default");
    /* Set value's to default */
    value1 = 0;
    value2 = 0;
    value3 = 0;
    value4 = 0;
    value5 = 0;
    /* Clean & Exit */
    return;
}


/* Called after encoding has ended. Variable can be reset etc. */
G_MODULE_EXPORT void cleanfilter_destruct(void) {
    g_log(DRIP_CB_LD,G_LOG_LEVEL_DEBUG,"Plugin: cleanfilter destruct");
    /* Clean & Exit */
    return;
}

/* ------------------------------------------------------------------------ */
/* FIXED FUNCTIONS AFTER THIS POINT, DONT EDIT, ONLY CHANGE FUNCTION NAMES! */
/* ------------------------------------------------------------------------ */


/* Return values of plugin (value1,value2,... value5) */
G_MODULE_EXPORT void cleanfilter_values(gdouble* values) {
    values[0] = value1;
    values[1] = value2;
    values[2] = value3;
    values[3] = value4;
    values[4] = value5;
    return;
}

/* Set plugins values */
G_MODULE_EXPORT void cleanfilter_values_function(gdouble *v) {
    value1 = v[0];
    value2 = v[1];
    value3 = v[2];
    value4 = v[3];
    value5 = v[4];
    return;
}

/* g_module_check_init is automatically executed upon loading */
G_MODULE_EXPORT const gchar* g_module_check_init(GModule *module) {
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Plugin: cleanfilter selftest passed");
    return NULL;
}

} //extern "C"

