/* Drip - a transcoder for Unix
 * Copyright (C) 2001-2003 Jarl van Katwijk
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "demuxer.hh"
#include "autoclipper.hh"
#include "encoder.hh"
#include "main.hh"
#include <time.h>

static gint clip_frames = 0;
static gint clip_frames_total = 0;
static gint autoclip_total_frames;
static gint autoclip_pass_count;
static gint autoclip_samples = 0;
static gint frames_per_sample;


/* Return TRUE if line is black, else FALSE */
/* NOTE: Only luminance of used to check for darkness */
inline gboolean vertical_black(unsigned char* framebuffer,gint line,gint width,gint height) {
    register glong pixel;
    register guint8* base;
    register guint8 Y;

    line += (gint)framebuffer; // point to offset in Y color plane

    for (pixel=0;pixel<height;pixel++) {
        base = (guint8*)(line+pixel*width);
        Y = *base;
        if (Y<65) {
            /* Dark pixel, if whole row < 30 consider it Black */
        } else {
            return FALSE;
        }
    }
    return TRUE;
}



/* Return TRUE if line is black, else FALSE */
inline gboolean horizontal_black(unsigned char* framebuffer,gint line,gint width) {
    register glong pixel;
    register guint* base = (guint*)(line*width+framebuffer); // point to offset in Y color plane
    register guint8 Y;

    for (pixel=0;pixel<width;pixel++) {
        Y = *(base+pixel);
        if (Y<65) {
            /* Dark pixel, if whole line < 30 consider it Black */
        } else {
            return FALSE;
        }
    }
    return TRUE;
}


gboolean autoclipper_init(void) {
    return TRUE;
}



/* Determine amount of clipping for top, bottom, left and right. */
void autoclipper_process(guint8 *frame[3],uint32_t pts,gint PTSaudio) {
    //gint file, result;
    //gint vstrm,vistream = 0;
    gint i;
    //gulong framenr;
    gint cliptop30 = 10000,clipbottom30 = 10000,clipleft30 = 10000,clipright30 = 10000;
    //gint cliptop70 = 10000,clipbottom70 = 10000,clipleft70 = 10000,clipright70 = 10000;
    gboolean top,bottom,right,left;
    gdouble perc;

    /* Read bunch of frames to reach a full one, hope we passed a keyframe;) */
    clip_frames++;
    clip_frames_total++;
    if (clip_frames<frames_per_sample) {
        return;
    } else {
        clip_frames = 0; // reset clip frames
    }
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Next autoclip frame ready");

    /* Allocate buffers */
    guchar* Ybuffer = frame[0];
    //guchar* CRbuffer = frame[1];
    //guchar* CBbuffer = frame[2];

    /* DETERMINE TOP\BOTTOM\LEFT\RIGHT CLIPPING FOR A RANDOM FRAME */
    do {
        perc = random();
        perc = perc/(gdouble)RAND_MAX;
        perc = perc * (gdouble)(autoclip_total_frames-40)/(gdouble)autoclip_total_frames;
    } while (perc<0.01 || perc>0.99); // ???
    g_log(DRIP_LD,G_LOG_LEVEL_INFO,"Autoclipper sample %i of %i taken at frame %i",autoclip_samples+1,autoclip_pass_count,clip_frames_total);

    //TODO mpeg seeking
    //framenr = drip_mpeg_seek_percentage(perc); // seek for next frame

    /* Top\Bottom clipping */
    top = TRUE;bottom = TRUE;
    for (i=0;i<(drip_mpeg_height()/2);i++) {
        /* Check lines */
        if (bottom==TRUE)    { bottom = horizontal_black(Ybuffer,drip_mpeg_height()-10-i,drip_mpeg_width());}
        if (top==TRUE)       { top =    horizontal_black(Ybuffer,i,drip_mpeg_width());}
        /* Non black line found -> set clipping */
        if (bottom==FALSE)   { clipbottom30 = i+1;bottom=2;}
        if (top==FALSE)      { cliptop30 = i;top=2;}
        /* Ready? */
        if ((top==FALSE) && (bottom==FALSE)) {
            /* Found both lines that are not 100% black */
            i = drip_mpeg_height();
        }
    }

    /* Left\Right clipping */
    left = TRUE;right = TRUE;
    for (i=0;i<(drip_mpeg_width()/2);i++) {
        /* Check lines */
        if (left==TRUE)   { left = vertical_black(Ybuffer,i,drip_mpeg_width(),drip_mpeg_height());}
        if (right==TRUE)  { right = vertical_black(Ybuffer,drip_mpeg_width()-1-i,drip_mpeg_width(),drip_mpeg_height());}
        /* Non black line found -> set clipping */
        if (left==FALSE)  { clipleft30 = i+1;left=2;}
        if (right==FALSE) { clipright30 = i;right=2;}
        /* Ready? */
        if ((left==FALSE) && (right==FALSE)) {
            /* Found both lines that are not 100% black */
            i = drip_mpeg_width();
        }
    }

    clipleft30 += 1; //adjust to real clip.. TODO: better logic for this ;) 
    cliptop30  += 1; //adjust to real clip.. TODO: better logic for this ;)

    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Auto clipping sample top %i,bottom %i,left %i,right %i lines",cliptop30,clipbottom30,clipleft30,clipright30);

    autoclip_samples++;
    if (autoclip_samples==(autoclip_pass_count)) {
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Autoclipper closing input");
        quit = TRUE; //drip_mpeg_close(TRUE);
    }

    /* Store results (in factors of 8?16) */
    if (cliptop    > cliptop30)    { 
                                       cliptop    = cliptop30;
    }
    if (clipbottom > clipbottom30) { 
                                       clipbottom = clipbottom30;
    }
    if (clipleft   > clipleft30)   { 
                                       clipleft   = clipleft30;
    }
    if (clipright  > clipright30)  { 
                                       clipright  = clipright30;
    }
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Auto clipping is now at top %i,bottom %i,left %i,right %i lines",cliptop,clipbottom,clipleft,clipright);

    /* Clean & Exit */
    return;
}


/* this function corrects the clipping, so that a multiple of 16 is clipped */
void correct_clipping(gint* clipvalue1, gint* clipvalue2) {
    gint  cliprange  = *clipvalue1 + *clipvalue2;
    gint  cliprangecorrected = cliprange - (cliprange % 16);

    if (cliprange != cliprangecorrected) {
        *clipvalue1 = (gint) (cliprangecorrected * ((double) *clipvalue1 / cliprange));
        *clipvalue2 = cliprangecorrected - *clipvalue1;
    }
    return;
}

/* Check amount of clipping for top, bottom, left and right. There're 2 samples
   taken, to reduce to possibility a whole black frame is selected */
gboolean autoclipper_apply(gchar* filename, gulong frames,gint count,gint total, gint _frames_per_sample) {
    gint file; //, result;
    //gint vstrm,vistream = 0;
    //guint i;
    //gulong framenr;
    //gint cliptop30 = 10000,clipbottom30 = 10000,clipleft30 = 10000,clipright30 = 10000;
    //gint cliptop70 = 10000,clipbottom70 = 10000,clipleft70 = 10000,clipright70 = 10000;
    //gboolean top,bottom,right,left;
    //guint width,height;
    //gdouble perc;
    //GString *text = g_string_new("");
    //guchar* frame;

    /* Globalise totalframes and autoclipping pass count */
    autoclip_total_frames = frames;
    autoclip_pass_count   = total;
    clip_frames           = 0;
    autoclip_samples      = 0;
    clip_frames_total     = 0;
    frames_per_sample = frames / 6; 
    if (frames_per_sample>_frames_per_sample) frames_per_sample = _frames_per_sample;
    /* init random generator */
    srand(time(0));

    /* Open input MPEG2 stream */
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Autoclipper about to open input %s",filename);
    file = drip_mpeg_open(filename,autoclipper_process);

    if (!file) {
        /* Error, not able to open MPEG2 stream properly */
        g_log(DRIP_LD,G_LOG_LEVEL_WARNING,"Unable to open %s, possibly not an MPEG2 or IFO file?",filename);
        return FALSE;
    } else {
        g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Autoclipper input %s opened",filename);
    }


    /* Start demuxing process */
    quit = FALSE;
    gint **in_streams = (gint**)malloc(2*sizeof(gint*));
    in_streams[0] = (gint*)malloc(sizeof(gint));	*in_streams[0] = -1;
    in_streams[1] = (gint*)malloc(sizeof(gint));	*in_streams[1] = -1;
    drip_mpeg_start_demuxing(in_streams);

    /* Cast clip vlues to multiples of 16 */
    correct_clipping(&clipleft, &clipright);
    correct_clipping(&cliptop, &clipbottom);
    g_log(DRIP_LD,G_LOG_LEVEL_DEBUG,"Auto clipping corrected to: top %i,bottom %i,left %i,right %i lines",cliptop,clipbottom,clipleft,clipright);
 
    /* Clean & Exit */
    drip_mpeg_close(FALSE);
    return TRUE;
}

