require 'socket'
require 'openssl'
require 'drb/drb'
require 'singleton'

module DRb

  class DRbSSLSocket < DRbTCPSocket

    class SSLConfig

      DEFAULT = {
	:SSLCertificate       => nil,
	:SSLPrivateKey        => nil,
	:SSLCACertificate     => nil,
	:SSLCACertificatePath => nil,
	:SSLVerifyMode        => ::OpenSSL::SSL::VERIFY_NONE, 
	:SSLVerifyDepth       => nil,
	:SSLVerifyCallback    => nil,   # custom verification
	# Must specify if you use auto generated certificate.
	:SSLCertName          => nil,
	:SSLCertComment       => "Generated by Ruby/OpenSSL"
      }

      def initialize(config)
	@config = config
      end

      def [](key); 
	@config[key] || DEFAULT[key]
      end
      
      def connect(tcp)
	ssl = ::OpenSSL::SSL::SSLSocket.new(tcp)
	ssl.ca_cert         = self[:SSLCACertificate]
	ssl.ca_path         = self[:SSLCACertificatePath]
	ssl.verify_mode     = self[:SSLVerifyMode]
	ssl.verify_depth    = self[:SSLVerifyDepth]
	ssl.verify_callback = self[:SSLVerifyCallback]
	ssl.sync = true
	ssl.connect
	ssl
      end
      
      def accept(tcp)
	ssl = OpenSSL::SSL::SSLSocket.new(tcp, @cert, @rsa)
	ssl.ca_cert         = self[:SSLCACertificate]
	ssl.ca_path         = self[:SSLCACertificatePath]
	ssl.verify_mode     = self[:SSLVerifyMode]
	ssl.verify_depth    = self[:SSLVerifyDepth]
	ssl.verify_callback = self[:SSLVerifyCallback]
	ssl.sync = true
	ssl.accept
	ssl
      end
      
      def setup_keys
	if self[:SSLCertificate]
	  @cert = self[:SSLCertificate]
	  @rsa  = self[:SSLPrivateKey]
	  return
	end

	rsa = OpenSSL::PKey::RSA.new(512){|p, n|
	  next unless self[:verbose]
	  case p
	  when 0; $stderr.putc "."  # BN_generate_prime
	  when 1; $stderr.putc "+"  # BN_generate_prime
	  when 2; $stderr.putc "*"  # searching good prime,
	                            # n = #of try,
                          	    # but also data from BN_generate_prime
	  when 3; $stderr.putc "\n" # found good prime, n==0 - p, n==1 - q,
                         	    # but also data from BN_generate_prime
	  else;   $stderr.putc "*"  # BN_generate_prime
	  end
	}

	cert = OpenSSL::X509::Certificate.new
	cert.version = 3
	cert.serial = 0
	name = OpenSSL::X509::Name.new(self[:SSLCertName])
	cert.subject = name
	cert.issuer = name
	cert.not_before = Time.now
	cert.not_after = Time.now + (365*24*60*60)
	cert.public_key = rsa.public_key
	
	ef = OpenSSL::X509::ExtensionFactory.new(nil,cert)
	cert.extensions = [
	  ef.create_extension("basicConstraints","CA:FALSE"),
	  ef.create_extension("subjectKeyIdentifier", "hash") ]
	ef.issuer_certificate = cert
	cert.add_extension(ef.create_extension("authorityKeyIdentifier",
					       "keyid:always,issuer:always"))
	if comment = self[:SSLCertComment]
	  cert.add_extension(ef.create_extension("nsComment", comment))
	end
	
	cert.sign(rsa, OpenSSL::Digest::SHA1.new)
	
	@rsa = rsa
	@cert = cert
      end
    end

    def self.parse_uri(uri)
      if uri =~ /^drbssl:\/\/(.*?):(\d+)(\?(.*))?$/
	host = $1
	port = $2.to_i
	option = $4
	[host, port, option]
      else
	raise(DRbBadScheme, uri) unless uri =~ /^drbssl:/
	raise(DRbBadURI, 'can\'t parse uri:' + uri)
      end
    end

    def self.open(uri, config)
      host, port, option = parse_uri(uri)
      host.untaint
      port.untaint

      soc = TCPSocket.open(host, port)

      ssl_conf = SSLConfig.new(config)
      ssl = ssl_conf.connect(soc)
      
      self.new(uri, soc, ssl, config)
    end

    def self.open_server(uri, config)
      uri = 'drbssl://:0' unless uri
      host, port, opt = parse_uri(uri)
      if host.size == 0
	soc = TCPServer.open(port)
	host = Socket.gethostname
      else
	soc = TCPServer.open(host, port)
      end
      port = soc.addr[1] if port == 0
      @uri = "drbssl://#{host}:#{port}"
      
      self.new(@uri, soc, nil, config)
    end

    def self.uri_option(uri, config)
      host, port, option = parse_uri(uri)
      return "drbssl://#{host}:#{port}", option
    end

    def initialize(uri, soc, ssl, config)
      super(uri, soc, config)
      @ssl = ssl
      unless ssl
	# server mode
	@ssl_conf = SSLConfig.new(config) 
	@ssl_conf.setup_keys
      end
    end
    
    def stream; @ssl; end

    def close
      if @ssl
	@ssl.close
	@ssl = nil
      end
      super
    end
      
    def accept
      while true
	s = @socket.accept
	break if (@acl ? @acl.allow_socket?(s) : true) 
	s.close
      end
      ssl = @ssl_conf.accept(s)
      self.type.new(uri, s, ssl, @config)
    end
  end
  
  DRbProtocol.add_protocol(DRbSSLSocket)
end
