/*
 * TDA8444 chip, Copyright 2000, Mika Korhonen
 *
 */

#include <linux/version.h>
#include <linux/config.h>

#if !defined(CONFIG_I2C) && !(defined(CONFIG_I2C_MODULE) && CONFIG_I2C_MODULE==1)
#error DMX4Linux I2C devices need I2C support in your kernel
#endif

#include <linux/module.h>
#include <linux/slab.h>
#include <linux/i2c.h>
#include <linux/init.h>

#ifndef I2C_DRIVERID_TDA8444 /* it is defined since 2.4.x */
#define I2C_DRIVERID_TDA8444 I2C_DRIVERID_EXP0
#endif

struct tda8444_args {
	unsigned char subaddress;
	unsigned char count;
	unsigned char values[8];
};

/* addresses to scan */
#define I2C_TDA8444_RANGE_BEGIN 0x40
#define I2C_TDA8444_RANGE_END 0x4E

#define TDA8444_AUTOINCR_SUBADDR 0x00   /* autoincrement subaddress */
#define TDA8444_FIXED_SUBADDR 0xF0      /* all writes to the same DAC-latch */

/* commands */
#define TDA8444_SETDAC_SAME 0x00
#define TDA8444_SETDAC_CONSECUTIVE 0x01
#define TDA8444_LASTCOMMAND TDA8444_SETDAC_CONSECUTIVE

static unsigned short normal_i2c[] = {I2C_CLIENT_END};
static unsigned short normal_i2c_range[] = { I2C_TDA8444_RANGE_BEGIN >> 1,
					     I2C_TDA8444_RANGE_END >> 1,
					     I2C_CLIENT_END };
static unsigned short probe[2]        = { I2C_CLIENT_END, I2C_CLIENT_END };
static unsigned short probe_range[2]  = { I2C_CLIENT_END, I2C_CLIENT_END };
static unsigned short ignore[2]       = { I2C_CLIENT_END, I2C_CLIENT_END };
static unsigned short ignore_range[2] = { I2C_CLIENT_END, I2C_CLIENT_END };
static unsigned short force[2]        = { I2C_CLIENT_END, I2C_CLIENT_END };

static struct i2c_client_address_data addr_data = {
        normal_i2c, normal_i2c_range, 
	probe, probe_range, 
	ignore, ignore_range, 
	force
};

static struct i2c_driver driver;
static int tda8444_id = 0;

int tda8444_command(struct i2c_client *client, unsigned int cmd, void *arg)
{
	unsigned short int i;
	unsigned char buffer[9];

	if (cmd > TDA8444_LASTCOMMAND) 
		return -EINVAL;
	else 
{
		struct tda8444_args *ta = arg;

		if (cmd ==  TDA8444_SETDAC_SAME) 
			buffer[0] = TDA8444_FIXED_SUBADDR | (0x0F & ta->subaddress);
		else 
			buffer[0] = TDA8444_AUTOINCR_SUBADDR | (0x0F & ta->subaddress);

		for (i = 0; i < ta->count; i++)
			buffer[i+1] = ta->values[i];
		
		return i2c_master_send(client, buffer, ta->count + 1);
	}
}

static int tda8444_detect_client(struct i2c_adapter *adapter, int address,
				 unsigned short flags, int kind)
{
	struct i2c_client *client;
	struct tda8444_args test_args = { 0, 8, {0, 0, 0, 0, 0, 0, 0, 0} };
	int r;

	const char *name = "TDA8444";
	
	if (!(client = kmalloc(sizeof(struct i2c_client), GFP_KERNEL)))
		return -ENOMEM;

	client->addr = address;
	client->data = NULL;
	client->adapter = adapter;
	client->driver = &driver;
	client->flags  = I2C_CLIENT_ALLOW_USE;  /* nessesary if searching this client using i2c_get_client */

	r = tda8444_command(client, TDA8444_SETDAC_CONSECUTIVE, &test_args);

	if (r > 0) 
{
		strcpy(client->name, name);
		client->id = tda8444_id++;
		
		r = i2c_attach_client(client);
		if (r) kfree(client);
		return r;
	}

	kfree(client);
	return 0;
}




int tda8444_attach_adapter(struct i2c_adapter *adapter)
{
	return i2c_probe(adapter, &addr_data, &tda8444_detect_client);
}

int tda8444_detach_client(struct i2c_client *client)
{
	i2c_detach_client(client);
	return 0;
}

void tda8444_inc_use (struct i2c_client *client)
{
#ifdef MODULE
	MOD_INC_USE_COUNT;
#endif
}

void tda8444_dec_use (struct i2c_client *client)
{
#ifdef MODULE
	MOD_DEC_USE_COUNT;
#endif
}

static struct i2c_driver driver = {
	.name = "I2C TDA8444 driver version 0.1",
	.id = I2C_DRIVERID_TDA8444,
	.flags = I2C_DF_NOTIFY,
	.attach_adapter = &tda8444_attach_adapter,
	.detach_client = &tda8444_detach_client,
	.command = &tda8444_command,
};

int __init tda8444_init(void)
{
/*  printk ("TDA8444\n");
*/
	i2c_add_driver(&driver);
	return 0;
}

void __exit tda8444_exit(void)
{
	i2c_del_driver(&driver);
}

#ifdef MODULE

MODULE_AUTHOR("Mika Korhonen <mikak@ee.oulu.fi>");
MODULE_DESCRIPTION("Driver for TDA8444 chip");
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,17)
MODULE_LICENSE("GPL");
#endif

module_init(tda8444_init);
module_exit(tda8444_exit);
#endif
