/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/unittest/std_test_case.hpp
 *
 * @brief [LEVEL: beta] @ref diagnostics::unittest::Std_Test_Case class 
 *
 * $Id: std_test_case.hpp,v 1.12 2005/06/23 09:54:24 esdentem Exp $
 * 
 * @author Christian Schallhart
 *
 * @test diagnostics/unittest/std_test_case.t.cpp
 */

#ifndef DIAGNOSTICS__UNITTEST__STD_TEST_CASE_HPP__INCLUDE_GUARD
#define DIAGNOSTICS__UNITTEST__STD_TEST_CASE_HPP__INCLUDE_GUARD

// used as base class
#include <diagnostics/unittest/test_case.hpp>

DIAGNOSTICS_NAMESPACE_BEGIN;
UNITTEST_NAMESPACE_BEGIN;

/**
 * @class Std_Test_Case diagnostics/unittest/std_test_case.hpp
 *
 * @brief The Test_Case class is a stdandard implementation of the
 * @ref Test_Case interface.
 *
 * Only the @ref run method is left pure virtual.
 *
 * @nosubgrouping
 */
class Std_Test_Case :
    public Test_Case
{
    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Types
     * @{
     */
private:
    typedef Std_Test_Case Self;

    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Disabled Creation
     * @{
     */
private:
    /**
     * @brief no default cstr
     */
    Std_Test_Case();

    /**
     * @brief no copy cstr
     */
    Std_Test_Case(Self const & other);

    /**
     * @brief no assignement
     */
    Self & operator=(Self const & other);
    // @} 

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Creation
     * @{
     */
protected:
    /**
     * @brief the only way to construct a Test_Case. To be used by a derived class.
     * 
     * @param name is the test case name.
     * @param timeout is the timeout of the Test_Case in seconds (HYBRID: 0 means no timeout at all).
     * @param is_rerunable is true iff the @ref run method can be called twice.
     * @param testable_at_prod is true iff the Test_Case is meant to succeed in a @ref LEVEL_PROD build.
     * @param testable_at_debug symmetric to @a testable_at_prod
     * @param testable_at_audit symmetric to @a testable_at_prod
     *
     * @throw Test_System_Exception if 
     * @arg @a name does not match the naming convention, i.e.,
     * it does contains a @ref DIAGNOSTICS_UNITTEST_TEST_CASE_NAME_SEPARATOR or a 
     * @ref DIAGNOSTICS_UNITTEST_TEST_DATA_NAME_SEPARATOR or equals the empty string.
     * @arg @a timeout is negative.
     */
    Std_Test_Case(::std::string const & name,
		  int const timeout,
		  bool const is_rerunable,
		  bool const testable_at_prod,
		  bool const testable_at_debug,
		  bool const testable_at_audit);

    /**
     * @throw never
     */
    virtual ~Std_Test_Case();
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Hooks
     * @{
     */
public:
    /**
     * @brief The hook method called by the testing environment to invoke the Test_Case
     * 
     * This method is supposed to terminate within @ref timeout() seconds, and to produce
     * no error in the build levels which are described by @ref is_testable_at().
     * This method is only allowed to be invoked twice within the same process iff
     * @ref is_rerunable() returns true. The latter is useful for testcases which
     * test static entities. 
     * @throw anything
     */
    virtual void run(Test_Data & test_data) const=0;
    // @}
    
    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Accessors
     * @{
     */
public:
    /**
     * @brief returns true for @a level if the Test_Case is supposed
     * to succeed at build level @a level 
     *
     * @pre @a level != @ref LEVEL_TEST
     *
     * @throw Test_System_Exception
     */
    virtual bool is_testable_at(Level_t const level) const;

    /**
     * @brief the name of the Test_Case
     *
     * @throw never
     */
    virtual ::std::string const & name() const;

    /**
     * @brief the timeout of the Test_Case in seconds
     *
     * @throw never
     */
    virtual int timeout() const;
    
    /**
     * @brief returns true if the Test_Case can be run more than once
     *
     * @throw never
     */
    virtual bool is_rerunable() const;
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Encapsulated State
     * @{
     */
private:
    ::std::string const m_name;
    int const m_timeout;
    bool const m_is_rerunable;
    int m_testable_at;
    // @}
};


UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;


#endif
// vim:ts=4:sw=4
