/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/frame/record.cpp
 *
 * $Id: record.cpp,v 1.7 2005/06/23 09:54:20 esdentem Exp $
 *
 * @author Christian Schallhart
 *
 * @brief [LEVEL: beta] Implementation of @ref diagnostics::Record
 */

#include <diagnostics/frame/record.hpp>

DIAGNOSTICS_NAMESPACE_BEGIN;

Record::Record(Level_t const level,
	       Type_t const type,
	       int const nr_what,
	       ::std::string const & str_what,
	       char const * const base_file_name,
	       char const * const file_name,
	       int const line
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
	       ,
	       Pid_t const pid,
	       Tid_t const tid,
	       Sec_t const sec,
	       Usec_t const usec,
	       char const * const hostname
#endif
	       ) 
    : m_level(level),
      m_type(type),
      m_nr_what(nr_what),
      m_str_what(str_what),
      m_base_file_name(base_file_name),
      m_file_name(file_name),
      m_line(line)
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
      ,
      m_pid(pid),
      m_tid(tid),
      m_sec(sec),
      m_usec(usec),
      m_hostname(hostname)
#endif
{
}

Record::Record(Self const & other) 
    : m_level(other.m_level),
      m_type(other.m_type),
      m_nr_what(other.m_nr_what),
      m_str_what(other.m_str_what),
      m_base_file_name(other.m_base_file_name),
      m_file_name(other.m_file_name),
      m_line(other.m_line)
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
      ,
      m_pid(other.m_pid),
      m_tid(other.m_tid),
      m_sec(other.m_sec),
      m_usec(other.m_usec),
      m_hostname(other.m_hostname)
#endif
{
}

Record & Record::operator=(Self const & other)
{
    m_level=other.m_level;
    m_type=other.m_type;
    m_nr_what=other.m_nr_what;
    m_str_what=other.m_str_what;
    m_base_file_name=other.m_base_file_name;
    m_file_name=other.m_file_name;
    m_line=other.m_line;
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
    m_pid=other.m_pid;
    m_tid=other.m_tid;
    m_sec=other.m_sec;
    m_usec=other.m_usec;
    m_hostname=other.m_hostname;
#endif
    return *this;
}

bool Record::operator==(Self const & other) const
{
    return m_level==other.m_level
	&& m_type==other.m_type
	&& m_nr_what==other.m_nr_what
	&& m_str_what==other.m_str_what
	&& m_base_file_name==other.m_base_file_name
	&& m_file_name==other.m_file_name
	&& m_line==other.m_line
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
	&& m_pid==other.m_pid
	&& m_tid==other.m_tid
	&& m_sec==other.m_sec
	&& m_usec==other.m_usec
	&& m_hostname==other.m_hostname
#endif
	;
}


DIAGNOSTICS_NAMESPACE_END;

::std::ostream & operator<<(::std::ostream & stream, ::diagnostics::Record const & record)
{
    using namespace ::diagnostics;
    
    stream << level_to_string(record.level()) << " "
	   << type_to_string(record.type()) << " "
	   << record.nr_what() << " " 
	   << record.str_what() << " ["
	   << record.base_file_name() << " "
	   << record.file_name() << " "
	   << record.line() << "] "
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
	   << record.pid() << "/" 
	   << record.tid() << "/"
	   << record.sec() << "-"
	   << record.usec() << " " 
	   << record.hostname()
#endif
	;
    return stream;
}

// vim:ts=4:sw=4
