/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


#ifndef DIAGNOSTICS__EXTENSIONS__MEMORY__ANNOTATIONS__INCLUDE_GUARD
#define DIAGNOSTICS__EXTENSIONS__MEMORY__ANNOTATIONS__INCLUDE_GUARD

#include <diagnostics/frame/platform_config.hpp>

#if DIAGNOSTICS_SWITCH_MEMORY_ANALYSIS_ENABLED == 0

#  define DIAGNOSTICS_MEMORY_BASE_BALANCE_ENTER(LEVEL,LOG,CHECK) do { ((void)0)
#  define DIAGNOSTICS_MEMORY_BASE_BALANCE_EXIT                   } while(false)

#  define DIAGNOSTICS_MEMORY_BASE_LOG_BALANCE ((void)0)
#  define DIAGNOSTICS_MEMORY_BASE_ASSERT_BALANCE ((void)0)
#  define DIAGNOSTICS_MEMORY_BASE_LOG_TOTAL_BALANCE ((void)0)

#else

#  include <diagnostics/extensions/memory/tick.hpp>

#  include <diagnostics/frame/level.hpp>
#  include <diagnostics/frame/type.hpp>
#  include <diagnostics/frame/pp_config.hpp>

#  include <string>



DIAGNOSTICS_NAMESPACE_BEGIN;
MEMORY_NAMESAPCE_BEGIN;
INTERNAL_NAMESPACE_BEGIN;

class Allocation_Balance_Guard
{
private:
	typedef Allocation_Balance_Guard Self;
	Allocation_Balance_Guard(Self const & other);
	Self & operator=(Self const & other);
public:
    Allocation_Balance_Guard(char const * const base_file_name,
							 char const * const file_name,
							 Level_t const level,
							 int const line,
							 bool const log,
							 bool const check,
							 Type_t const type);
	~Allocation_Balance_Guard();

	
    void assert_balance   (int const line,Type_t const type);
    void log_balance      (int const line);
    void log_total_balance(int const line);

    
private:
    char const * const m_base_file_name;
    char const * const m_file_name;
	int const m_line;
	Level_t const m_level;
	bool const m_log;
	bool const m_check;
	Type_t const m_type;
    Tick_t const m_creation_tick;
    Tick_t m_entering_total_number;
    ::std::size_t m_entering_total_size;
};

INTERNAL_NAMESPACE_END;
MEMORY_NAMESAPCE_END;
DIAGNOSTICS_NAMESPACE_END;

#  define DIAGNOSTICS_MEMORY_BASE_BALANCE_ENTER(LEVEL,LOG,CHECK,TYPE) \
          do { ::DIAGNOSTICS_NAMESPACE::MEMORY_NAMESAPCE::INTERNAL_NAMESPACE::Allocation_Balance_Guard \
			  diagnostics_internal_memory_guard(DIAGNOSTICS_BASE_FILE, __FILE__, LEVEL, __LINE__, LOG, CHECK,TYPE); { ((void)0)
#  define DIAGNOSTICS_MEMORY_BASE_BALANCE_EXIT } } while(false)

#  define DIAGNOSTICS_MEMORY_BASE_LOG_BALANCE diagnostics_internal_memory_guard.log_balance(__LINE__)
#  define DIAGNOSTICS_MEMORY_BASE_ASSERT_BALANCE diagnostics_internal_memory_guard.assert_balance(__LINE__, TYPE_TRACE)
#  define DIAGNOSTICS_MEMORY_BASE_LOG_TOTAL_BALANCE diagnostics_internal_memory_guard.log_total_balance(__LINE__)


#endif


/**
 * @brief This macro starts a block which turns on memory analysis --
 * but without logging memory allocation/deallocation and without
 * checking whether the allocations/deallocations are balanced, i.e.,
 * all allocations/deallocations are registered in the internal
 * allocation-database. Consequently, double-deallocations are
 * detected (in terms of unknown block deallocation) and a complete
 * allocation-database is maintained which avoids @ref
 * TYPE_FAILED_ASSERTION for blocks which have been allocated before
 * using @ref TEST_MEMORY_BALANCE_ENTER or a @ref
 * TEST_MEMORY_LOG_ENTER macro.
 */
#define TEST_MEMORY_ANALYSIS_ENTER \
    DIAGNOSTICS_MEMORY_BASE_BALANCE_ENTER(::diagnostics::LEVEL_TEST,false,false, ::diagnostics::TYPE_TRACE)
#define TEST_MEMORY_ANALYSIS_EXIT  DIAGNOSTICS_MEMORY_BASE_BALANCE_EXIT


/**
 * @brief This macro checks whether @arg all chunks of which are
 * allocated inside of the framed block are deallocated again @arg and
 * no other blocks are deallocated inside of the framed block, i.e., a
 * block framed by these two macros must be completely neutral with
 * respect to memory consumption.
 *
 * If this is not the case, a corresponding message is written to the
 * log: Either a Leak (not freeing memory) or an Antileak (freeing too
 * much memory) is reported. These messages come with type @param TYPE
 * -- usually it is set to @ref ::diagnostics::TYPE_FAILED_ASSERTION
 * or @ref ::diagnostics::TYPE_TRACE.
 *
 */
#define TEST_MEMORY_BALANCE_ENTER(TYPE) \
    DIAGNOSTICS_MEMORY_BASE_BALANCE_ENTER(::diagnostics::LEVEL_TEST,false,true, (TYPE))
#define TEST_MEMORY_BALANCE_EXIT  DIAGNOSTICS_MEMORY_BASE_BALANCE_EXIT

/**
 * @brief This macro starts a block which switches on memory logging
 * -- all allocations and deallocations are logged (LEVEL_AUDIT).
 */
#define TEST_MEMORY_LOG_ENTER \
    DIAGNOSTICS_MEMORY_BASE_BALANCE_ENTER(::diagnostics::LEVEL_TEST,true,false, ::diagnostics::TYPE_TRACE)
#define TEST_MEMORY_LOG_EXIT  DIAGNOSTICS_MEMORY_BASE_BALANCE_EXIT

/**
 * @brief Logs the number of blocks and the summed size of the
 * currently allocated memory (LEVEL_TEST) with respect to next @ref
 * TEST_MEMORY_LOG_ENTER or @ref TEST_MEMORY_BALANCE_ENTER context.
 */
#define TEST_MEMORY_LOG_BALANCE DIAGNOSTICS_MEMORY_BASE_LOG_BALANCE

/**
 * @brief Asserts (LEVEL_TEST) that the currently allocation status is
 * balanced with respect to next @ref TEST_MEMORY_LOG_ENTER or @ref
 * TEST_MEMORY_BALANCE_ENTER context. It is also asserted that no
 * blocks which had not been allocated within this context have been
 * deallocated.
 */
#define TEST_MEMORY_ASSERT_BALANCE DIAGNOSTICS_MEMORY_BASE_ASSERT_BALANCE

/**
 * @brief Logs the number of blocks and the summed size of the
 * currently allocated memory (LEVEL_TEST) with respect to outer-most
 * @ref TEST_MEMORY_LOG_ENTER, @ref TEST_MEMORY_BALANCE_ENTER or @ref
 * TEST_MEMORY_ANALYSIS_ENTER context.
 */
#define TEST_MEMORY_LOG_TOTAL_BALANCE DIAGNOSTICS_MEMORY_BASE_LOG_TOTAL_BALANCE

#endif

// vim:ts=4:sw=4
