/*
    Copyright (C) 2008-2010 Stefan Haller

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include "wallpapers.h"

#include "desktopnova-daemon.h"
#include "../misc/general.h"
#include "../misc/xml.h"

gchar ** filters = NULL;
GPtrArray * wallpapers = NULL;

void init_filters(const gchar * filter_string)
{
	const gchar * tmp = filter_string;
	if (tmp == NULL)
	{
		tmp = "";
	}
	if (g_strcmp0(tmp, "") == 0)
	{
		tmp = DEFAULT_FILTERS;
	}

	filters = g_strsplit(tmp, " ", -1);
}

void free_filters()
{
	g_strfreev(filters);
}

gboolean check_extension(const gchar * filename)
{
	/* get last . */
	gchar * found = g_strrstr(filename, ".\0");
	if (found == NULL)
	{
		return FALSE;
	}

	/* Is dot last character in string? */
	if (found == filename + strlen(filename))
	{
		return FALSE;
	}
	/* remove dot */
	found += 1;

	gboolean result = FALSE;
	if (filters != NULL)
	{
		gchar ** filter = filters;
		while (*filter != NULL)
		{
			if (g_strcasecmp(found, *filter) == 0)
			{
				result = TRUE;
				break;
			}
			filter += 1;
		}
	}

	return result;
}

void add_wallpaper(GPtrArray * array, const gchar * filename)
{
	/* check file extension */
	if (! check_extension(filename))
		return;	

	/* quick and dirty: Can we read the file? */
	int fd = open(filename, O_RDONLY);
	if (fd > -1)
	{
		/* file opened - we can read :) */
		gchar * filename_dup = g_strdup(filename);
		g_ptr_array_add(array, filename_dup);
	}
	close(fd);
}

void clear_wallpapers(GPtrArray * array)
{
	guint i;
	for (i = 0; i < array->len; i++)
	{
		gchar * filename = (gchar *)g_ptr_array_index(array, i);
		g_free(filename);
	}
	g_ptr_array_free(array, TRUE);
}

void browse_directory(GPtrArray * array, const gchar * dirname, gboolean subfolders)
{
	GDir * dir = g_dir_open(dirname, 0, NULL);
	if (dir != NULL)
	{
		const gchar * filename;
		while ((filename = g_dir_read_name(dir)) != NULL)
		{
			gchar * absolute_filename = g_build_filename(dirname,
			                                             filename,
			                                             NULL);
			struct stat buf;
			if (/*g_*/stat(absolute_filename, &buf) == 0)
			{
				if (S_ISDIR(buf.st_mode))
				{
					if (subfolders)
					{
						browse_directory(array, absolute_filename, subfolders);
					}
				}
				else
				{
					add_wallpaper(array, absolute_filename);
				}
			}
			else
			{
				/*
				TODO: Error handling
				*/
				error("stat");
			}
			g_free(absolute_filename);
		}
		g_dir_close(dir);
	}
}

void scan_files(struct file_node * node)
{
	while (node != NULL)
	{
		if (node->is_dir)
		{
			browse_directory(wallpapers, node->filename, node->use_subfolders);
		}
		else
		{
			add_wallpaper(wallpapers, node->filename);
		}
		node = node->next;
	}

	return;
}
