/* processstaffname.c
 * for Denemo, the GNU graphical frontend to GNU Lilypond
 *
 * (c) 2000, 2001 Matthew Hiller */

/* This function accepts a Denemo staff's canonicalized name
 * denemo_name, and sets lily_name to be what the corresponding voice
 * will be called in the mudela representation. Both GString *s should
 * be initialized before this function is called */
/* e.g., "bass voice"->"BassVoice" */

#include <glib.h>
#include <string.h>

void
set_lily_name (GString * denemo_name, GString * lily_name)
{
  gint i;
  gchar c;
  gboolean last_char_was_space = TRUE;

  g_string_assign (lily_name, "");
  for (i = 0; i < denemo_name->len; i++)
    {
      c = denemo_name->str[i];
      if (c == ' ')
	last_char_was_space = TRUE;
      else
	{
	  if (last_char_was_space && 'a' <= c && c <= 'z')
	    /* Make the character uppercase */
	    c -= ('a' - 'A');
	  g_string_append_c (lily_name, c);
	  last_char_was_space = FALSE;
	}
    }
}

/* This function accepts a Lilypond voice name lily_name and calculates
 * from it the canonicalized denemo_name. Both Gstring *s should be
 * initialized before this function is called */
/* e.g., "BassVoice"->"bass voice" */

void
set_denemo_name (GString * lily_name, GString * denemo_name)
{
  gint i;
  gchar c;

  g_string_assign (denemo_name, "");
  for (i = 0; i < lily_name->len; i++)
    {
      c = lily_name->str[i];
      if ('A' <= c && c <= 'Z')
	{
	  /* Need to make the character lowercase and insert a space */
	  c += ('a' - 'A');
	  if (i)
	    /* insert a space before it as well */
	    g_string_append_c (denemo_name, ' ');
	}
      g_string_append_c (denemo_name, c);
    }
}

/* This function accepts a gchar *proposal as an argument. If 
 * this proposal can be made into a canonical Denemo staff name,
 * it is stored in denemo_name. If not, denemo_name is left
 * unchanged. It returns -1 if the name could not be canonicalized. */

gint canonicalize_denemo_name (gchar * proposal, GString * denemo_name)
{
  gchar *accept = "abcdefghijklmnopqrstuvwxyz1234567890 ";
  gint i;
  gchar c;
  gboolean last_char_was_space = FALSE;

  g_strdown (proposal);

  if (strspn (proposal, accept) == strlen (proposal))
    {
      /* Okay; we have only acceptable characters. Let's boogie */
      g_string_assign (denemo_name, proposal);
      for (i = 0; i < denemo_name->len;)
	{
	  c = denemo_name->str[i];
	  if (c == ' ')
	    {
	      if (last_char_was_space)
		g_string_erase (denemo_name, i, 1);
	      else
		{
		  last_char_was_space = TRUE;
		  i++;
		}
	    }
	  else
	    {
	      last_char_was_space = FALSE;
	      i++;
	    }
	}
      return 0;
    }
  return -1;
}
