/*
 * 	Denemo File IO 
 * 	(c) Adam Tee, Matthew Hiller 2000, 2001
 * 	(c) University of Leeds 2000, 2001
 */

#include "calculatepositions.h"
#include "contexts.h"
#include "datastructures.h"
#include "dialogs.h"
#include "exportabc.h"
#include "exportmudela.h"
#include "file.h"
#include "frogio.h"
#include "frogdefs.h"
#include "moveviewport.h"
#include "staffops.h"
#include "scoreops.h"
#include "utils.h"
#include "exportxml.h"
#include "exportmidi.h"
#include "importxml.h"

#include <stdio.h>
#include <string.h>

#define DENEMO_PATTERN "Denemo XML format (*.denemo)"
#define MUDELA_PATTERN "Mudela (*.ly)"
#define ABC_PATTERN    "ABC (*.abc)"
#define JTF_PATTERN    "Unnamed file format (*.jtf)"
#define MIDI_PATTERN   "Midi (*.midi)"
#define JTF_POLY_PATTERN "Unnamed file format (*.jtf) Poly"

struct callbackdata
{
  struct scoreinfo *si;
  GtkWidget *fs;
  GtkWidget *comboentry;
};

void
confirmbox (struct scoreinfo *si, GtkSignalFunc signalfunc)
{
  GtkWidget *dialog, *yes, *no, *label;

  if (si->haschanged)
    {
      dialog = gtk_dialog_new ();
      gtk_window_set_title (GTK_WINDOW (dialog), _("Confirm"));

      label = gtk_label_new
	(_("The current score has changes in it which you have not saved."));
      gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label,
			  TRUE, TRUE, 0);
      gtk_widget_show (label);
      label = gtk_label_new (_("Proceed with the operation nonetheless?"));
      gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label,
			  TRUE, TRUE, 0);
      gtk_widget_show (label);
      yes = gtk_button_new_with_label (_("Yes"));
      gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area),
			  yes, TRUE, TRUE, 0);
      gtk_widget_show (yes);

      no = gtk_button_new_with_label (_("No"));
      gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area),
			  no, TRUE, TRUE, 0);
      gtk_widget_show (no);

      gtk_signal_connect (GTK_OBJECT (yes), "clicked",
			  GTK_SIGNAL_FUNC (signalfunc), si);
      gtk_signal_connect_object (GTK_OBJECT (yes), "clicked",
				 GTK_SIGNAL_FUNC (gtk_widget_destroy),
				 GTK_OBJECT (dialog));
      gtk_signal_connect_object (GTK_OBJECT (no), "clicked",
				 GTK_SIGNAL_FUNC (gtk_widget_destroy),
				 GTK_OBJECT (dialog));

      gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
      gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_MOUSE);
      gtk_widget_show (dialog);
    }
  else
    signalfunc (NULL, si);
}

/* This function is to be called after a file load of any sort.  */

void
updatescoreinfo (struct scoreinfo *si)
{
  staffnode *curstaff;

  for (curstaff = si->thescore; curstaff; curstaff = curstaff->next)
    {
      beamsandstemdirswholestaff (curstaff->data);
      showwhichaccidentalswholestaff (curstaff->data);
    }
  find_xes_in_all_measures (si);
  find_leftmost_allcontexts (si);

  si->currentstaff = si->thescore;
  si->currentmeasure = firstmeasurenode (si->currentstaff);
  si->currentobject = firstobjnode (si->currentmeasure);
  if (!si->currentobject)
    si->cursor_appending = TRUE;
  else
    si->cursor_appending = FALSE;
  si->leftmeasurenum = si->currentstaffnum = si->currentmeasurenum = 1;
  si->cursor_x = 0;
  si->haschanged = FALSE;
  set_rightmeasurenum (si);
  set_bottom_staff (si);
  update_hscrollbar (si);
  update_vscrollbar (si);
  gtk_widget_draw (si->scorearea, NULL);
}

void
set_si_filename (struct scoreinfo *si, gchar * filename)
{
  gchar *dialog_title = g_strconcat ("Denemo - ", filename, NULL);;
  gtk_window_set_title (GTK_WINDOW (si->window), dialog_title);
  g_free (dialog_title);
  g_string_assign (si->filename, filename);
}

/* The function that actually determines the file type and calls the
   function that opens the file.  (So many layers of indirection...)  */

void
open_for_real (gchar * filename, struct scoreinfo *si)
{
  gint result;

  if (strcmp (filename + strlen (filename) - 7, ".denemo") == 0)
    result = importXML (filename, si);
  else if (strcmp (filename + strlen (filename) - 3, ".ly") == 0)
    result = lyinput (filename, si);
  else
    result = froginput (filename, si);
  if (result != -1)
    {
      set_si_filename (si, filename);
      updatescoreinfo (si);
      gtk_signal_emit_by_name (GTK_OBJECT (si->hadjustment), "changed");
      gtk_signal_emit_by_name (GTK_OBJECT (si->vadjustment), "changed");
    }
}

/* A callback that acts as a wrapper to open_for_real */

void
filesel_open (GtkWidget * widget, gpointer data)
{
  struct callbackdata *cbdata = data;
  struct scoreinfo *si = cbdata->si;
  gchar *filename =
    gtk_file_selection_get_filename (GTK_FILE_SELECTION (cbdata->fs));

  open_for_real (filename, si);
}

/* A callback that actually does the saving based on what was entered
   in the file selection widget.  */

void
filesel_save (GtkWidget * widget, gpointer data)
{
  struct callbackdata *cbdata = data;
  struct scoreinfo *si = cbdata->si;
  gchar *filename =
    gtk_file_selection_get_filename (GTK_FILE_SELECTION (cbdata->fs));
  gchar *etext = gtk_entry_get_text (GTK_ENTRY (cbdata->comboentry));

  if (strcmp (etext, _(DENEMO_PATTERN)) == 0)
    {
      if (strcmp (filename + strlen (filename) - 7, ".denemo") == 0)
	filename = g_strdup (filename);
      else
	filename = g_strconcat (filename, ".denemo", NULL);
      exportXML (filename, si, 0, 0);
    }
  else if (strcmp (etext, _(MUDELA_PATTERN)) == 0)
    {
      if (strcmp (filename + strlen (filename) - 3, ".ly") == 0)
	filename = g_strdup (filename);
      else
	filename = g_strconcat (filename, ".ly", NULL);
      exportmudela (filename, si, 0, 0);
    }
  else if (strcmp (etext, _(ABC_PATTERN)) == 0)
    {
      if (strcmp (filename + strlen (filename) - 4, ".abc") == 0)
	filename = g_strdup (filename);
      else
	filename = g_strconcat (filename, ".abc", NULL);
      exportabc (filename, si, 0, 0);
    }
  else if (strcmp (etext, _(JTF_PATTERN)) == 0)
    {
      if (strcmp (filename + strlen (filename) - 4, ".jtf") == 0)
	filename = g_strdup (filename);
      else
	filename = g_strconcat (filename, ".jtf", NULL);
      filesave (filename, si, 0, 0, 0);
    }
  else if (strcmp (etext, _(JTF_POLY_PATTERN)) == 0)
    {
      if (strcmp (filename + strlen (filename) - 4, ".jtf") == 0)
	filename = g_strdup (filename);
      else
	filename = g_strconcat (filename, ".jtf", NULL);
      filesave (filename, si, 0, 0, 1);
    }
  else
    {
      if (strcmp (filename + strlen (filename) - 5, ".midi") == 0)
	filename = g_strdup (filename);
      else
	filename = g_strconcat (filename, ".midi", NULL);
      exportmidi (filename, si, 0, 0);
    }

  si->haschanged = FALSE;
  set_si_filename (si, filename);
  g_free (filename);
}

void
changecompletion (GtkWidget * widget, gpointer data)
{
  GtkWidget *filesel = data;
  gchar *etext = gtk_entry_get_text (GTK_ENTRY (widget));

  if (strcmp (etext, _(DENEMO_PATTERN)) == 0)
    gtk_file_selection_set_filename (GTK_FILE_SELECTION (filesel),
				     "*.denemo");
  else if (strcmp (etext, _(MUDELA_PATTERN)) == 0)
    gtk_file_selection_set_filename (GTK_FILE_SELECTION (filesel), "*.ly");
  else if (strcmp (etext, _(ABC_PATTERN)) == 0)
    gtk_file_selection_set_filename (GTK_FILE_SELECTION (filesel), "*.abc");
  else if (strcmp (etext, _(JTF_PATTERN)) == 0)
    gtk_file_selection_set_filename (GTK_FILE_SELECTION (filesel), "*.jtf");
  else if (strcmp (etext, _(MIDI_PATTERN)) == 0)
    gtk_file_selection_set_filename (GTK_FILE_SELECTION (filesel), "*.midi");
  else if (strcmp (etext, _(JTF_POLY_PATTERN)) == 0)
    gtk_file_selection_set_filename (GTK_FILE_SELECTION (filesel), "*.jtf");
}

void
file_openwrapper (gpointer callback_data,
		  guint callback_action, GtkWidget * widget)
{
  confirmbox (callback_data, GTK_SIGNAL_FUNC (file_open));
}

/* File open dialog - opened where appropriate */

void
file_open (GtkWidget * widget, gpointer callback_data)
{
  GtkWidget *filesel;
  GList *glist = NULL;
  GtkWidget *label;
  GtkWidget *combobox;
  static struct callbackdata cbdata;

  cbdata.si = callback_data;
  filesel = gtk_file_selection_new (_("Open"));
  gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (filesel)->ok_button),
		      "clicked", filesel_open, &cbdata);
  cbdata.fs = filesel;
  gtk_signal_connect_object (GTK_OBJECT
			     (GTK_FILE_SELECTION (filesel)->ok_button),
			     "clicked", gtk_widget_destroy,
			     (gpointer) filesel);

  gtk_signal_connect_object (GTK_OBJECT
			     (GTK_FILE_SELECTION (filesel)->cancel_button),
			     "clicked", gtk_widget_destroy,
			     (gpointer) filesel);

  label = gtk_label_new (_("Select file format:"));
  gtk_box_pack_start (GTK_BOX (GTK_FILE_SELECTION (filesel)->main_vbox),
		      label, TRUE, TRUE, 0);
  gtk_widget_show (label);

  glist = g_list_append (glist, _(DENEMO_PATTERN));
  glist = g_list_append (glist, _(MUDELA_PATTERN));
  glist = g_list_append (glist, _(JTF_PATTERN));

  combobox = gtk_combo_new ();
  gtk_combo_set_popdown_strings (GTK_COMBO (combobox), glist);
  gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (combobox)->entry),
		      _(DENEMO_PATTERN));
  gtk_file_selection_set_filename (GTK_FILE_SELECTION (filesel), "*.denemo");
  cbdata.comboentry = GTK_COMBO (combobox)->entry;
  gtk_editable_set_editable (GTK_EDITABLE (cbdata.comboentry), FALSE);
  gtk_box_pack_start (GTK_BOX (GTK_FILE_SELECTION (filesel)->main_vbox),
		      combobox, TRUE, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (GTK_COMBO (combobox)->entry),
		      "changed", GTK_SIGNAL_FUNC (changecompletion), filesel);
  gtk_widget_show (combobox);
  gtk_window_set_modal (GTK_WINDOW (filesel), TRUE);
  gtk_window_set_position (GTK_WINDOW (filesel), GTK_WIN_POS_MOUSE);
  gtk_widget_show (filesel);
}

void
file_saveaswrapper (gpointer callback_data, guint callback_action,
		    GtkWidget * widget)
{
  file_saveas (widget, callback_data);
}

void
file_savewrapper (gpointer callback_data, guint callback_action,
		  GtkWidget * widget)
{
  file_save (widget, callback_data);
}

void
file_save (GtkWidget * widget, gpointer callback_data)
{
  struct scoreinfo *si = callback_data;
  if (strcmp (si->filename->str, "") == 0)
    /* No filename's been given */
    file_saveas (widget, callback_data);

  else if (strcmp (si->filename->str + si->filename->len - 7, ".denemo") == 0)
    exportXML (si->filename->str, si, 0, 0);
  else if (strcmp (si->filename->str + si->filename->len - 3, ".ly") == 0)
    exportmudela (si->filename->str, si, 0, 0);
  else if (strcmp (si->filename->str + si->filename->len - 4, ".abc") == 0)
    exportabc (si->filename->str, si, 0, 0);
  else if (strcmp (si->filename->str + si->filename->len - 4, ".jtf") == 0)
    filesave (si->filename->str, si, 0, 0, 0);
  else
    exportmidi (si->filename->str, si, 0, 0);
  si->haschanged = FALSE;
}

void
file_saveas (GtkWidget * widget, gpointer callback_data)
{
  GtkWidget *filesel;
  GList *glist = NULL;
  GtkWidget *label;
  GtkWidget *combobox;
  static struct callbackdata cbdata;

  filesel = gtk_file_selection_new (_("Save As"));
  gtk_window_set_modal (GTK_WINDOW (filesel), TRUE);
  gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (filesel)->ok_button),
		      "clicked", filesel_save, &cbdata);
  cbdata.si = callback_data;
  cbdata.fs = filesel;
  gtk_signal_connect_object (GTK_OBJECT
			     (GTK_FILE_SELECTION (filesel)->ok_button),
			     "clicked", gtk_widget_destroy,
			     (gpointer) filesel);

  gtk_signal_connect_object (GTK_OBJECT
			     (GTK_FILE_SELECTION (filesel)->cancel_button),
			     "clicked", gtk_widget_destroy,
			     (gpointer) filesel);

  label = gtk_label_new (_("Select file format:"));
  gtk_box_pack_start (GTK_BOX (GTK_FILE_SELECTION (filesel)->main_vbox),
		      label, TRUE, TRUE, 0);
  gtk_widget_show (label);

  glist = g_list_append (glist, _(DENEMO_PATTERN));
  glist = g_list_append (glist, _(MUDELA_PATTERN));
  glist = g_list_append (glist, _(ABC_PATTERN));
  glist = g_list_append (glist, _(JTF_PATTERN));
  glist = g_list_append (glist, _(JTF_POLY_PATTERN));
  glist = g_list_append (glist, _(MIDI_PATTERN));
  combobox = gtk_combo_new ();
  gtk_combo_set_popdown_strings (GTK_COMBO (combobox), glist);
  gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (combobox)->entry),
		      _(DENEMO_PATTERN));
  gtk_file_selection_set_filename (GTK_FILE_SELECTION (filesel), "*.denemo");
  cbdata.comboentry = GTK_COMBO (combobox)->entry;
  gtk_editable_set_editable (GTK_EDITABLE (cbdata.comboentry), FALSE);
  gtk_box_pack_start (GTK_BOX (GTK_FILE_SELECTION (filesel)->main_vbox),
		      combobox, TRUE, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (GTK_COMBO (combobox)->entry),
		      "changed", GTK_SIGNAL_FUNC (changecompletion), filesel);
  gtk_widget_show (combobox);
  gtk_window_set_position (GTK_WINDOW (filesel), GTK_WIN_POS_MOUSE);
  gtk_widget_show (filesel);
}

void
file_newwrapper (gpointer callback_data,
		 guint callback_action, GtkWidget * widget)
{
  confirmbox (callback_data, GTK_SIGNAL_FUNC (deletescore));
}

void
deletescore (GtkWidget * widget, struct scoreinfo *si)
{
  free_score (si);
  newstaff (si, INITIAL, NULL);
  set_rightmeasurenum (si);
  update_hscrollbar (si);
  update_vscrollbar (si);
  gtk_widget_draw (si->scorearea, NULL);
}
