/* exportxml.c
 * Functions for exporting what Denemo's working on to an XML file (adapted
 * somewhat from exportabc.c)
 *
 * for Denemo, a gtk+ frontend to GNU Lilypond
 * (c) 2001 Eric Galluzzo
 */

#include "config.h"
#include "datastructures.h"
#include "exportxml.h"
#include "utils.h"

#include <stdlib.h>
#include <string.h>

/* libxml includes: for libxml2 this should be <libxml/tree.h> */
#include <tree.h>


#define XML_COMPRESSION_RATIO 3


/* The list of export handlers */
static GList *sExportHandlers = NULL;

/*
 * The map from staves, voices, chords, etc. to XML IDs
 *
 * FIXME: This won't work for multi-threaded applications!  I should really
 *        encapsulate these in a nice DenemoXMLContext struct or some such
 *        thing (or does libxml already provide generating unique IDs?) that I
 *        pass around.
 */
static gint sNextXMLID = 0;
static GHashTable *sStructToXMLIDMap = NULL;


/*
 * Free the value (a string) of the given key/value pair.  For use with
 * g_hash_table_foreach.
 */
static void
freeHashTableValue (gpointer key, gpointer value, gpointer userData)
{
  g_free (value);
}


static void
determineKeySignature (gint number, gboolean isMinor, gchar ** baseNote,
		       gchar ** baseAcc, gchar ** mode)
{
  if (isMinor)
    {
      number += 3;
      *mode = "minor";
    }
  else
    {
      *mode = "major";
    }

  switch (number)
    {
    case -7:
      *baseNote = "C";
      *baseAcc = "flat";
      break;
    case -6:
      *baseNote = "G";
      *baseAcc = "flat";
      break;
    case -5:
      *baseNote = "D";
      *baseAcc = "flat";
      break;
    case -4:
      *baseNote = "A";
      *baseAcc = "flat";
      break;
    case -3:
      *baseNote = "E";
      *baseAcc = "flat";
      break;
    case -2:
      *baseNote = "B";
      *baseAcc = "flat";
      break;
    case -1:
      *baseNote = "F";
      *baseAcc = NULL;
      break;
    case 0:
      *baseNote = "C";
      *baseAcc = NULL;
      break;
    case 1:
      *baseNote = "G";
      *baseAcc = NULL;
      break;
    case 2:
      *baseNote = "D";
      *baseAcc = NULL;
      break;
    case 3:
      *baseNote = "A";
      *baseAcc = NULL;
      break;
    case 4:
      *baseNote = "E";
      *baseAcc = NULL;
      break;
    case 5:
      *baseNote = "B";
      *baseAcc = NULL;
      break;
    case 6:
      *baseNote = "F";
      *baseAcc = "sharp";
      break;
    case 7:
      *baseNote = "C";
      *baseAcc = "sharp";
      break;
    case 8:
      *baseNote = "G";
      *baseAcc = "sharp";
      break;
    case 9:
      *baseNote = "D";
      *baseAcc = "sharp";
      break;
    case 10:
      *baseNote = "A";
      *baseAcc = "sharp";
      break;
    default:
      if (isMinor)
	number -= 3;
      g_warning ("Unknown key signature with %d %s, using C major",
		 abs (number), number < 0 ? "flats" : "sharps");
      *baseNote = "C";
      *baseAcc = NULL;
      *mode = "major";
      break;
    }
}


static void
determineClef (gint type, gchar ** clefName)
{
  switch (type)
    {
    case TREBLE:
      *clefName = "treble";
      break;
    case BASS:
      *clefName = "bass";
      break;
    case ALTO:
      *clefName = "alto";
      break;
    case G_8:
      *clefName = "treble-8vb";
      break;
    case TENOR:
      *clefName = "tenor";
      break;
    case SOPRANO:
      *clefName = "soprano";
      break;
    default:
      g_warning ("Unknown clef type %d, using treble", type);
      *clefName = "treble";
      break;
    }
}


/*
 * Determine the duration as it should be written to the XML file.  duration
 * is the Denemo duration, and the XML duration is returned in durationName.
 */
static void
determineDuration (gint duration, gchar ** durationName)
{
  switch (duration)
    {
    case 0:
      *durationName = "whole";
      break;
    case 1:
      *durationName = "half";
      break;
    case 2:
      *durationName = "quarter";
      break;
    case 3:
      *durationName = "eighth";
      break;
    case 4:
      *durationName = "sixteenth";
      break;
    case 5:
      *durationName = "thirty-second";
      break;
    case 6:
      *durationName = "sixty-fourth";
      break;
    default:
      g_warning ("Unknown note duration 1/%d, using quarter", 1 << duration);
      *durationName = "quarter";
      break;
    }
}


/*
 * Output an integer child as a child of the given node.
 */
static xmlNodePtr
newXMLIntChild (xmlNodePtr parent, xmlNsPtr ns, const xmlChar * name,
		gint content)
{
  static gchar sIntBuf[12];	/* enough for -2000000000 + '\0' */
  sprintf (sIntBuf, "%d", content);
  return xmlNewChild (parent, ns, name, sIntBuf);
}


/*
 * Return a newly allocated unique XML ID string which must be freed by the
 * caller when it is no longer needed.
 */
static gchar *
newXMLID ()
{
  /* Allocate enough space for "id2000000000" + '\0'. */

  gchar *result = g_new (char, 13);
  sprintf (result, "id%d", sNextXMLID++);
  return result;
}


/*
 * Return a unique XML ID for the given pointer, and register it in the global
 * structure -> XML ID map.  If it already exists in the map, return its XML
 * ID; otherwise, create a new XML ID and return that.
 */
static gchar *
getXMLID (gpointer ptr)
{
  gchar *xmlID = g_hash_table_lookup (sStructToXMLIDMap, ptr);
  if (xmlID == NULL)
    g_hash_table_insert (sStructToXMLIDMap, ptr, (xmlID = newXMLID ()));
  return xmlID;
}


/*
 * Output a fraction (<numerator>, <denominator>) as a child of the given
 * node.
 */
static void
newXMLFraction (xmlNodePtr parent, xmlNsPtr ns, gint num, gint denom)
{
  newXMLIntChild (parent, ns, "numerator", num);
  newXMLIntChild (parent, ns, "denominator", denom);
}


/*
 * Output a clef of the form:
 *
 *   <clef name="treble"/>
 *
 * as a child of the given node.
 */
static xmlNodePtr
newXMLClef (xmlNodePtr parent, xmlNsPtr ns, gint clef)
{
  gchar *clefName = NULL;
  xmlNodePtr clefElem = NULL;

  determineClef (clef, &clefName);
  clefElem = xmlNewChild (parent, ns, "clef", NULL);
  xmlSetProp (clefElem, "name", clefName);
  return clefElem;
}


/*
 * Output a key signature of the form:
 *
 *   <key-signature>
 *     <modal-key-signature note-name="B" accidental="flat" mode="major"/>
 *   </key-signature>
 *
 * as a child of the given node.
 */
static xmlNodePtr
newXMLKeySignature (xmlNodePtr parent, xmlNsPtr ns, gint keySig,
		    gboolean isMinor)
{
  gchar *noteName = NULL, *accidental = NULL, *mode = NULL;
  xmlNodePtr keySigElem = NULL, modalKeySigElem = NULL;

  determineKeySignature (keySig, isMinor, &noteName, &accidental, &mode);
  keySigElem = xmlNewChild (parent, ns, "key-signature", NULL);
  modalKeySigElem = xmlNewChild (keySigElem, ns, "modal-key-signature", NULL);
  xmlSetProp (modalKeySigElem, "note-name", noteName);
  if (accidental != NULL)
    xmlSetProp (modalKeySigElem, "accidental", accidental);
  xmlSetProp (modalKeySigElem, "mode", mode);

  return keySigElem;
}


/*
 * Output a time signature of the form:
 *
 *   <time-signature>
 *     <simple-time-signature>
 *       <numerator>3</numerator>
 *       <denominator>4</denominator>
 *     </simple-time-signature>
 *   </time-signature>
 *
 * as a child of the given node.
 */
static xmlNodePtr
newXMLTimeSignature (xmlNodePtr parent, xmlNsPtr ns, gint numerator,
		     gint denominator)
{
  xmlNodePtr timeSigElem = xmlNewChild (parent, ns, "time-signature", NULL);
  newXMLFraction (xmlNewChild
		  (timeSigElem, ns, "simple-time-signature", NULL), ns,
		  numerator, denominator);
  return timeSigElem;
}


/*
 * Output a decoration of the form:
 *
 *   <decoration type="staccato"/>
 *
 * as a child of the given node.
 */
static xmlNodePtr
newXMLDecoration (xmlNodePtr parent, xmlNsPtr ns, gchar * decorationName)
{
  xmlNodePtr decorationElem = xmlNewChild (parent, ns, "decoration", NULL);
  xmlSetProp (decorationElem, "type", decorationName);
  return decorationElem;
}


/*
 * Output an accidental of the form:
 *
 *   <accidental name="double-sharp" show="true"/>
 *
 * as a child of the given node.
 */
static xmlNodePtr
newXMLAccidental (xmlNodePtr parent, xmlNsPtr ns, gint enshift, gboolean show)
{
  xmlNodePtr accElem = xmlNewChild (parent, ns, "accidental", NULL);
  gchar *accName = NULL;
  switch (enshift)
    {
    case -2:
      accName = "double-flat";
      break;
    case -1:
      accName = "flat";
      break;
    case 0:
      accName = "natural";
      break;
    case 1:
      accName = "sharp";
      break;
    case 2:
      accName = "double-sharp";
      break;
    default:
      g_warning ("Illegal accidental shift %d: must be between -2 and 2; "
		 "using 0", enshift);
      accName = "natural";
      break;
    }
  xmlSetProp (accElem, "name", accName);
  xmlSetProp (accElem, "show", show ? "true" : "false");
  return accElem;
}


/*
 * Output a stem directive of the form:
 *
 *   <stem-directive type="down"/>
 *
 * as a child of the given node.
 */
static xmlNodePtr
newXMLStemDirective (xmlNodePtr parent, xmlNsPtr ns, enum stemdirections type)
{
  xmlNodePtr stemElem = xmlNewChild (parent, ns, "stem-directive", NULL);
  gchar *stemName = NULL;
  switch (type)
    {
    case STEMDOWN:
      stemName = "down";
      break;
    case STEMBOTH:
      stemName = "auto";
      break;
    case STEMUP:
      stemName = "up";
      break;
    default:
      g_warning ("Unknown stem directive type %d, using auto", type);
      stemName = "auto";
      break;
    }
  xmlSetProp (stemElem, "type", stemName);
  return stemElem;
}


/*
 * Output a notehead element of the form:
 *
 *   <note-head type="diamond"/>
 *
 * as a child of the given node.
 */
static xmlNodePtr
newXMLNoteHead (xmlNodePtr parent, xmlNsPtr ns, enum headtype noteHeadType)
{
  xmlNodePtr noteHeadElem = xmlNewChild (parent, ns, "note-head", NULL);
  gchar *headTypeName = NULL;
  switch (noteHeadType)
    {
    case NORMAL:
      headTypeName = "normal";
      break;
    case CROSS:
      headTypeName = "cross";
      break;
    case HARMONIC:
      headTypeName = "harmonic";
      break;
    case DIAMOND:
      headTypeName = "diamond";
      break;
    default:
      g_warning ("Unknown notehead type %d, using normal", noteHeadType);
      headTypeName = "normal";
      break;
    }
  xmlSetProp (noteHeadElem, "type", headTypeName);
  return noteHeadElem;
}


/*
 * Export the given score (from measure start to measure end) as a "native"
 * Denemo XML file to the given file.
 */
void
exportXML (gchar * thefilename, struct scoreinfo *si, gint start, gint end)
{
  GString *filename = g_string_new (thefilename);
  xmlDocPtr doc;
  xmlNodePtr scoreElem, stavesElem, voicesElem, voiceElem;
  xmlNodePtr measuresElem, measureElem, objElem, prevTieElem;
  xmlNodePtr curElem, parentElem;
  xmlNsPtr ns;
  staffnode *curStaff;
  staff *curStaffStruct;
  gchar *staffXMLID, *voiceXMLID;
  gchar *lastBeamStartXMLID, *chordXMLID, *noteXMLID;
  gchar *lastTupletStartXMLID, *lastGraceStartXMLID;
  /*gchar *clef, *baseKeyName, *accidental;*/
  measurenode *curMeasure;
  gboolean emptyMeasure;
  objnode *curObjNode;
  mudelaobject *curObj;
  GList *curNoteNode;
  note *curNote;
  GList *slurElemStack;
  gint curTime1, curTime2;
  /*gint numerator, denominator;*/
  gchar *durationType;

  /* Append .denemo onto the filename if necessary. */

  if (strcmp (filename->str + filename->len - 7, ".denemo"))
    g_string_append (filename, ".denemo");

  /* Initialize score-wide variables. */

  sStructToXMLIDMap = g_hash_table_new (NULL, NULL);

  /* Create the XML document and output the root element. */

  doc = xmlNewDoc ("1.0");
  xmlSetDocCompressMode (doc, XML_COMPRESSION_RATIO);
  doc->xmlRootNode = scoreElem = xmlNewDocNode (doc, NULL, "score", NULL);
  ns = xmlNewNs (doc->xmlRootNode, DENEMO_XML_NAMESPACE, NULL);
  xmlSetProp (scoreElem, "version", "1.0");
  /* FIXME: Put comment here ("Denemo XML file generated by..."). */

  /* Output the score-info element and its children. */

  parentElem = xmlNewChild (scoreElem, ns, "score-info", NULL);
  curElem = xmlNewChild (parentElem, ns, "tempo", NULL);
  newXMLFraction (xmlNewChild (curElem, ns, "duration", NULL), ns, 1, 4);
  newXMLIntChild (curElem, ns, "bpm", si->tempo);
  xmlNewChild (parentElem, ns, "title", si->title->str);
  xmlNewChild (parentElem, ns, "subtitle", si->subtitle->str);
  xmlNewChild (parentElem, ns, "composer", si->composer->str);

  /* Output each (primary) staff, and store the IDs in a hash table. */

  stavesElem = xmlNewChild (scoreElem, ns, "staves", NULL);
  for (curStaff = si->thescore; curStaff != NULL; curStaff = curStaff->next)
    {
      curStaffStruct = (staff *) curStaff->data;
      if (curStaffStruct->voicenumber != 2)
	{
	  parentElem = xmlNewChild (stavesElem, ns, "staff", NULL);
	  staffXMLID = getXMLID (curStaffStruct);
	  xmlSetProp (parentElem, "id", staffXMLID);
	  curElem = xmlNewChild (parentElem, ns, "staff-info", NULL);
	  newXMLIntChild (curElem, ns, "number-of-lines",
			  curStaffStruct->no_of_lines);
	  newXMLIntChild (curElem, ns, "transpose",
			  curStaffStruct->transposition);
	  xmlNewChild (curElem, ns, (xmlChar *)"instrument",
		       (xmlChar *)curStaffStruct->midi_instrument->str);
	}
    }

  /* Output each voice. */

  voicesElem = xmlNewChild (scoreElem, ns, "voices", NULL);
  for (curStaff = si->thescore; curStaff != NULL; curStaff = curStaff->next)
    {
      curStaffStruct = (staff *) curStaff->data;

      /* Initialize voice-wide variables. */

      prevTieElem = NULL;
      slurElemStack = NULL;
      lastBeamStartXMLID = NULL;
      lastTupletStartXMLID = NULL;
      lastGraceStartXMLID = NULL;

      /*
       * If this is a primary voice, find the ID of its staff, which applies
       * until the next primary voice we run across.
       */

      if (curStaffStruct->voicenumber != 2)
	{
	  staffXMLID = getXMLID (curStaffStruct);
	}

      voiceElem = xmlNewChild (voicesElem, ns, "voice", NULL);
      voiceXMLID = newXMLID ();
      xmlSetProp (voiceElem, "id", voiceXMLID);

      /* Nobody actually needs the voice ID right now, so we throw it away. */

      g_free (voiceXMLID);

      /*
       * Output the voice info (voice name and first measure number, which
       * currently is always 1.
       */

      parentElem = xmlNewChild (voiceElem, ns, "voice-info", NULL);
      xmlNewChild (parentElem, ns, "voice-name",
		   curStaffStruct->denemo_name->str);
      newXMLIntChild (parentElem, ns, "first-measure-number", 1);

      /*
       * Output the initial voice parameters:
       *     - staff on which this voice resides
       *     - clef
       *     - key signature
       *     - time signature
       */

      parentElem = xmlNewChild (voiceElem, ns, "initial-voice-params", NULL);
      curElem = xmlNewChild (parentElem, ns, "staff-ref", NULL);
      xmlSetProp (curElem, "staff", staffXMLID);
      newXMLClef (parentElem, ns, curStaffStruct->sclef);
      newXMLKeySignature (parentElem, ns, curStaffStruct->skey,
			  curStaffStruct->skey_isminor);
      curTime1 = curStaffStruct->stime1;
      curTime2 = curStaffStruct->stime2;
      newXMLTimeSignature (parentElem, ns, curTime1, curTime2);

      /* Write out the measures. */

      measuresElem = xmlNewChild (voiceElem, ns, "measures", NULL);
      for (curMeasure = curStaffStruct->measures;
	   curMeasure != NULL; curMeasure = curMeasure->next)
	{
	  emptyMeasure = TRUE;
	  measureElem = xmlNewChild (measuresElem, ns, "measure", NULL);

	  for (curObjNode = curMeasure->data;
	       curObjNode != NULL; curObjNode = curObjNode->next)
	    {
	      curObj = (mudelaobject *) curObjNode->data;

	      switch (curObj->type)
		{
		case CHORD:
		  emptyMeasure = FALSE;

		  /* If this is the start of a beam, output a <beam-start>. */

		  if (curObj->isstart_beamgroup && !curObj->isend_beamgroup)
		    {
		      curElem = xmlNewChild (measureElem, ns, "beam-start",
					     NULL);
		      lastBeamStartXMLID = newXMLID ();
		      xmlSetProp (curElem, "id", lastBeamStartXMLID);
		    }

		  /* Output the root element, "rest" or "chord". */

		  if (curObj->u.chordval.tones == NULL)
		    objElem = xmlNewChild (measureElem, ns, "rest", NULL);
		  else
		    objElem = xmlNewChild (measureElem, ns, "chord", NULL);
		  chordXMLID = getXMLID (curObj);
		  xmlSetProp (objElem, "id", chordXMLID);

		  /* Output the duration. */

		  determineDuration (curObj->u.chordval.baseduration,
				     &durationType);
		  parentElem = xmlNewChild (objElem, ns, "duration", NULL);
		  xmlSetProp (parentElem, "base", durationType);
		  if (curObj->u.chordval.numdots != 0)
		    newXMLIntChild (parentElem, ns, "dots",
				    curObj->u.chordval.numdots);

		  /* Output all the decorations. */

		  if (curObj->u.chordval.has_stacatto_p
		      || curObj->u.chordval.is_accented_p
		      || curObj->u.chordval.has_fermata_p
		      || curObj->u.chordval.has_tenuto_p
		      || curObj->u.chordval.has_trill_p
		      || curObj->u.chordval.has_turn_p
		      || curObj->u.chordval.has_mordent_p
		      || curObj->u.chordval.has_staccatissimo_p
		      || curObj->u.chordval.has_marcato_p
		      || curObj->u.chordval.has_ubow_p
		      || curObj->u.chordval.has_dbow_p
		      || curObj->u.chordval.has_rheel_p
		      || curObj->u.chordval.has_lheel_p
		      || curObj->u.chordval.has_rtoe_p
		      || curObj->u.chordval.has_ltoe_p)
		    {
		      parentElem = xmlNewChild (objElem, ns, "decorations",
						NULL);
		      if (curObj->u.chordval.has_stacatto_p)
			newXMLDecoration (parentElem, ns, "staccato");
		      if (curObj->u.chordval.is_accented_p)
			newXMLDecoration (parentElem, ns, "accent");
		      if (curObj->u.chordval.has_fermata_p)
			newXMLDecoration (parentElem, ns, "fermata");
		      if (curObj->u.chordval.has_tenuto_p)
			newXMLDecoration (parentElem, ns, "tenuto");
		      if (curObj->u.chordval.has_trill_p)
			newXMLDecoration (parentElem, ns, "trill");
		      if (curObj->u.chordval.has_turn_p)
			newXMLDecoration (parentElem, ns, "turn");
		      if (curObj->u.chordval.has_mordent_p)
			newXMLDecoration (parentElem, ns, "mordent");
		      if (curObj->u.chordval.has_staccatissimo_p)
			newXMLDecoration (parentElem, ns, "staccatissimo");
		      if (curObj->u.chordval.has_marcato_p)
			newXMLDecoration (parentElem, ns, "marcato");
		      if (curObj->u.chordval.has_ubow_p)
			newXMLDecoration (parentElem, ns, "up-bow");
		      if (curObj->u.chordval.has_dbow_p)
			newXMLDecoration (parentElem, ns, "down-bow");
		      if (curObj->u.chordval.has_rheel_p)
			newXMLDecoration (parentElem, ns, "right-heel");
		      if (curObj->u.chordval.has_lheel_p)
			newXMLDecoration (parentElem, ns, "left-heel");
		      if (curObj->u.chordval.has_rtoe_p)
			newXMLDecoration (parentElem, ns, "right-toe");
		      if (curObj->u.chordval.has_ltoe_p)
			newXMLDecoration (parentElem, ns, "left-toe");
		    }
		   /*
		    *  Output Dynamic which is now part of note
		    *
		    */
		    if(curObj->u.chordval.dynamics)
		      {
		        GString *string =
		         (GString *)curObj->u.chordval.dynamics->data;
		        parentElem = xmlNewChild (objElem, ns,
		                                  (xmlChar *)"dynamic", NULL);
		        xmlSetProp (parentElem, (xmlChar *)"name",
		                    (xmlChar *)string->str);
		      }
				    
		  /*
		   * If this is the end of a slur, terminate the previous
		   * <slur> element.
		   */

		  if (curObj->u.chordval.slur_end_p)
		    {
		      if (slurElemStack == NULL)
			{
			  g_warning ("Encountered slur end without a "
				     "beginning");
			}
		      else
			{
			  xmlSetProp ((xmlNodePtr) slurElemStack->data, "to",
				      chordXMLID);

			  /* Pop the top element off the stack. */

			  slurElemStack = g_list_remove (slurElemStack,
							 slurElemStack->data);
			}
		    }

		  /*
		   * Output a <slur> element (to be filled in by the end of
		   * the slur) if there's a slur beginning on this chord.
		   */

		  if (curObj->u.chordval.slur_begin_p)
		    {
		      parentElem = xmlNewChild (objElem, ns, "slurs", NULL);
		      curElem = xmlNewChild (parentElem, ns, "slur", NULL);

		      /* Push the <slur> element onto the slur stack. */

		      slurElemStack = g_list_prepend (slurElemStack, curElem);
		    }

		  /*
		   * If the previous chord was tied, fill in the <tie>
		   * element to point to this one.
		   */

		  if (prevTieElem != NULL)
		    {
		      xmlSetProp (prevTieElem, "to", chordXMLID);
		      prevTieElem = NULL;
		    }

		  /* Output a <tie> element if this chord is tied. */

		  if (curObj->u.chordval.is_tied)
		    {
		      prevTieElem = xmlNewChild (objElem, ns, "tie", NULL);
		    }

		  /* Output all the notes, if this isn't a rest. */

		  if (curObj->u.chordval.tones != NULL)
		    {
		      parentElem = xmlNewChild (objElem, ns, "notes", NULL);
		      for (curNoteNode = curObj->u.chordval.tones;
			   curNoteNode != NULL;
			   curNoteNode = curNoteNode->next)
			{
			  curNote = (note *) curNoteNode->data;
			  curElem =
			    xmlNewChild (parentElem, ns, "note", NULL);
			  noteXMLID = getXMLID (curNote);
			  xmlSetProp (curElem, "id", noteXMLID);
			  newXMLIntChild (curElem, ns, "middle-c-offset",
					  curNote->mid_c_offset);
			  if (curNote->enshift != 0
			      || curNote->showaccidental)
			    {
			      newXMLAccidental (curElem, ns, curNote->enshift,
						curNote->showaccidental);
			    }
			  if (curNote->noteheadtype != NORMAL)
			    {
			      newXMLNoteHead (curElem, ns,
					      curNote->noteheadtype);
			    }
			}
		    }

		  /* If this is the end of a beam, output a <beam-end>. */

		  if (curObj->isend_beamgroup && !curObj->isstart_beamgroup)
		    {
		      curElem = xmlNewChild (measureElem, ns, "beam-end",
					     NULL);
		      if (lastBeamStartXMLID == NULL)
			{
			  g_warning
			    ("Encountered the end of a beam without a "
			     "beginning");
			}
		      else
			{
			  xmlSetProp (curElem, "beam", lastBeamStartXMLID);
			  g_free (lastBeamStartXMLID);
			  lastBeamStartXMLID = NULL;
			}
		    }

		  break;

		case TUPOPEN:
		  objElem = xmlNewChild (measureElem, ns, "tuplet-start",
					 NULL);
		  /*
		   * FIXME: This code does not yet handle nested tuplets.  For
		   *        that, we'd need a stack of "tuplet-start" IDs
		   *        instead of just a single "last ID."
		   */

		  lastTupletStartXMLID = getXMLID (curObj);
		  xmlSetProp (objElem, "id", lastTupletStartXMLID);
		  newXMLFraction (xmlNewChild (objElem, ns, "multiplier",
					       NULL),
				  ns, curObj->u.tupval.numerator,
				  curObj->u.tupval.denominator);
		  break;

		case TUPCLOSE:
		  objElem = xmlNewChild (measureElem, ns, "tuplet-end", NULL);
		  if (lastTupletStartXMLID == NULL)
		    {
		      g_warning ("Encountered nested tuplets or tuplet end "
				 "without start");
		    }
		  else
		    {
		      xmlSetProp (objElem, "tuplet", lastTupletStartXMLID);
		      lastTupletStartXMLID = NULL;
		    }
		  break;

		case CLEF:
		  objElem = newXMLClef (measureElem, ns,
					curObj->u.clefval.type);
		  break;

		case TIMESIG:
		  objElem = newXMLTimeSignature (measureElem, ns,
						 curObj->u.timeval.time1,
						 curObj->u.timeval.time2);
		  break;

		case KEYSIG:
		  objElem = newXMLKeySignature (measureElem, ns,
						curObj->u.keyval.number,
						curObj->u.keyval.isminor);
		  break;

		case STEMDIRECTIVE:
		  objElem = newXMLStemDirective (measureElem, ns,
						 curObj->u.stemval.type);
		  break;

		case DYNAMIC:
		  /*objElem = xmlNewChild (measureElem, ns, "dynamic", NULL);
		  xmlSetProp (objElem, "name", curObj->u.dynval.type->str);*/
		  break;

		case GRACE_START:
		  objElem =
		    xmlNewChild (measureElem, ns, "grace-start", NULL);

		  /*
		   * FIXME: This code does not yet handle nested grace note
		   *        passages.  For that, we'd need a stack of
		   *        "grace-start" IDs instead of just a single "last
		   *        ID."
		   */

		  lastGraceStartXMLID = getXMLID (curObj);
		  xmlSetProp (objElem, "id", lastGraceStartXMLID);
		  xmlSetProp (objElem, "on-beat",
			      curObj->u.graceval.on_beat ? "true" : "false");
		  /*
		   * FIXME: What's curObj->u.graceval.duration for?  It
		   *        doesn't seem to be used.
		   */
		  break;

		case GRACE_END:
		  objElem = xmlNewChild (measureElem, ns, "grace-end", NULL);
		  if (lastGraceStartXMLID == NULL)
		    {
		      g_warning ("Encountered nested grace note passages or "
				 "grace note end without start");
		    }
		  else
		    {
		      xmlSetProp (objElem, "grace", lastGraceStartXMLID);
		      lastGraceStartXMLID = NULL;
		    }
		  break;

		case BARLINE:
		case MEASUREBREAK:
		case COMMENT:	/* FIXME: Put this in Lily handler */
		case LILYDIRECTIVE:	/* FIXME: Put this in Lily handler */
		  g_warning ("Cannot yet handle mudelaobjects of type %d",
			     curObj->type);
		  break;

		default:
		  /* FIXME: First try handlers. */
		  g_warning ("Got a mudelaobject of unknown type %d",
			     curObj->type);
		  break;
		}		/* end switch on object type */
	    }			/* end for each object in measure */

	  if (emptyMeasure)
	    {
	      curElem = xmlNewChild (measureElem, ns, "rest", NULL);
	      xmlSetProp (curElem, "show", "false");
	      newXMLFraction (xmlNewChild (curElem, ns, "duration", NULL),
			      ns, curTime1, curTime2);
	    }
	}			/* end for each measure in voice */

      /* Clean up voice-specific variables. */

      g_list_free (slurElemStack);
      g_free (lastBeamStartXMLID);
    }				/* end for each voice in score */

  /* Save the file. */

  if (xmlSaveFile (filename->str, doc) < 0)
    g_warning ("Could not save file %s", filename->str);

  /* Clean up all the memory we've allocated. */

  xmlFreeDoc (doc);
  g_hash_table_foreach (sStructToXMLIDMap, freeHashTableValue, NULL);
  g_hash_table_destroy (sStructToXMLIDMap);
  sNextXMLID = 0;

  g_string_free (filename, FALSE);
}
