/* DCTC - a Direct Connect text clone for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * userinfo.c: Copyright (C) Eric Prevoteau <www@a2pb.gotdns.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: userinfo.c,v 1.4 2003/12/28 08:12:38 uid68112 Exp $
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <glib.h>

#include "userinfo.h"
#include "lmp.h"
#include "misc.h"

static LMP_ENTRY *local_dctc_user_info_lmp=NULL;       /* LMP for the user information file */
																		/* NOTE: the first record (idx=0) of the file is always its header */

/****************************************************************************************************************/
/* these functions handles management of user informations stored in the file $HOME/.dctc/running/xxxx-userinfo */
/****************************************************************************************************************/

/************************************/
/* create the userinfo LMP and file */
/************************************/
void uinfo_create(const char *filename)
{
	local_dctc_user_info_lmp=lmp_new(filename,sizeof(LMP_UINFO),-1);
}

/**************************/
/* close the userinfo LMP */
/**************************/
void uinfo_delete(void)
{
	if(local_dctc_user_info_lmp)
	{
		lmp_close(local_dctc_user_info_lmp);
		local_dctc_user_info_lmp=NULL;
	}
}

/**********************************************/
/* find the array index of the given nickname */
/**********************************************/
/* output: idx or -1 if not found */
/**********************************/
static int uinfo_idx(const char *nickname)
{
	int i;
	LMP_UINFO *lu=local_dctc_user_info_lmp->mapped_addr;

	for(i=1;i<local_dctc_user_info_lmp->nb_records;i++)
	{
		if(lu[i].entry_is_busy)
		{
			if(!strcmp(lu[i].nickname,nickname))
				return i;
		}
	}
	return -1;
}

/**********************************/
/* find a free entry in the array */
/**********************************/
/* output: idx or -1 if not found */
/**********************************/
static int uinfo_free_idx(void)
{
	int i;
	LMP_UINFO *lu=local_dctc_user_info_lmp->mapped_addr;

	for(i=1;i<local_dctc_user_info_lmp->nb_records;i++)
	{
		if(lu[i].entry_is_busy==0)
		{
			return i;
		}
	}
	return -1;
}

/*******************************************************/
/* add or update the information of the given nickname */
/****************************************************************/
/* mandatory: nickname                                          */
/* optional: usize, utype, uemail, udesc (NULL means unchanged) */
/*           uflag (0 means unchanged)                          */
/*           is_op, is_online (-1 means unchanged)              */
/****************************************************************/
void uinfo_update_user_info(const char *nickname, const char *usize, const char *utype, const unsigned char uflag, const char *uemail, const char *udesc, const int is_op, const int is_online)
{
	int idx;
	gboolean full_info=FALSE;

	if(local_dctc_user_info_lmp==NULL)
		return;

	if(lmp_lock_and_map(local_dctc_user_info_lmp))
		return;

	idx=uinfo_idx(nickname);
	if(idx==-1)
	{
		idx=uinfo_free_idx();
	}

	if(idx!=-1)
	{
		/* 1) the entry already exists, just update it */
		/* 2) the entry does not exist but there is free space to add it */
		LMP_UINFO *udp=&(((LMP_UINFO*)(local_dctc_user_info_lmp->mapped_addr))[idx]);
		
		udp->entry_is_busy|=UINFO_BIT_BUSY;											/* we set it to 1 because it can be an add or an update */

		if(is_op!=-1)
		{
			if(is_op)
				udp->entry_is_busy=SET_BIT(udp->entry_is_busy,UINFO_BIT_IS_OP);
			else
				udp->entry_is_busy=CLEAR_BIT(udp->entry_is_busy,UINFO_BIT_IS_OP);
		}
		if(is_online!=-1)
		{
			if(is_online)
				udp->entry_is_busy=SET_BIT(udp->entry_is_busy,UINFO_BIT_IS_ONLINE);
			else
				udp->entry_is_busy=CLEAR_BIT(udp->entry_is_busy,UINFO_BIT_IS_ONLINE);
		}

		strncpy_max(udp->nickname,nickname,UINFO_NICK_SIZE);

		if(usize)
		{
			if(strlen(usize))
				sscanf(usize,"%lld",&(udp->share_size));
			else
				udp->share_size=0;
			full_info=TRUE;
		}

		if(utype)
		{
			strncpy_max(udp->cnx_type,utype,UINFO_CNX_SIZE);
			full_info=TRUE;
		}

		if(uflag)
		{
			udp->uflag=uflag;
		}

		if(uemail)
		{
			strncpy_max(udp->uemail,uemail,UINFO_UEMAIL_SIZE);
			full_info=TRUE;
		}

		if(uemail)
		{
			strncpy_max(udp->udesc,udesc,UINFO_UDESC_SIZE);
			full_info=TRUE;
		}

		if(full_info)
			udp->entry_is_busy=SET_BIT(udp->entry_is_busy,UINFO_BIT_POPULATED);
	}
	else
	{
		/* there is no free space, we must append the new record to the file */
		LMP_UINFO upd;
		upd.entry_is_busy=UINFO_BIT_BUSY;
		if(is_op!=-1)
		{
			if(is_op)
				upd.entry_is_busy=SET_BIT(upd.entry_is_busy,UINFO_BIT_IS_OP);
			else
				upd.entry_is_busy=CLEAR_BIT(upd.entry_is_busy,UINFO_BIT_IS_OP);
		}
		if(is_online!=-1)
		{
			if(is_online)
				upd.entry_is_busy=SET_BIT(upd.entry_is_busy,UINFO_BIT_IS_ONLINE);
			else
				upd.entry_is_busy=CLEAR_BIT(upd.entry_is_busy,UINFO_BIT_IS_ONLINE);
		}

		strncpy_max(upd.nickname,nickname,UINFO_NICK_SIZE);
		if((usize)&&(strlen(usize)))
		{
			sscanf(usize,"%lld",&(upd.share_size));
			full_info=TRUE;
		}
		else
			upd.share_size=0;

		if(utype)
		{
			strncpy_max(upd.cnx_type,utype,UINFO_CNX_SIZE);
			full_info=TRUE;
		}
		else
			upd.cnx_type[0]='\0';

		if(uflag)
			upd.uflag=uflag;
		else
			upd.uflag=1;

		if(uemail)
		{
			strncpy_max(upd.uemail,uemail,UINFO_UEMAIL_SIZE);
			full_info=TRUE;
		}
		else
			upd.uemail[0]='\0';

		if(udesc)
		{
			strncpy_max(upd.udesc,udesc,UINFO_UDESC_SIZE);
			full_info=TRUE;
		}
		else
			upd.udesc[0]='\0';

		if(full_info)
			upd.entry_is_busy=SET_BIT(upd.entry_is_busy,UINFO_BIT_POPULATED);

		lmp_append_record(local_dctc_user_info_lmp,&upd);
	}
	lmp_unmap_and_unlock(local_dctc_user_info_lmp);
}

/******************************************************************************/
/* reset a user entry to its default value (erasing an existing one if exists */
/******************************************************************************/
void uinfo_reset_user_info(const char *nickname, gboolean is_op, gboolean is_online)
{
	int idx;

	if(local_dctc_user_info_lmp==NULL)
		return;

	if(lmp_lock_and_map(local_dctc_user_info_lmp))
		return;

	idx=uinfo_idx(nickname);
	if(idx==-1)
	{
		idx=uinfo_free_idx();
	}

	if(idx!=-1)
	{
		/* 1) the entry already exists, just update it */
		/* 2) the entry does not exist but there is free space to add it */
		LMP_UINFO *udp=&(((LMP_UINFO*)(local_dctc_user_info_lmp->mapped_addr))[idx]);
		udp->entry_is_busy=1;											/* we set it to 1 because it can be an add or an update */
		if(is_op)
			udp->entry_is_busy=SET_BIT(udp->entry_is_busy,UINFO_BIT_IS_OP);
		if(is_online)
			udp->entry_is_busy=SET_BIT(udp->entry_is_busy,UINFO_BIT_IS_ONLINE);
		strncpy_max(udp->nickname,nickname,UINFO_NICK_SIZE);
		udp->share_size=0;
		udp->cnx_type[0]='\0';
		udp->uflag=0;
		udp->uemail[0]='\0';
		udp->udesc[0]='\0';
	}
	else
	{
		/* there is no free space, we must append the new record to the file */
		LMP_UINFO upd;
		upd.entry_is_busy=1;
		if(is_op)
			upd.entry_is_busy=SET_BIT(upd.entry_is_busy,UINFO_BIT_IS_OP);
		if(is_online)
			upd.entry_is_busy=SET_BIT(upd.entry_is_busy,UINFO_BIT_IS_ONLINE);
		strncpy_max(upd.nickname,nickname,UINFO_NICK_SIZE);
		upd.share_size=0;
		upd.cnx_type[0]='\0';
		upd.uflag=0;
		upd.uemail[0]='\0';
		upd.udesc[0]='\0';

		lmp_append_record(local_dctc_user_info_lmp,&upd);
	}
	lmp_unmap_and_unlock(local_dctc_user_info_lmp);
}

/************************************************/
/* delete the information of the given nickname */
/************************************************/
void uinfo_delete_user_info(const char *nickname)
{
	int idx;

	if(local_dctc_user_info_lmp==NULL)
		return;

	if(lmp_lock_and_map(local_dctc_user_info_lmp))
		return;

	idx=uinfo_idx(nickname);
	if(idx!=-1)			/* idx should never be == -1 here except if the userinfo file was deleted */
	{
		((LMP_UINFO *)(local_dctc_user_info_lmp->mapped_addr))[idx].entry_is_busy=0;
	}
	lmp_unmap_and_unlock(local_dctc_user_info_lmp);
}

/***************************************/
/* delete the information of all users */
/***************************************/
void uinfo_delete_users(void)
{
	int idx;

	if(local_dctc_user_info_lmp==NULL)
		return;

	if(lmp_lock_and_map(local_dctc_user_info_lmp))
		return;

	for(idx=1;idx<local_dctc_user_info_lmp->nb_records;idx++)
	{
		((LMP_UINFO *)(local_dctc_user_info_lmp->mapped_addr))[idx].entry_is_busy=0;
	}
	lmp_unmap_and_unlock(local_dctc_user_info_lmp);
}

