/*
 * generate_map.c
 *
 * crafted - a pud editor for the freecraft project.
 *
 * Copyright (C) 2001-2002 DindinX <David@dindinx.org>
 * Copyright (C) 2001-2002 Jan Uerpmann <j.uerpmann@tu-bs.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <sys/time.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <gtk/gtk.h>
#include <math.h>

#include "crafted.h"
#include "craftedintl.h"
#include "tileset.h"
#include "tile.h"
#include "mini_map.h"

// Different tile types

#define TYPE_LIGHT_WATER  0x10
#define TYPE_DARK_WATER    0x20
#define TYPE_LIGHT_MUD    0x30
#define TYPE_DARK_MUD    0x40
#define TYPE_LIGHT_GRASS  0x50
#define TYPE_DARK_GRASS    0x60
#define TYPE_TREE      0x70
#define TYPE_ROCK      0x80

struct Adjust
{
  GtkAdjustment  *DeepWater;
  GtkAdjustment  *Water;
  GtkAdjustment  *Grass;
  GtkAdjustment  *Tree;
};
struct Adjust  gAdjust;

// number of different tiles per type

char gTypeVar[9] =
{
  0,    // unused
  4,    // light water
  4,    // dark water
  3,    // light mud
  3,    // dark mud
  3,    // light grass
  3,    // dark grass
  3,    // tree
  4    // rock
};

int gXDim;
int gYDim;

gdouble  gNumMountain;
gdouble  gNumLake;
gdouble  gNumRiver;

//            default    islands  mountain
gdouble  gDeepWaterLimit  = 102.0;  // 102  100
gdouble  gWaterLimit    = 122.0;  // 126  110
gdouble  gGrassLimit    = 128.0;  // 128  126
gdouble  gTreeLimit    = 134.0;  // 255  132

gboolean  gGenerateIsland = FALSE;


extern tPud Pud;

// from generate_map2.c
void double_adjustment_update(GtkAdjustment *adjustment, gpointer data);

void InitRandom(void)
{
  struct timeval tv;

  gettimeofday(&tv, NULL);
  srand(tv.tv_usec);
}

int Random(int max)
{
  return (int) (((double) max)*rand()/(RAND_MAX+1.0));
}

void Blur(unsigned char *height, int x, int y)
{
  int sum=0, i, j;

  for (i=-2; i<3; i++)
    for (j=-2; j<3; j++)
      sum += height[((x+i+gXDim)%gXDim) + gXDim*((y+j+gYDim)%gYDim)];
  height[x + gXDim*y] = sum/25;
}

void CreateMap(unsigned char *height, unsigned char *map)
{
  int x, y;
  
  for (x=0; x<gXDim; x++)
  {
    for (y=0; y<gYDim; y++)
    {
      map[x + gXDim*y] = TYPE_ROCK;
      if (height[x + gXDim*y] < gTreeLimit)    map[x + gXDim*y] = TYPE_TREE;
      if (height[x + gXDim*y] < gGrassLimit)    map[x + gXDim*y] = TYPE_DARK_GRASS;
      if (height[x + gXDim*y] < gWaterLimit)    map[x + gXDim*y] = TYPE_LIGHT_WATER;
      if (height[x + gXDim*y] < gDeepWaterLimit)  map[x + gXDim*y] = TYPE_DARK_WATER;
    }
  }
}

void CreateRiver(unsigned char *height)
{
  int x, y, i, j, xflow, yflow, boundary=0;

  x = Random(gXDim);
  y = Random(gYDim);
  while (height[x + gXDim*y] > 0 && height[x + gXDim*y]<0xFF && boundary==0)
  {
    xflow = x;
    yflow = y;
    for (i=-1; i<=1; i++)
    {
      for (j=-1; j<=1; j++)
      {
        if ((0<=x+i) && (x+i<gXDim) &&
            (0<=y+i) && (y+i<gYDim))
        {
          if (height[x+i + gXDim*(y+j)] < height[x + gXDim*y]+0)
          {
            xflow = x+i;
            yflow = y+j;
          }
        }
      }
    }
    height[x + gXDim*y]=0xFF;  // This is after blur, revert to 0x00 afterwards
    x = xflow;
    y = yflow;
    if (x<0 || y<0 || x>=gXDim || y>=gYDim) boundary=1; 
  }
  for (x=0; x<gXDim; x++)
    for (y=0; y<gYDim; y++)
      if (height[x + gXDim*y]==0xFF) height[x + gXDim*y]=0x00;
}

void Intensify(unsigned char *height)
{
  int x, y;

  for (x=0; x<gXDim; x++)
  {
    for (y=0; y<gYDim; y++)
    {
      if (height[x + gXDim*y] < gWaterLimit)
        height[x + gXDim*y]=0;
      if (height[x + gXDim*y] > gTreeLimit)
        height[x + gXDim*y]=0xFF;
    }
  }
}

static void TerraForm(GtkObject *button, GtkObject *dialog)
{
  unsigned char map[gXDim * gYDim];
  unsigned char height[gXDim * gYDim];
  int        x, y, i;
  
//  gtk_object_destroy(dialog);
  progressbar_init();
  progressbar_update(0, "Terraforming");
  InitRandom();
  for (x=0; x<gXDim; x++)
    for (y=0; y<gYDim; y++)
      height[x + gXDim*y] = 0x80 + Random(40) - 18;
  progressbar_update(2.0/100.0, "Creating Mountains");
  for (i=0; i<gNumMountain; i++)
  {
    x = Random(gXDim);
    y = Random(gYDim);
    height[x + gXDim*y] = 255;  // mountain tops
  }
  progressbar_update(5.0/100.0, "Creating Lakes");
  for (i=0; i<gNumLake; i++)
  {
    x = Random(gXDim);
    y = Random(gYDim);
    height[x + gXDim*y] = 0;    // lakes
  }
  if (gGenerateIsland)
  {
    progressbar_update(7.0/100.0, "Creating Ocean");
    for (x=0; x<gXDim; x++)
    {
      height[x + gXDim*0] = 0;      // ocean
      if (Random(10)<5) height[x + gXDim*3] = 0;
      height[x + gXDim*(gYDim-1)] = 0;      // ocean
      if (Random(10)<5) height[x + gXDim*(gYDim-4)] = 0;
    }
    for (y=0; y<gXDim; y++)
    {
      height[0 + gXDim*y] = 0;      // ocean
      if (Random(10)<5) height[3 + gXDim*y] = 0;
      height[gXDim-1 + gXDim*y] = 0;      // ocean
      if (Random(10)<5) height[gXDim-4 + gXDim*y] = 0;
    }
  }
  progressbar_update(10.0/100.0, "Eroding");
  for (x=0; x<gXDim; x++)
    for (y=0; y<gYDim; y++)
      Blur(height, x, y);    // Blur
  for (i=0; i<gNumRiver; i++)  
  {
    if ((i%10)==0)
      progressbar_update((12.0 + i*100.0/gNumRiver/4.0)/100.0, "Creating Rivers");
    CreateRiver(height);
  }
  progressbar_update(37.0/100.0, "Intensifying");
  Intensify(height);
  progressbar_update(40.0/100.0, "Eroding");
  for (x=0; x<gXDim; x++)
    for (y=0; y<gYDim; y++)
      Blur(height, x, y);    // Blur
  for (i=0; i<gNumRiver; i++)  
  {
    if ((i%10)==0)
      progressbar_update((42.0 + i*100.0/gNumRiver/4.0)/100.0, "Creating Rivers");
    CreateRiver(height);
  }
  progressbar_update(67.0/100.0, "Eroding");
  for (x=0; x<gXDim; x++)
    for (y=0; y<gYDim; y++)
      Blur(height, x, y);    // Blur
  progressbar_update(70.0/100.0, "Creating Map");
  CreateMap(height, map);
  for (x=0; x<gXDim; x++)
  {
    if ((x%100)==0)
      progressbar_update((75.0 + x*100.0/gXDim/4.0)/100.0, "Drawing Map");
    for (y=0; y<gYDim; y++)
    {
      affect_exact_tile(x , y, map[x + y*gXDim]
          + Random(gTypeVar[map[x + y*gXDim]>>4]));
      check_around(x, y, map[x + y*gXDim]>>4);
    }
  }
  progressbar_update(100.0/100.0, "Done");
  progressbar_finish();
  update_minimap();
}

void toggle_button(GtkWidget *widget, gpointer data)
{
  gboolean  *flag;
  
  flag = (gboolean*) data;
  *flag = !(*flag);
}

void DefaultSettings(GtkObject *button)
{
  gDeepWaterLimit  = 102.0;
  gWaterLimit    = 122.0;
  gGrassLimit    = 128.0;
  gTreeLimit    = 134.0;
  gtk_adjustment_set_value(GTK_ADJUSTMENT(gAdjust.DeepWater),  gDeepWaterLimit);
  gtk_adjustment_set_value(GTK_ADJUSTMENT(gAdjust.Water),    gWaterLimit);
  gtk_adjustment_set_value(GTK_ADJUSTMENT(gAdjust.Grass),    gGrassLimit);
  gtk_adjustment_set_value(GTK_ADJUSTMENT(gAdjust.Tree),    gTreeLimit);
}

void IslandSettings(GtkObject *button)
{
  gDeepWaterLimit  = 102.0;
  gWaterLimit    = 126.0;
  gGrassLimit    = 128.0;
  gTreeLimit    = 255.0;
  gtk_adjustment_set_value(GTK_ADJUSTMENT(gAdjust.DeepWater),  gDeepWaterLimit);
  gtk_adjustment_set_value(GTK_ADJUSTMENT(gAdjust.Water),    gWaterLimit);
  gtk_adjustment_set_value(GTK_ADJUSTMENT(gAdjust.Grass),    gGrassLimit);
  gtk_adjustment_set_value(GTK_ADJUSTMENT(gAdjust.Tree),    gTreeLimit);
}

void MountainSettings(GtkObject *button)
{
  gDeepWaterLimit  = 100.0;
  gWaterLimit    = 110.0;
  gGrassLimit    = 126.0;
  gTreeLimit    = 132.0;
  gtk_adjustment_set_value(GTK_ADJUSTMENT(gAdjust.DeepWater),  gDeepWaterLimit);
  gtk_adjustment_set_value(GTK_ADJUSTMENT(gAdjust.Water),    gWaterLimit);
  gtk_adjustment_set_value(GTK_ADJUSTMENT(gAdjust.Grass),    gGrassLimit);
  gtk_adjustment_set_value(GTK_ADJUSTMENT(gAdjust.Tree),    gTreeLimit);
}

static void CloseButton(GtkObject *button, GtkObject *dialog)
{
  // when calling gtk_widget_destroy() instead only the button is removed...
  gtk_object_destroy(dialog);
}

void MapGenerationDialog(void)
{
  GtkDialog    *dialog;
  GtkWidget    *hbbox, *button, *vbox, *label, *scale, *vbox2, *hbox, *frame;
  GtkAdjustment  *adj;
  GtkTooltips    *tooltips;
  
  dialog = GTK_DIALOG(gtk_dialog_new());
  gtk_window_set_title(GTK_WINDOW(dialog), "Map Generation Settings");
  gtk_window_set_wmclass(GTK_WINDOW(dialog), "Crafted", "Crafted");
  gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_MOUSE);
  gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, FALSE, TRUE);

  tooltips = gtk_tooltips_new();

  /* Action area (the buttons at the bottom) */
  gtk_container_set_border_width(GTK_CONTAINER(dialog->action_area), 10);
  gtk_box_set_homogeneous(GTK_BOX(dialog->action_area), FALSE);

  hbbox = gtk_hbutton_box_new();
  gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbbox), 10);
  gtk_box_pack_end(GTK_BOX(dialog->action_area), hbbox, FALSE, FALSE, 0);

  /* the "Apply" button */  
  button = gtk_button_new_with_label(_("Apply"));
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     (GtkSignalFunc)TerraForm,
                     GTK_OBJECT(dialog));
  GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
  gtk_box_pack_start(GTK_BOX(hbbox), button, FALSE, FALSE, 0);
  gtk_widget_grab_default(button);

  /* the "Close" button */
  button = gtk_button_new_with_label(_("Close"));
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
            (GtkSignalFunc)CloseButton,
            GTK_OBJECT(dialog));
  GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
  gtk_box_pack_start(GTK_BOX(hbbox), button, FALSE, FALSE, 0);
  gtk_widget_grab_default(button);

  /* The main part */
  hbox = gtk_hbox_new(FALSE, 10);
  gtk_box_pack_start_defaults(GTK_BOX(dialog->vbox), hbox);
  vbox = gtk_vbox_new(FALSE, 4);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), vbox);

  /* left side first */

  /* frame for init settings */
  
  frame = gtk_frame_new(_("Init Settings"));
  gtk_widget_ref(frame);
  gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 0);
  vbox2 = gtk_vbox_new(FALSE, 4);
  gtk_container_add(GTK_CONTAINER(frame), vbox2);

  /* number of initial mountain peaks */
  label = gtk_label_new(_("number of mountains"));
  gtk_box_pack_start_defaults(GTK_BOX(vbox2), label);
  adj = (GtkAdjustment *)gtk_adjustment_new(gNumMountain,
                        0.0, 2010.0, 10.0, 10.0, 10.0);
  scale = gtk_hscale_new(adj);
  gtk_scale_set_digits(GTK_SCALE(scale), 0);
  gtk_box_pack_start_defaults(GTK_BOX(vbox2), scale);
  gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
                     (GtkSignalFunc)double_adjustment_update,
                     &gNumMountain);
  gtk_tooltips_set_tip(tooltips, scale, _("Specify the number of mountains placed on the plain map"), NULL);

  /* number of initial lake holes */
  label = gtk_label_new(_("number of lakes"));
  gtk_box_pack_start_defaults(GTK_BOX(vbox2), label);
  adj = (GtkAdjustment *)gtk_adjustment_new(gNumLake,
                        0.0, 2010.0, 10.0, 10.0, 10.0);
  scale = gtk_hscale_new(adj);
  gtk_scale_set_digits(GTK_SCALE(scale), 0);
  gtk_box_pack_start_defaults(GTK_BOX(vbox2), scale);
  gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
                     (GtkSignalFunc)double_adjustment_update,
                     &gNumLake);
  gtk_tooltips_set_tip(tooltips, scale, _("Specify the number of lakes placed on the plain map"), NULL);
  
  /* number of rivers to place */
  label = gtk_label_new(_("number of rivers"));
  gtk_box_pack_start_defaults(GTK_BOX(vbox2), label);
  adj = (GtkAdjustment *)gtk_adjustment_new(gNumRiver,
                        0.0, 201.0, 1.0, 1.0, 1.0);
  scale = gtk_hscale_new(adj);
  gtk_scale_set_digits(GTK_SCALE(scale), 0);
  gtk_box_pack_start_defaults(GTK_BOX(vbox2), scale);
  gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
                     (GtkSignalFunc)double_adjustment_update,
                     &gNumRiver);
  gtk_tooltips_set_tip(tooltips, scale, _("Specify the number of rivers placed on the plain map (depending on the threshold settings rivers will most probably not show up as rivers on the map)"), NULL);


  /* frame for pre-defined settings */
  
  frame = gtk_frame_new(_("Pre-defined Settings"));
  gtk_widget_ref(frame);
  gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 0);
  vbox2 = gtk_vbox_new(FALSE, 4);
  gtk_container_add(GTK_CONTAINER(frame), vbox2);

  /* the default settings button */  
  button = gtk_button_new_with_label(_("Default Settings"));
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     (GtkSignalFunc)DefaultSettings,
                     NULL);
  gtk_box_pack_start(GTK_BOX(vbox2), button, FALSE, FALSE, 0);
  gtk_tooltips_set_tip(tooltips, button, _("Set threshold values to default"), NULL);

  /* the default settings button */  
  button = gtk_button_new_with_label(_("Islands"));
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     (GtkSignalFunc)IslandSettings,
                     NULL);
  gtk_box_pack_start(GTK_BOX(vbox2), button, FALSE, FALSE, 0);
  gtk_tooltips_set_tip(tooltips, button, _("Set threshold values to island settings"), NULL);

  /* the default settings button */  
  button = gtk_button_new_with_label(_("Mountains"));
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     (GtkSignalFunc)MountainSettings,
                     NULL);
  gtk_box_pack_start(GTK_BOX(vbox2), button, FALSE, FALSE, 0);
  gtk_tooltips_set_tip(tooltips, button, _("Set threshold values to mountain settings"), NULL);

  /* Generate Island check box */
  button = gtk_check_button_new_with_label(_("Generate Island"));
  gtk_box_pack_start_defaults(GTK_BOX(vbox), button);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     (GtkSignalFunc) toggle_button,
                     &gGenerateIsland);
  gtk_tooltips_set_tip(tooltips, button, _("Set to true to create one big island"), NULL);

  /* right side */

  vbox = gtk_vbox_new(FALSE, 4);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), vbox);

  /* frame for threshold settings */
  
  frame = gtk_frame_new(_("Threshold Settings"));
  gtk_widget_ref(frame);
  gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 0);
  vbox2 = gtk_vbox_new(FALSE, 4);
  gtk_container_add(GTK_CONTAINER(frame), vbox2);

  /* deep water height level */
  label = gtk_label_new(_("below value is deep water"));
  gtk_box_pack_start_defaults(GTK_BOX(vbox2), label);
  gAdjust.DeepWater = (GtkAdjustment *)gtk_adjustment_new(gDeepWaterLimit,
                        0.0, 256.0, 1.0, 1.0, 1.0);
  scale = gtk_hscale_new(gAdjust.DeepWater);
  gtk_scale_set_digits(GTK_SCALE(scale), 0);
  gtk_box_pack_start_defaults(GTK_BOX(vbox2), scale);
  gtk_signal_connect(GTK_OBJECT(gAdjust.DeepWater), "value_changed",
                     (GtkSignalFunc)double_adjustment_update,
                     &gDeepWaterLimit);
  gtk_tooltips_set_tip(tooltips, scale, _("Specify threshold value for internal height table"), NULL);

  /* water height level */
  label = gtk_label_new(_("below value is water"));
  gtk_box_pack_start_defaults(GTK_BOX(vbox2), label);
  gAdjust.Water = (GtkAdjustment *)gtk_adjustment_new(gWaterLimit,
                        0.0, 256.0, 1.0, 1.0, 1.0);
  scale = gtk_hscale_new(gAdjust.Water);
  gtk_scale_set_digits(GTK_SCALE(scale), 0);
  gtk_box_pack_start_defaults(GTK_BOX(vbox2), scale);
  gtk_signal_connect(GTK_OBJECT(gAdjust.Water), "value_changed",
                     (GtkSignalFunc)double_adjustment_update,
                     &gWaterLimit);
  gtk_tooltips_set_tip(tooltips, scale, _("Specify threshold value for internal height table"), NULL);

  /* grass height level */
  label = gtk_label_new(_("below value is grass"));
  gtk_box_pack_start_defaults(GTK_BOX(vbox2), label);
  gAdjust.Grass = (GtkAdjustment *)gtk_adjustment_new(gGrassLimit  ,
                        0.0, 256.0, 1.0, 1.0, 1.0);
  scale = gtk_hscale_new(gAdjust.Grass);
  gtk_scale_set_digits(GTK_SCALE(scale), 0);
  gtk_box_pack_start_defaults(GTK_BOX(vbox2), scale);
  gtk_signal_connect(GTK_OBJECT(gAdjust.Grass), "value_changed",
                     (GtkSignalFunc)double_adjustment_update,
                     &gGrassLimit);
  gtk_tooltips_set_tip(tooltips, scale, _("Specify threshold value for internal height table"), NULL);

  /* tree height level */
  label = gtk_label_new(_("below value is tree"));
  gtk_box_pack_start_defaults(GTK_BOX(vbox2), label);
  gAdjust.Tree = (GtkAdjustment *)gtk_adjustment_new(gTreeLimit  ,
                        0.0, 256.0, 1.0, 1.0, 1.0);
  scale = gtk_hscale_new(gAdjust.Tree);
  gtk_scale_set_digits(GTK_SCALE(scale), 0);
  gtk_box_pack_start_defaults(GTK_BOX(vbox2), scale);
  gtk_signal_connect(GTK_OBJECT(gAdjust.Tree), "value_changed",
                     (GtkSignalFunc)double_adjustment_update,
                     &gTreeLimit);
  gtk_tooltips_set_tip(tooltips, scale, _("Specify threshold value for internal height table"), NULL);

  gtk_widget_show_all(GTK_WIDGET(dialog));
}

void generate_map(void)
{
  gXDim = Pud.width;
  gYDim = Pud.height;
  // Set default settings
  gNumMountain  = sqrt(gXDim*gYDim);
  gNumLake    = sqrt(gXDim*gYDim);
  gNumRiver    = sqrt(gXDim*gYDim)/10;
/*  gDeepWaterLimit  = 0x66;
  gWaterLimit    = 0x76;
  gGrassLimit    = 0x7C;
  gTreeLimit    = 0x86;
  gGenerateIsland  = FALSE;
*/
  MapGenerationDialog();
}
