/*
	$Id: file_inputprovider.cpp,v 1.2 2000/04/25 21:40:16 mbn Exp $

	------------------------------------------------------------------------
	ClanLib, the platform independent game SDK.

	This library is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	version 2. See COPYING for details.

	For a total list of contributers see CREDITS.

	------------------------------------------------------------------------

	File purpose:
		File input source provider, part of the i/o data component.

*/

#include "Core/precomp.h"

#include <stdio.h>
#include <strstream>

#include <API/Core/System/error.h>
#include <API/Core/IOData/inputsource.h>
#include <API/Core/IOData/inputsource_provider.h>
#include "API/Core/System/cl_assert.h"
#include "file_inputprovider.h"

CL_InputSourceProvider *CL_InputSourceProvider::create_file_provider(const char * /*path*/)
{
	return new CL_InputSourceProvider_File(/*path*/);
}

CL_InputSource *CL_InputSourceProvider_File::open_source(const char *filename)
{
	return new CL_InputSource_File(filename);
}

CL_InputSourceProvider *CL_InputSourceProvider_File::clone()
{
	return new CL_InputSourceProvider_File(/*path*/);
}

/**********************
	CL_InputSource_File
**********************/

CL_InputSource_File::CL_InputSource_File(const char *_filename)
{
	filename = _filename;
	filehandle = NULL;
	open();
}

CL_InputSource_File::CL_InputSource_File(const CL_InputSource_File *source)
{
	filename = source->filename;
	filehandle = NULL;

	open();
	fseek(filehandle, ftell(source->filehandle), SEEK_SET);
}

CL_InputSource_File::~CL_InputSource_File()
{
	close();
}

void CL_InputSource_File::set_system_mode()
{
	// BUG BUG: Not implemented!
}

void CL_InputSource_File::set_big_endian_mode()
{
	// BUG BUG: Not implemented!
}

void CL_InputSource_File::set_little_endian_mode()
{
	// BUG BUG: Not implemented!
}

int CL_InputSource_File::read_int32()
{
	int input;
	if (fread(&input, 4, 1, filehandle) != 1)
	{
		throw CL_Error("CL_InputSource_File::read_int32() failed");
	}
	return input;
}

unsigned int CL_InputSource_File::read_uint32()
{
	unsigned int input;
	if (fread(&input, 4, 1, filehandle) != 1)
	{
		throw CL_Error("CL_InputSource_File::read_uint32() failed");
	}
	return input;
}

short CL_InputSource_File::read_short16()
{
	short input;
	if (fread(&input, 2, 1, filehandle) != 1)
	{
		throw CL_Error("CL_InputSource_File::read_short16() failed");
	}

	return input;
}

unsigned short CL_InputSource_File::read_ushort16()
{
	unsigned short input;
	if (fread(&input, 2, 1, filehandle) != 1)
	{
		throw CL_Error("CL_InputSource_File::read_ushort16() failed");
	}

	return input;
}

char CL_InputSource_File::read_char8()
{
	char input;
	if (fread(&input, 1, 1, filehandle) != 1)
	{
		throw CL_Error("CL_InputSource_File::read_char8() failed");
	}

	return input;
}

unsigned char CL_InputSource_File::read_uchar8()
{
	unsigned char input;
	if (fread(&input, 1, 1, filehandle) != 1)
	{
		throw CL_Error("CL_InputSource_File::read_uchar8() failed");
	}
	return input;
}

float CL_InputSource_File::read_float32()
{
	float input;
	if (fread(&input, 4, 1, filehandle) != 1)
	{
		throw CL_Error("CL_InputSource_File::read_float32() failed");
	}
	return input;
}

int CL_InputSource_File::read(void *data, int size)
{
	return fread(data, 1, size, filehandle);
}

void CL_InputSource_File::open()
{
	if (filehandle != NULL) return;

#ifndef WIN32 // hate win32 non posix conform
	if (filename[0] == '!')
	{
		filehandle = popen(std::string(filename, 1).c_str(), "rb");
		if (filehandle == NULL)
		{
			std::strstream err;
			err << "Could not open pipe: " << std::string(filename,1 );
			throw CL_Error(err.str());
		}
		filesize = 99999999;
	}
	else
#endif
	{
		filehandle = fopen(filename.c_str(), "rb");
		if (filehandle == NULL)
		{
			std::strstream err;
			err << "Could not open file: " << filename.c_str();
			throw CL_Error(err.str());
		}
		fseek(filehandle, 0, SEEK_END);
		filesize = ftell(filehandle);
		fseek(filehandle, 0, SEEK_SET);
	}
//	cl_assert(filehandle != NULL);

}

void CL_InputSource_File::close()
{
	if (filehandle == NULL) return;
	fclose(filehandle);

	filehandle = NULL;
}

CL_InputSource *CL_InputSource_File::clone() const
{
	return new CL_InputSource_File(this);
}

int CL_InputSource_File::tell() const
{
	return ftell(filehandle);
}

void CL_InputSource_File::seek(int pos, SeekEnum seek_type)
{
	switch (seek_type)
	{
	case seek_cur:
	fseek(filehandle, pos, SEEK_CUR);
		break;

	case seek_set:
	fseek(filehandle, pos, SEEK_SET);
		break;

	case seek_end:
	fseek(filehandle, pos, SEEK_END);
		break;
	}
}

int CL_InputSource_File::size() const
{
	return filesize;
}

std::string CL_InputSource_File::read_string()
{
	int size = read_int32();

	char *str = new char[size];
	read(str, size);
	
	std::string ret = str;
	delete[] str;

	return ret;
}

void CL_InputSource_File::push_position()
{
	int a = ftell(filehandle);

	stack.push(a);
}

void CL_InputSource_File::pop_position()
{
	int a = stack.top();
	stack.pop();

	fseek(filehandle, a, SEEK_SET);
}
