/*
	$Id: surface.h,v 1.1.1.1 2000/04/09 12:17:59 mbn Exp $

	------------------------------------------------------------------------
	ClanLib, the platform independent game SDK.

	This library is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	version 2. See COPYING for details.

	For a total list of contributers see CREDITS.

	------------------------------------------------------------------------
*/

//! component="Display 2D"

#ifndef header_surface
#define header_surface

class CL_DisplayCard;
class CL_SurfaceProvider;
class CL_Target;
class CL_ResourceManager;

class CL_Surface
//: Image class in ClanLib.
// <p>This class is used to draw 2D images onto the backbuffer or to a
// <a class="CL_Target">renderer target</a>.</p>
//
// <p>In ClanLib, a surface is understood
// to be an image represented in a form, where it is most suitable for as
// fast rendering as possible. Unlike many other libraries, it is not
// possible to get direct access to the surface data.</p>
//
// <p>A surface uses a <a class="CL_SurfaceProvider">surface provider</a> to
// load its data. It is passed at creation time, and you can either roll
// your own provider, or use one already available in ClanLib.</p>
//
// <p>It is also possible to create a surface from a resource source. This
// will allow you to load an image using a resource ID instead of directly
// specifying the image in your source code.</p>
//
// <b>Dynamic surfaces</b>
//
// <p>If you have an image which changes very frequently (eg. each
// frame), you want to use a dynamic surface instead. The difference is that
// the surface won't cache the image, and it won't convert it to another
// image format more suitable for the display target. Note that this is only
// faster if you change your image very frequently - otherwise a normal
// surface is the best solution.</p>
//
// <p>When constructing your own images, you will most likely find the <a
// class="CL_Canvas">canvas surface provider</a> interesting. This class
// will make it easier to create a custom drawing area where you can draw
// your image.</p>
//
// <p>All surfaceproviders is a <a class="CL_Target">renderer target</a>.
// However, most of the normal surface providers are "read only", meaning
// that their image data will only remain intact during a lock session. This
// is because the providers consume less memory when the image is only
// available, when someone needs it. Only the <a class="CL_Canvas">canvas
// provider</a> will not destroy the image.</p>
//
// <p>A surface can blit (draw itself) itself to the backbuffer, or to a
// renderer target. This means that you can use the surface to draw one
// surface provider to another.</p>
{
public:
	static CL_Surface *load(
		const char *resource_id,
		CL_ResourceManager *manager);
//: This function loads a surface from a resource file (usually a datafile).

	static CL_Surface *create(
		CL_SurfaceProvider *provider,
		bool delete_provider=false);
//: This function creates a CL_Surface instance being blitted by a caching blitter.
//!param: provider - Image source.
//!param: delete_provider - If true, the surface will delete the provider when deleted itself.
//!retval: The surface created.

	static CL_Surface *create_dynamic(
		CL_SurfaceProvider *provider,
		bool delete_provider=false);
//: This function creates a CL_Surface instance being blitted by the dynamic (non caching) blitter.
//: You don't need to call reload with dynamic surfaces as they do that by themselves. 
//!param: provider - Image source.
//!param: delete_provider - If true, the surface will delete the provider when deleted itself.
//!retval: The surface created.

	virtual ~CL_Surface() {;}

	virtual void reload()=0; 
//: Forces surface to reload surface data from provider.

	virtual CL_SurfaceProvider *get_provider() const =0;
//: Returns the surface provider used by this surface.
//!retval: Surface provider used by this surface.

//!info: Display functions.

	virtual void put_screen(
		int x,
		int y,
		int spr_no=0,
		CL_DisplayCard *card=NULL)=0;
//: Draws the surface onto the backbuffer.
//!param: x - x coordinate to draw surface.
//!param: y - y coordinate to draw surface.
//!param: spr_no - subsprite number to use.
//!param: card - Display card to be drawn onto. If NULL it will use the currently selected video card by CL_Display.

	virtual void put_screen(
		int x,
		int y,
		float scale_x,
		float scale_y,
		int spr_no=0,
		CL_DisplayCard *card=NULL)=0;
//: Draws the surface onto the backbuffer - scaling it to a specified size.
//!param: x - x coordinate to draw surface.
//!param: y - y coordinate to draw surface.
//!param: scale_x - scale x multiplyer.
//!param: scale_y - scale y multiplyer.
//!param: spr_no - subsprite number to use.
//!param: card - Display card to be drawn onto. If NULL it will use the currently selected video card by CL_Display.

	virtual void put_screen(
		int x,
		int y,
		int size_x,
		int size_y,
		int spr_no=0,
		CL_DisplayCard *card=NULL)=0;
//: Draws the surface onto the backbuffer - scaling it to a specified size.
//!param: x - x coordinate to draw surface.
//!param: y - y coordinate to draw surface.
//!param: size_x - width of the image when scaled.
//!param: size_y - height of the image when scaled.
//!param: spr_no - subsprite number to use.
//!param: card - Display card to be drawn onto. If NULL it will use the currently selected video card by CL_Display.

	virtual void put_target(
		int x,
		int y,
		int spr_no,
		CL_Target *target)=0;
//: Draws the surface to a target.
//!param: x - x coordinate to draw surface.
//!param: y - y coordinate to draw surface.
//!param: spr_no - subsprite number to use.
//!param: target - target.

	virtual unsigned int get_width() const =0;
//: Returns the width of the surface

	virtual unsigned int get_height() const =0;
//: Returns the height of the surface

	virtual unsigned int get_num_frames() const =0;
//: Returns the number of frames/subsprites in the surface

	virtual bool is_video(CL_DisplayCard *card = NULL) const =0;
//: returns true if in videomemory 
//: on the specified card (null = current dispcard).
	
	virtual bool is_loaded(CL_DisplayCard *card = NULL) const =0;
//: returns true if loaded in either video- or system-memory

	virtual bool convert_video(CL_DisplayCard *card = NULL)=0;
//: returns true if successfully loaded into videomemory, or already there

	virtual bool convert_system(CL_DisplayCard *card = NULL)=0;
//: convert surface to system memory - never fails! (or serious heap usage!)

	virtual void flush(CL_DisplayCard *card = NULL)=0;
//: completely flushes surface (removes from video/system-memory)
};

#endif
