;;;; -*- Mode: Lisp; Syntax: ANSI-Common-Lisp; Base: 10; Package: HYPEROBJECT -*-
;;;; *************************************************************************
;;;; FILE IDENTIFICATION
;;;;
;;;; Name:          hyperobject.lisp
;;;; Purpose:       Hyper Object (Plain - no Metaclass)
;;;; Programmer:    Kevin M. Rosenberg
;;;; Date Started:  Nov 2002
;;;;
;;;; This is a rewrite of hyperobjec't to avoid using metaclasses.
;;;;
;;;; $Id: hyperobject-no-mop.lisp 7061 2003-09-07 06:34:45Z kevin $
;;;;
;;;; This file is Copyright (c) 2000-2002 by Kevin M. Rosenberg
;;;;
;;;; *************************************************************************


(defpackage #:hyperobject-no-mop
  (:nicknames #:ho-no-mop)
  (:use #:common-lisp #:kmrcl)
  (:export
   #:define-hyperobject
   #:hyperobject
   #:hyperobject-base-url!
   #:load-all-subobjects
   #:print-hyperobject
   ))

(defpackage #:hyperobject-no-mop-user
  (:nicknames #:ho-no-mop-user)
  (:use #:hyperobject-no-mop #:cl #:cl-user))

(in-package :hyperobject-no-mop)


(eval-when (:compile-toplevel :execute)
  (declaim (optimize (speed 3) (safety 1) (compilation-speed 0) (debug 3))))


;; Main class

(defclass hyperobject ()
  ())

(defclass hyperobject-meta ()
  ((fields :initform nil :type list)
   (subobjects :initform nil :type list)
   (references :initform nil :type list)
   (value-func :initform nil :type function)
   (xml-value-func :initform nil :type function)
   (fmtstr-text :initform nil :type string)
   (fmtstr-html :initform nil :type string)
   (fmtstr-xml :initform nil :type string)
   (fmtstr-text-labels :initform nil :type string)
   (fmtstr-html-labels :initform nil :type string)
   (fmtstr-xml-labels :initform nil :type string)
   (fmtstr-html-ref :initform nil :type string)
   (fmtstr-xml-ref :initform nil :type string)
   (fmtstr-html-ref-labels :initform nil :type string)
   (fmtstr-xml-ref-labels :initform nil :type string)
   )
  (:documentation "Class holding meta information for hyperobjects"))

(defclass field ()
  ((name :type symbol :initform nil :initarg :name :reader name)
   (print-formatter :initform nil :initarg :print-formatter :reader print-formatter)
   (cl-type :initform nil :reader cl-type)
   (ho-type :initform nil :reader ho-type)
   (subobject :initform nil :reader subobject)
   (reference :initform nil :reader reference)
   ))


(defmethod print-object ((obj field) (s stream))
  (print-unreadable-object (obj s :type t :identity t)
    (format s "~S" (name obj))))

(defclass subobject ()
  ((name :type symbol :initform nil :initarg :name :reader name)
   (reader :type function :initform nil :initarg :reader :reader reader)))

(defmethod print-object ((obj subobject) (s stream))
  (print-unreadable-object (obj s :type t :identity t)
    (format s "~S" (name obj))))

(defclass reference ()
  ((name :type symbol :initform nil :initarg :name :reader name)
   (lookup :type function :initform nil :initarg :lookup :reader lookup)
   (link-parameters :type list :initform nil :initarg :link-parameters
		    :reader link-parameters)))

(defmethod print-object ((obj reference) (s stream))
  (print-unreadable-object (obj s :type t :identity t)
    (format s "~S" (name obj))))
   
(defun remove-keys (key-names args)
  (loop for ( name val ) on args by #'cddr
	unless (member (symbol-name name) key-names 
		       :key #'symbol-name :test 'equal)
	append (list name val)))

(defun convert-ho-type (ho-type)
  (check-type ho-type symbol)
  (case (intern (symbol-name ho-type) (symbol-name :keyword))
    (:string
     'string)
    (:fixnum
     'fixnum)
    (:boolean
     'boolean)
    (:integer
     'integer)
    (:cdata
     'string)
    (:float
     'float)
    (otherwise
     ho-type)))

(defun process-hyper-fields (raw-fields meta)
  (let* ((fields '())
	 (references '())
	 (subobjects '())
	 (processed-fields
	  (loop for field in raw-fields
	     collecting
	       (destructuring-bind
		     (name &rest rest &key
			   ;; the following list of keywords is reproduced below in the
			   ;; remove-keys form.  important to keep them in sync
			   type reader writer accessor initform print-formatter initarg
			   reference subobject
			   ;; list ends
			   &allow-other-keys) field
		 (let ((other-args (remove-keys
				    '(type reader writer accessor initform print-formatter
				      initarg reference subobject)
			       rest))
		       (field-obj (make-instance 'field :name name)) 
		       (kv nil))
		   (declare (ignore other-args))
		   (push field-obj fields)
		   (loop for (k v) in `((:type ,type) (:reader ,reader) (:writer ,writer)
					(:accessor ,accessor) (:initform ,initform)
					(:print-formatter ,print-formatter) (:initarg ,initarg)
					(:subobject ,subobject) (:reference ,reference))
		      do
			(case k
			  (:initarg
			   (push (list :initarg
				       (if v
				     v
				     (intern (symbol-name name)
					     (symbol-name :keyword))))
				 kv))
			  (:accessor
			   (when v
			     (push (list :accessor v) kv)))
			  (:print-formatter
			   (when v
			     (setf (slot-value field-obj 'print-formatter) v)))
			  (:writer
			   (when v
			     (push (list :writer v) kv)))
			  (:reader
			   (if v
			       (push (list :reader v) kv)
		       (push (list :reader name) kv)))
			  (:type
			   (when v
			     (setf (slot-value field-obj 'ho-type) v)
			     (setf (slot-value field-obj 'cl-type) (convert-ho-type v))
			     (push (list :type (cl-type field-obj)) kv)))
			  (:subobject
			   (when v
			     (let ((subobj (make-instance
					    'subobject :name name
					    :reader
					    (if (eq v t)
						name
						v))))
			       (setf (slot-value field-obj 'subobject) subobj)
			       (push subobj subobjects))))
			  (:reference
			   (when v
			     (let ((ref (make-instance 'reference :name name :lookup v)))
			       (setf (slot-value field-obj 'reference) ref)
			       (push ref references))))
			  ))
		   (append
		    (list name)
		    (loop for (k v) in kv
		       collecting k
		       collecting v)))))))
    (setf (slot-value meta 'fields) (nreverse fields))
    (setf (slot-value meta 'references) (nreverse references))
    (setf (slot-value meta 'subobjects) (nreverse subobjects))
    processed-fields))
    
  
(defun process-title (name meta)
  (let ((title (cadr (assoc :title meta))))
    (if title
	(if (symbolp title)
	    (symbol-name title)
	    title)
	(symbol-name name))))

(defun process-documentation (meta)
  (let ((doc (cadr (assoc :title meta))))
    (if (and doc (symbolp doc))
	(symbol-name doc)
	doc)))

;;;; Class initialization function
(defun init-hyperobject-class (name meta)
    (let ((fmtstr-text "")
	  (fmtstr-html "")
	  (fmtstr-xml "")
	  (fmtstr-text-labels "")
	  (fmtstr-html-labels "")
	  (fmtstr-xml-labels "")
	  (fmtstr-html-ref "")
	  (fmtstr-xml-ref "")
	  (fmtstr-html-ref-labels "")
	  (fmtstr-xml-ref-labels "")
	  (first-field t)
	  (value-func '())
	  (xml-value-func '())
	  (package (symbol-package name)))
      (dolist (field (slot-value meta 'fields))
	(let* ((name (name field))
	       (print-formatter (print-formatter field))
	       (type (ho-type field))
	       (reference (reference field))
	       (namestr (symbol-name name))
	       (namestr-lower (string-downcase (symbol-name name)))
	       (value-fmt "~a")
	       (plain-value-func nil)
	       html-str xml-str html-label-str xml-label-str)
	  (unless (subobject field)
	    (when (or (eql type :integer) (eql type :fixnum))
	      (setq value-fmt "~d"))
	    
	    (when (eql type :boolean)
	      (setq value-fmt "~a"))
	    
	    (if first-field
		(setq first-field nil)
		(progn
		  (string-append fmtstr-text " ")
		  (string-append fmtstr-html " ")
		  (string-append fmtstr-xml " ")
		  (string-append fmtstr-text-labels " ")
		  (string-append fmtstr-html-labels " ")
		  (string-append fmtstr-xml-labels " ")
		  (string-append fmtstr-html-ref " ")
		  (string-append fmtstr-xml-ref " ")
		  (string-append fmtstr-html-ref-labels " ")
		  (string-append fmtstr-xml-ref-labels " ")))
	    
	    (setq html-str (concatenate 'string "<span class=\"" namestr-lower "\">" value-fmt "</span>"))
	    (setq xml-str (concatenate 'string "<" namestr-lower ">" value-fmt "</" namestr-lower ">"))
	    (setq html-label-str (concatenate 'string "<span class=\"label\">" namestr-lower "</span> <span class=\"" namestr-lower "\">" value-fmt "</span>"))
	    (setq xml-label-str (concatenate 'string "<label>" namestr-lower "</label> <" namestr-lower ">" value-fmt "</" namestr-lower ">"))
	    
	    (string-append fmtstr-text value-fmt)
	    (string-append fmtstr-html html-str)
	    (string-append fmtstr-xml xml-str)
	    (string-append fmtstr-text-labels namestr-lower " " value-fmt)
	    (string-append fmtstr-html-labels html-label-str)
	    (string-append fmtstr-xml-labels xml-label-str)

	    (if reference
		(progn
		  (string-append fmtstr-html-ref "<~~a>" value-fmt "</~~a>")
		  (string-append fmtstr-xml-ref "<~~a>" value-fmt "</~~a>")
		  (string-append fmtstr-html-ref-labels "<span class=\"label\">" namestr-lower "</span> <~~a>" value-fmt "</~~a>")
		  (string-append fmtstr-xml-ref-labels "<label>" namestr-lower "</label> <~~a>" value-fmt "</~~a>"))
		(progn
		  (string-append fmtstr-html-ref html-str)
		  (string-append fmtstr-xml-ref xml-str)
		  (string-append fmtstr-html-ref-labels html-label-str)
		  (string-append fmtstr-xml-ref-labels xml-label-str)))

	    (if print-formatter
		(setq plain-value-func 
		      (list `(,print-formatter (,(intern namestr package) x))))
		(setq plain-value-func 
		      (list `(,(intern namestr package) x))))
	    (setq value-func (append value-func plain-value-func))
	    
	    (if (eql type :cdata)
		(setq xml-value-func (append xml-value-func (list `(xml-cdata ,@plain-value-func))))
		(setq xml-value-func (append xml-value-func plain-value-func)))
	    )))
	
      (if value-func
	  (setq value-func `(lambda (x) (values ,@value-func)))
	  (setq value-func `(lambda () (values))))
      
      (if xml-value-func
	  (setq xml-value-func `(lambda (x) (values ,@xml-value-func)))
	  (setq xml-value-func `(lambda () (values))))
	
      (setf (slot-value meta 'fmtstr-text) fmtstr-text)
      (setf (slot-value meta 'fmtstr-html) fmtstr-html)
      (setf (slot-value meta 'fmtstr-xml) fmtstr-xml)
      (setf (slot-value meta 'fmtstr-text-labels) fmtstr-text-labels)
      (setf (slot-value meta 'fmtstr-html-labels) fmtstr-html-labels)
      (setf (slot-value meta 'fmtstr-xml-labels) fmtstr-xml-labels)
      (setf (slot-value meta 'fmtstr-html-ref) fmtstr-html-ref)
      (setf (slot-value meta 'fmtstr-xml-ref) fmtstr-xml-ref)
      (setf (slot-value meta 'fmtstr-html-ref-labels) fmtstr-html-ref-labels)
      (setf (slot-value meta 'fmtstr-xml-ref-labels) fmtstr-xml-ref-labels)
      (setf (slot-value meta 'value-func) value-func)
      (setf (slot-value meta 'xml-value-func) xml-value-func))
    (values))

(defgeneric ho-title (obj) )
(defgeneric ho-name (obj) )
(defgeneric ho-value-func (obj) )
(defgeneric ho-xml-value-func (obj) )
(defgeneric ho-fmtstr-text (obj) )
(defgeneric ho-fmtstr-html (obj) )
(defgeneric ho-fmtstr-xml (obj) )
(defgeneric ho-fmtstr-text-labels (obj) )
(defgeneric ho-fmtstr-html-labels (obj) )
(defgeneric ho-fmtstr-xml-labels (obj) )
(defgeneric ho-fmtstr-html-ref (obj) )
(defgeneric ho-fmtstr-xml-ref (obj) )
(defgeneric ho-fmtstr-html-ref-labels (obj) )
(defgeneric ho-fmtstr-xml-ref-labels (obj) )
(defgeneric ho-fields (obj) )
(defgeneric ho-references (obj) )
(defgeneric ho-subobjects (obj) )
  
(defmacro define-hyperobject (name parents fields &rest meta-fields)
  (let* ((meta (make-instance 'hyperobject-meta))
	 (cl-fields (process-hyper-fields fields meta))
	 (title (process-title name meta-fields))
	 (documentation (process-documentation meta-fields))
	 (value-func (gensym))
	 (xml-value-func (gensym)))
    (init-hyperobject-class name meta)
    `(progn
       (eval-when (:compile-toplevel :load-toplevel :execute)
	 (defclass ,name ,(append parents (list 'hyperobject)) ,cl-fields
	   ,@(and documentation (list (list :documentation documentation)))))
       (let ((,value-func (compile nil (eval (slot-value ,meta 'value-func))))
	     (,xml-value-func (compile nil (eval (slot-value ,meta 'xml-value-func)))))
	 (defmethod ho-title ((obj ,name))
	   ,title)
	 (defmethod ho-name ((obj ,name))
	   ,(string-downcase (symbol-name name)))
	 (defmethod ho-fields ((obj ,name))
	   ',(slot-value meta 'fields))
	 (defmethod ho-references ((obj ,name))
	   ',(slot-value meta 'references))
	 (defmethod ho-subobjects ((obj ,name))
	   ',(slot-value meta 'subobjects))
	 (defmethod ho-value-func ((obj ,name))
	   ,value-func)
	 (defmethod ho-xml-value-func ((obj ,name))
	   ,xml-value-func)
	 (defmethod ho-fmtstr-text ((obj ,name))
	   ,(slot-value meta 'fmtstr-text))
	 (defmethod ho-fmtstr-html ((obj ,name))
	   ,(slot-value meta 'fmtstr-html))
	 (defmethod ho-fmtstr-xml ((obj ,name))
	   ,(slot-value meta 'fmtstr-xml))
	 (defmethod ho-fmtstr-text-labels ((obj ,name))
	   ,(slot-value meta 'fmtstr-text-labels))
	 (defmethod ho-fmtstr-html-labels ((obj ,name))
	   ,(slot-value meta 'fmtstr-html-labels))
	 (defmethod ho-fmtstr-xml-labels ((obj ,name))
	   ,(slot-value meta 'fmtstr-xml-labels))
	 (defmethod ho-fmtstr-html-ref ((obj ,name))
	   ,(slot-value meta 'fmtstr-html-ref))
	 (defmethod ho-fmtstr-xml-ref ((obj ,name))
	   ,(slot-value meta 'fmtstr-xml-ref))
	 (defmethod ho-fmtstr-html-ref-labels ((obj ,name))
	   ,(slot-value meta 'fmtstr-html-ref-labels))
	 (defmethod ho-fmtstr-xml-ref-labels ((obj ,name))
	   ,(slot-value meta 'fmtstr-xml-ref-labels))
	 ))))

;;;; Generic Print functions

(defparameter *default-textformat* nil)
(defparameter *default-htmlformat* nil)
(defparameter *default-htmlrefformat* nil)
(defparameter *default-xhtmlformat* nil)
(defparameter *default-xhtmlrefformat* nil)
(defparameter *default-xmlformat* nil)
(defparameter *default-xmlrefformat* nil)
(defparameter *default-ie-xmlrefformat* nil)
(defparameter *default-nullformat* nil)
(defparameter *default-init-format?* nil)

(defun make-format-instance (fmt)
  (unless *default-init-format?*
    (setq *default-textformat* (make-instance 'textformat))
    (setq *default-htmlformat* (make-instance 'htmlformat))
    (setq *default-htmlrefformat* (make-instance 'htmlrefformat))
    (setq *default-xhtmlformat* (make-instance 'xhtmlformat))
    (setq *default-xhtmlrefformat* (make-instance 'xhtmlrefformat))
    (setq *default-xmlformat* (make-instance 'xmlformat))
    (setq *default-xmlrefformat* (make-instance 'xmlrefformat))
    (setq *default-ie-xmlrefformat* (make-instance 'ie-xmlrefformat))
    (setq *default-nullformat* (make-instance 'nullformat))
    (setq *default-init-format?* t))
  
  (case fmt
      (:text *default-textformat*)
      (:html *default-htmlformat*)
      (:htmlref *default-htmlrefformat*)
      (:xhtml  *default-xhtmlformat*)
      (:xhtmlref *default-xhtmlrefformat*)
      (:xml  *default-xmlformat*)
      (:xmlref *default-xmlrefformat*)
      (:ie-xmlref *default-ie-xmlrefformat*)
      (:null *default-nullformat*)
      (otherwise *default-textformat*)))
    
;;;; Output format classes for print hoes

(defclass dataformat ()
  ((file-start-str :type string :initarg :file-start-str :reader file-start-str)
   (file-end-str :type string :initarg :file-end-str :reader file-end-str)
   (list-start-fmtstr :type string :initarg :list-start-fmtstr :reader list-start-fmtstr)
   (list-start-value-func :type function :initarg :list-start-value-func :reader list-start-value-func)
   (list-start-indent :initarg :list-start-indent :reader list-start-indent)
   (list-end-fmtstr :type string :initarg :list-end-fmtstr :reader list-end-fmtstr)
   (list-end-value-func :type function :initarg :list-end-value-func :reader list-end-value-func)
   (list-end-indent :initarg :list-end-indent :reader list-end-indent)
   (obj-start-fmtstr :type string :initarg :obj-start-fmtstr :reader obj-start-fmtstr)
   (obj-start-value-func :initarg :obj-start-value-func :reader obj-start-value-func)
   (obj-start-indent :initarg :obj-start-indent :reader obj-start-indent)
   (obj-end-fmtstr :type string :initarg :obj-end-fmtstr :reader obj-end-fmtstr)
   (obj-end-value-func :initarg :obj-end-value-func :reader obj-end-value-func)
   (obj-end-indent :initarg :obj-end-indent :reader obj-end-indent)
   (obj-data-indent :initarg :obj-data-indent :reader obj-data-indent)
   (obj-data-fmtstr :initarg :obj-data-fmtstr :reader  obj-data-fmtstr)
   (obj-data-fmtstr-labels :initarg :obj-data-fmtstr-labels :reader  obj-data-fmtstr-labels)
   (obj-data-end-fmtstr :initarg :obj-data-end-fmtstr :reader obj-data-end-fmtstr)
   (obj-data-value-func :initarg :obj-data-value-func :reader obj-data-value-func)
   (link-ref :initarg :link-ref :reader link-ref))
  (:default-initargs :file-start-str nil :file-end-str nil :list-start-fmtstr nil :list-start-value-func nil
		     :list-start-indent nil :list-end-fmtstr nil :list-end-value-func nil :list-end-indent nil
		     :obj-start-fmtstr nil :obj-start-value-func nil :obj-start-indent nil
		     :obj-end-fmtstr nil :obj-end-value-func nil :obj-end-indent nil
		     :obj-data-indent nil :obj-data-fmtstr nil :obj-data-fmtstr-labels nil :obj-data-end-fmtstr nil
		     :obj-data-value-func nil :link-ref nil)
  (:documentation "Parent for all dataformat objects"))

(defclass binaryformat (dataformat)
  ())

(defclass nullformat (dataformat)
  ())

(defun text-list-start-value-func (obj nitems)
  (values (ho-title obj) nitems))

(defclass textformat (dataformat) 
  ()	
  (:default-initargs :list-start-fmtstr "~a~P:~%"
    :list-start-value-func #'text-list-start-value-func
    :list-start-indent t
    :obj-data-indent t
    :obj-data-fmtstr #'ho-fmtstr-text
    :obj-data-fmtstr-labels #'ho-fmtstr-text-labels
    :obj-data-end-fmtstr "~%"
    :obj-data-value-func #'ho-value-func))


(defun htmlformat-list-start-value-func (x nitems) 
  (values (ho-title x) nitems (ho-name x)))

(defclass htmlformat (textformat) 
  ()
  (:default-initargs :file-start-str "<html><body>~%"
    :file-end-str "</body><html>~%"
    :list-start-indent t
    :list-start-fmtstr "<p><b>~a~p:</b></p><div class=\"~A\"><ul>~%"
    :list-start-value-func #'htmlformat-list-start-value-func
    :list-end-fmtstr "</ul></div>~%"
    :list-end-indent t
    :list-end-value-func #'identity
    :obj-start-indent t
    :obj-start-fmtstr "<li>"
    :obj-start-value-func #'identity
    :obj-end-indent  t
    :obj-end-fmtstr  "</li>~%"
    :obj-end-value-func #'identity
    :obj-data-indent t
    :obj-data-fmtstr #'ho-fmtstr-html-labels
    :obj-data-fmtstr-labels #'ho-fmtstr-html-labels
    :obj-data-value-func #'ho-value-func))

(defclass xhtmlformat (textformat) 
  ()
  (:default-initargs :file-start-str "<html><body>~%"
    :file-end-str "</body><html>~%"
    :list-start-indent t
    :list-start-fmtstr "<p><b>~a~p:</b></p><div class=\"~A\"><ul>~%"
    :list-start-value-func #'htmlformat-list-start-value-func
    :list-end-fmtstr "</ul></div>~%"
    :list-end-indent t
    :list-end-value-func #'identity
    :obj-start-indent t
    :obj-start-fmtstr "<li>"
    :obj-start-value-func #'identity
    :obj-end-indent  t
    :obj-end-fmtstr  "</li>~%"
    :obj-end-value-func #'identity
    :obj-data-indent t
    :obj-data-fmtstr #'ho-fmtstr-html-labels
    :obj-data-fmtstr-labels #'ho-fmtstr-html-labels
    :obj-data-value-func #'ho-xml-value-func))


(defun xmlformat-list-end-value-func (x)
  (format nil "~alist" (ho-name x)))

(defun xmlformat-list-start-value-func (x nitems) 
  (values (format nil "~alist" (ho-name x)) (ho-title x) nitems))

(defclass xmlformat (textformat) 
  ()
  (:default-initargs :file-start-str "" ; (std-xml-header)
    :list-start-indent  t
    :list-start-fmtstr "<~a><title>~a~p:</title> ~%"
    :list-start-value-func #'xmlformat-list-start-value-func
    :list-end-indent  t
    :list-end-fmtstr "</~a>~%"
    :list-end-value-func #'xmlformat-list-end-value-func
    :obj-start-fmtstr "<~a>"
    :obj-start-value-func #'ho-name
    :obj-start-indent t
    :obj-end-fmtstr "</~a>~%"
    :obj-end-value-func #'ho-name
    :obj-end-indent nil
    :obj-data-indent nil
    :obj-data-fmtstr #'ho-fmtstr-xml
    :obj-data-fmtstr-labels #'ho-fmtstr-xml-labels
    :obj-data-value-func #'ho-xml-value-func))

(defclass link-ref ()
  ((fmtstr :type function :initarg :fmtstr :accessor fmtstr)
   (fmtstr-labels :type function :initarg :fmtstr-labels :accessor fmtstr-labels)
   (page-name :type string :initarg :page-name :accessor page-name)
   (href-head :type string :initarg :href-head :accessor href-head)
   (href-end :type string :initarg :href-end :accessor href-end)
   (ampersand :type string :initarg :ampersand :accessor ampersand))
  (:default-initargs :fmtstr nil 
    :fmtstr-labels nil 
    :page-name "disp-func1" 
    :href-head nil :href-end nil :ampersand nil)
  (:documentation "Formatting for a linked reference"))

(defclass html-link-ref (link-ref)
  ()
  (:default-initargs :fmtstr #'ho-fmtstr-html-ref  
    :fmtstr-labels #'ho-fmtstr-html-ref-labels
    :href-head "a href=" 
    :href-end "a" 
    :ampersand "&"))

(defclass xhtml-link-ref (link-ref)
  ()
  (:default-initargs :fmtstr #'ho-fmtstr-html-ref  
    :fmtstr-labels #'ho-fmtstr-html-ref-labels
    :href-head "a href=" 
    :href-end "a" 
    :ampersand "&amp;"))

(defclass xml-link-ref (link-ref)
  ()
  (:default-initargs :fmtstr #'ho-fmtstr-xml-ref 
		     :fmtstr-labels #'ho-fmtstr-xml-ref-labels
		     :href-head "xmllink xlink:type=\"simple\" xlink:href=" 
		     :href-end "xmllink" 
		     :ampersand "&amp;")
  (:documentation "Mozilla's and W3's idea of a link with XML"))

(defclass ie-xml-link-ref (xml-link-ref)
  ()
  (:default-initargs :href-head "html:a href=" 
		     :href-end "html:a" )
  (:documentation "Internet Explorer's idea of a link with XML"))


(defclass htmlrefformat (htmlformat)
  ()
  (:default-initargs :link-ref (make-instance 'html-link-ref)))

(defclass xhtmlrefformat (xhtmlformat)
  ()
  (:default-initargs :link-ref (make-instance 'xhtml-link-ref)))

(defclass xmlrefformat (xmlformat)
  ()
  (:default-initargs :link-ref (make-instance 'xml-link-ref)))

(defclass ie-xmlrefformat (xmlformat)
  ()
  (:default-initargs :link-ref (make-instance 'ie-xml-link-ref)))


;;; File Start and Ends

(defgeneric fmt-file-start (fmt s))
(defmethod fmt-file-start ((fmt dataformat) (s stream)))

(defmethod fmt-file-start ((fmt textformat) (s stream))
  (aif (file-start-str fmt)
      (format s it)))

(defgeneric fmt-file-end (fmt s))
(defmethod fmt-file-end ((fmt textformat) (s stream))
  (aif (file-end-str fmt)
	  (format s it)))

;;; List Start and Ends

(defgeneric fmt-list-start (obj fmt s &optional indent num-items))
(defmethod fmt-list-start (x (fmt textformat) (s stream) &optional (indent 0) (num-items 1))
  (if (list-start-indent fmt)
      (indent-spaces indent s))
  (aif (list-start-fmtstr fmt)
	  (apply #'format s it
		 (multiple-value-list
		  (funcall (list-start-value-func fmt) x num-items)))))

(defgeneric fmt-list-end (obj fmt s &optional indent num-items))
(defmethod fmt-list-end (x (fmt textformat) (s stream) &optional (indent 0) (num-items 1))
  (declare (ignore num-items))
  (if (list-end-indent fmt)
      (indent-spaces indent s))
  (aif (list-end-fmtstr fmt)
	  (apply #'format s it
		 (multiple-value-list
		  (funcall (list-end-value-func fmt) x)))))

;;; Object Start and Ends

(defgeneric fmt-obj-start (obj fmt s &optional indent))
(defmethod fmt-obj-start (x (fmt textformat) (s stream) &optional (indent 0))
  (if (obj-start-indent fmt)
      (indent-spaces indent s))
  (aif (obj-start-fmtstr fmt)
	  (apply #'format s it
		 (multiple-value-list
		  (funcall (obj-start-value-func fmt) x)))))

(defgeneric fmt-obj-end (obj fmt s &optional indent))
(defmethod fmt-obj-end (x (fmt textformat) (s stream) &optional (indent 0))
  (if (obj-end-indent fmt)
      (indent-spaces indent s))
  (aif (obj-end-fmtstr fmt)
	  (apply #'format s it
		 (multiple-value-list
		  (funcall (obj-end-value-func fmt) x)))))
  
;;; Object Data 

(defgeneric make-link-start (obj ref fieldname fieldfunc fieldvalue refvars))
(defmethod make-link-start (obj (ref link-ref) fieldname fieldfunc fieldvalue refvars)
  (declare (ignore obj fieldname))
  (format nil "~a\"~a?func=~a~akey=~a~a\"" 
	  (href-head ref) (make-url (page-name ref)) fieldfunc 
	  (ampersand ref) fieldvalue
	  (if refvars
	      (let ((varstr ""))
		(dolist (var refvars)
		  (string-append varstr (format nil "~a~a=~a" 
						(ampersand ref) (car var) (cadr var))))
		varstr)
	    "")))

(defgeneric make-link-end (obj ref fieldname)) 
(defmethod make-link-end (obj (ref link-ref) fieldname)
  (declare (ignore obj fieldname))
  (format nil "~a" (href-end ref))
  )

(defgeneric fmt-obj-data (obj fmt s &optional indent label refvars))
(defmethod fmt-obj-data (x (fmt textformat) s
			 &optional (indent 0) (label nil) (refvars nil))
  (if (obj-data-indent fmt)
      (indent-spaces indent s))
  (if (link-ref fmt)
      (fmt-obj-data-with-ref x fmt s label refvars)
    (fmt-obj-data-plain x fmt s label))
  (aif (obj-data-end-fmtstr fmt)
       (format s it)))

(defgeneric fmt-obj-data-plain (obj fmt s label))
(defmethod fmt-obj-data-plain (x (fmt textformat) s label)
  (if label
      (apply #'format s
	     (funcall (obj-data-fmtstr-labels fmt) x)
	     (multiple-value-list 
	      (funcall (funcall (obj-data-value-func fmt) x) x)))
    (apply #'format s (funcall (obj-data-fmtstr fmt) x)
	   (multiple-value-list
	    (funcall (funcall (obj-data-value-func fmt) x) x)))))

(defgeneric fmt-obj-data-with-ref (obj fmt s label refvars))
(defmethod fmt-obj-data-with-ref (x (fmt textformat) s label refvars)
  (let ((refstr (make-ref-data-str x fmt label))
	(refvalues nil)
	(field-values 
	 (multiple-value-list
	  (funcall (funcall (obj-data-value-func fmt) x) x))))
    
    ;; make list of reference link fields for printing to refstr template
    (dolist (ref (ho-references x))
      (let ((link-start 
	     (make-link-start x (link-ref fmt) (name ref) (lookup ref)
			      (nth (position (name ref) (ho-fields x)
					     :key #'(lambda (x) (name x)))
				   field-values)  
			      (append (link-parameters ref) refvars)))
	    (link-end (make-link-end x (link-ref fmt) (name ref))))
	(push link-start refvalues)
	(push link-end refvalues)))
    (setq refvalues (nreverse refvalues))
    
    (apply #'format s refstr refvalues)))

(defgeneric obj-data (obj))
(defmethod obj-data (x)
  "Returns the objects data as a string. Used by common-graphics outline function"
  (let ((fmt (make-format-instance :text)))
    (apply #'format nil (funcall (obj-data-fmtstr fmt) x)
	   (multiple-value-list 
	    (funcall (funcall (obj-data-value-func fmt) x) x)))))

(defgeneric make-ref-data-str (obj fmt &optional label))
(defmethod make-ref-data-str (x (fmt textformat) &optional (label nil))
  "Return fmt string for that contains ~a slots for reference link start and end"
  (unless (link-ref fmt)
    (error "fmt does not contain a link-ref"))
  (let ((refstr 
	 (if label
	     (apply #'format nil (funcall (fmtstr-labels (link-ref fmt)) x)
		    (multiple-value-list
		      (funcall (funcall (obj-data-value-func fmt) x) x)))
	   (apply #'format nil (funcall (fmtstr (link-ref fmt)) x)
		  (multiple-value-list (funcall (funcall (obj-data-value-func fmt) x) x))))))
    refstr))
  
;;; Display method for objects


(defgeneric load-all-subobjects (objs))
(defmethod load-all-subobjects (objs)
  "Load all subobjects if they have not already been loaded."
  (when objs
    (let ((objlist (mklist objs)))
      (dolist (obj objlist)
        (awhen (ho-subobjects obj)  ;; access list of functions
          (dolist (subobj it)   ;; for each child function
            (awhen (funcall (reader subobj) obj)
              (load-all-subobjects it))))))
    objs))

(defgeneric print-hyperobject-class (objs fmt strm
				  &optional label english-only-function
				  indent subobjects refvars))

(defmethod print-hyperobject-class (objs (fmt dataformat) (strm stream) 
				 &optional (label nil) (indent 0)
				 (english-only-function nil)
				 (subobjects nil) (refvars nil))
"Display a single or list of hyperobject-class instances and their subobjects"
  (when objs
    (setq objs (mklist objs))
    (let ((nobjs (length objs)))
      (fmt-list-start (car objs) fmt strm indent nobjs)
      (dolist (obj objs)
        (unless (and english-only-function
		  (multiple-value-bind (eng term) (funcall english-only-function obj)
		    (and term (not eng))))
          (fmt-obj-start obj fmt strm indent)
          (fmt-obj-data obj fmt strm (1+ indent) label refvars)
          (if subobjects
              (awhen (ho-subobjects obj)  ;; access list of functions
		     (dolist (subobj it)   ;; for each child function
		       (awhen (funcall (reader subobj) obj) ;; access set of child objects
			      (print-hyperobject-class it fmt strm label 
						       (1+ indent) english-only-function
						       subobjects refvars)))))
          (fmt-obj-end obj fmt strm indent)))
      (fmt-list-end (car objs) fmt strm indent nobjs))
    t))



(defun print-hyperobject (objs &key (os *standard-output*) (format :text)
		      (label nil) (english-only-function nil)
		      (subobjects nil) (file-wrapper t) (refvars nil))
  "EXPORTED Function: prints hyperobject-class objects. Simplies call to print-hyperobject-class"
  (let ((fmt (make-format-instance format)))
    (if file-wrapper
	(fmt-file-start fmt os))
    (when objs
      (print-hyperobject-class objs fmt os label 0 english-only-function subobjects refvars))
    (if file-wrapper
	(fmt-file-end fmt os)))
  objs)


(defmethod print-object ((obj hyperobject) (s stream))
  (print-unreadable-object (obj s :type t :identity t)
    (let ((fmt (make-instance 'hyperobject::textformat)))
      (apply #'format 
	     s (funcall (hyperobject::obj-data-fmtstr fmt) obj)
	     (multiple-value-list 
	      (funcall (funcall (hyperobject::obj-data-value-func fmt) obj) obj))))))

