#include "cp_types.h"
#include "cp_proto.h"

/* Given a red chain, want to cut 'inclusions' and reentrant
peninsulas to get a new redchain. E.g., if p is simply connected,
can cookie this new redchain to get smallest simply connected 
sub-complex containing the original red chain. (I don't know
what might happen when p is not simply connected.)

Idea is to walk around outside of redchain and find any shortcuts. 
To identify the "outside", which parts of redchain are kept and
which removed as inclusions, we use 'lifeline', 'keep_vert', and
redchain entry 'keep_ptr'. The later two are 'persistent', should 
remain in final redchain. lifeline will be NULL if keep_vert is
in bdry of p. When p is top sphere, lifeline should run to some
designated outside vert; otherwise, should run to bdry; verts
along lifeline should end up outside of the final redchain.

Catalog of p must be stored and destroyed by calling program;
we use util_flag. Return 0 on error. */

extern int **face_org;

int simplify_redchain(struct p_data *p,struct RedList **redlist,
		      struct Edgelist *lifeline,int keep_vert,
		      struct RedList *keep_ptr)
{
  int i,n,num,dum,n_fan,new_vert,up_new_vert;
  int focus_vert,up_vert,down_vert,v1,v2,u;
  int down_indx,up_indx,indx,indx1,indx2;
  int trip_flag=1,hit_flag,begin_flag,end_flag,last_flag,bflag,*f_util;
  int db=0;
  struct RedList *rtrace,*ftrace,*stop_ptr,*next_red,*ftc;
  struct RedList *fan,*fan2,*focus_red;
  struct RedList *kill_start,*kill_stop,*up_red;
  struct Edgelist *etrace;
  struct K_data *pK_ptr;

  if (!p->status || (*redlist)==NULL || !keep_ptr || !keep_vert
      || find_index(p,keep_ptr->face,keep_vert)<0
      || (keep_vert!=red_share_vert(p,keep_ptr) 
	  && keep_vert!=red_share_vert(p,keep_ptr->prev)
	  && keep_ptr->next->face!=keep_ptr->prev->face) ) 
    return 0;

  pK_ptr=p->packK_ptr;

    /* persistent keep_vert must be on outside of keep_ptr face
       NOTE: lifeline must be arranged by the calling routine; eg,
       it can't cross redchain, should have at most one vert
       (namely at one end), etc. */

  /* mark lifeline */
  for (n=1;n<=p->nodecount;n++) pK_ptr[n].util_flag=0;
  etrace=lifeline;
  while (etrace)
    {
      pK_ptr[etrace->v].util_flag=pK_ptr[etrace->w].util_flag=1;
      etrace=etrace->next;
    }

  /* vector to keep track of faces currently in redchain */
  f_util=(int *)calloc((size_t)(p->facecount+1),sizeof(int));
  f_util[(*redlist)->face]=1;
  ftrace=(*redlist)->next;
  while (ftrace!=(*redlist))
    {
      f_util[ftrace->face]=1;
      ftrace=ftrace->next;
    }

/* Modify red chain to fill 'inclusions'. Method: walk around 
   outer edge verts and look for 'shortcuts': when vert is
   repeated (outside the current fan of red faces) or when a
   nghb of vert is encounterd. Discard the intervening red chain.
   (Stop search when you reach the keep_ptr to ensure that keep_ptr
   is kept). Repeat as needed. Resulting red chain should define 
   simply connected sub-complex (assuming p is simply connected).
*/
   
 if (db)
   {
     etrace=lifeline;
     while (etrace)
       {
	 draw_edge(p,etrace->v,etrace->w,255,1);
	 etrace=etrace->next;
       }
     ftrace=*redlist;
     while(ftrace->next!=*redlist)
       {
	 draw_any_face(p,ftrace->face,33,1,1,1);
	 ftrace=ftrace->next;
       }
     
   }

/* ------------------------ main loop ------------------*/

   while (trip_flag) /* main loop: go until pruning done */
    {
      trip_flag=0;
      stop_ptr=keep_ptr;
      
      /* focus_red, focus_vert are fixed while we search for 
	 shortcuts to them; start from keep_vert. */
      
      focus_red=keep_ptr;
      focus_vert=keep_vert;
      /* collect some local info around focus; set focus_red
	 as far down stream as possible with focus_vert as
	 a vertex. */
      up_red=upstream_red(p,focus_vert,focus_red,&up_indx);
      focus_red=downstream_red(p,focus_vert,focus_red,&down_indx);

if (db)
  {
    draw_any_face(p,focus_red->face,3,1,1,1);
  }

/* ------------------------ outer loop ------------------*/
      do
	{
if (db)
  {
    draw_any_face(p,focus_red->face,3,1,1,1);
  }
	  /* look for shortcuts; start with new_vert strictly
	     downstream of verts in focus_red->face. */
	  rtrace=focus_red;
	  new_vert=red_share_vert(p,focus_red);
	  /* have to know when to stop search; namely, when 
	     keep_ptr is encountered. Complications: may run
	     into keep_ptr right at beginning or right at end.
	     Set up 'begin_flag' and 'end_flag'; former is
	     set only on first pass, latter influences last
	     pass. Their states depend on relationship
	     of focus_red and keep_ptr. */

	  if (focus_red==keep_ptr) begin_flag=1;
	  else begin_flag=0;
	  if (keep_ptr==upstream_red(p,keep_vert,keep_ptr,&dum))
	    end_flag=1;
	  else end_flag=0;
	  last_flag=0;
	    /* this means that rtrace could end up matching 
	       keep_ptr when we reach the end of search for
	       shortcuts. In this 'while' call, last_flag gets
	       set only if end_flag is set and rtrace==keep_ptr. */
	       
/* ------------------------ inner loop ------------------*/

	  while (!last_flag &&
		 (new_vert=jump_down_red(p,rtrace,new_vert,&rtrace,
					 keep_ptr,&hit_flag))
		 && new_vert
		 && (!hit_flag || begin_flag 
		     || (last_flag=(end_flag && (rtrace==keep_ptr)
				    && red_share_vert(p,rtrace->prev)
				    !=keep_vert))))
	    /* summary: keep going if: haven't set last_flag; have found
	       new_vert to check for shortcut; didn't encounter keep_ptr,
	       or if we did, it's okay because: it's the first pass and
	       we knew we would encounter it, or we hit it just at the
	       end and it's a valid shortcut (don't screw up keep_vert).*/
	    {

	      begin_flag=0;
	      
	      /* Various criteria disqualify shortcuts; some of
		 these are also error checks. */

	      indx=nghb(p,focus_vert,new_vert);
	      if (new_vert!=focus_vert && indx<0)
		goto CONT;
	      if (rtrace==focus_red || rtrace==up_red
		  || rtrace->face==focus_red->face
		  ||(dum=find_index(p,rtrace->face,new_vert))<0)
		goto CONT;
	      if ((up_new_vert=p->faces[rtrace->face].vert[(dum+2)%3])
		  ==focus_vert)
		goto CONT; /* shouldn't happen */
	      /*	      if (new_vert!=focus_vert
			      && nghb(p,focus_vert,up_new_vert)==down_indx)
		goto CONT; */
	      down_vert=pK_ptr[focus_vert].flower[down_indx];
	      if (ck_in_elist(lifeline,new_vert,up_new_vert)
		  || ck_in_elist(lifeline,focus_vert,down_vert))
		goto CONT; /* first or last face crosses lifeline */
	      num=pK_ptr[focus_vert].num;
	      if (pK_ptr[focus_vert].bdry_flag) /* bdry case */
		{
		  if (new_vert==focus_vert 
		      && ((dum=nghb(p,focus_vert,up_new_vert))<0 
		      || dum>down_indx))
		    goto CONT;
		  else if (nghb(p,focus_vert,new_vert)>=down_indx)
		    goto CONT;
		}
	      else if (new_vert!=focus_vert) /* interior petal case */
		{
		  /* new connections to petals between down_vert
		     and up_vert (including down_vert, which would
		     be taken care of later) aren't allowed */
		  if (up_indx==down_indx) goto CONT;
		  dum=up_indx;
		  if (up_indx<down_indx) dum += num;
		  if (indx<down_indx) indx += num;
		  if (indx<=dum) 
		    goto CONT;
		}

	      /* Now, do shortcut */

	      kill_start=kill_stop=NULL;

	      if (new_vert==focus_vert) 
		/* may be direct shortcut; have further checking to do */
		{
                  indx=nghb(p,focus_vert,up_new_vert); 
		  fan=build_fan(p,focus_vert,indx,down_indx,&bflag,&n_fan);
		  if (bflag && fan) free_redfaces(&fan);
		  if (!bflag) /* Note: avoid problems, like bdry 
				 verts ending up on inside of redchain. */
		    {
		      kill_start=focus_red->next;
		      kill_stop=rtrace->prev;
		    }
		}
	      else 
		/* have to bridge across gap using two fans; 
		   first fan is clockwise from down_red until we
		   get a face containing new_vert, then clockwise
		   around new_vert until we link up to rtrace. */
		{
                  up_vert=pK_ptr[focus_vert].flower[up_indx]; 
                  if(upstream_red(p,up_vert,up_red,&dum)==rtrace) 
		    goto CONT;
		  num=pK_ptr[focus_vert].num;
		  fan=build_fan(p,focus_vert,indx,down_indx,&bflag,&n_fan);
		  if (bflag && fan) free_redfaces(&fan);
		  if (!bflag && fan) /* should get non-empty fan */
		    {
		      u=pK_ptr[focus_vert].flower[(indx+1)%num];
		      indx2=nghb(p,new_vert,u);
		      indx1=nghb(p,new_vert,up_new_vert);
		      fan2=build_fan(p,new_vert,indx1,indx2,&bflag,&n);
		      n_fan += n;
		      if (!bflag && fan2) /* concatenate fans */
			{
			  ftrace=fan;
			  while(ftrace->next && ftrace->next!=fan) 
			    ftrace=ftrace->next;
			  ftrace->next=fan2;
			  fan2->prev=ftrace;
			  ftrace=fan2;
			  while(ftrace->next && ftrace->next!=fan2)
			    ftrace=ftrace->next;
			  ftrace->next=fan;
			  fan->prev=ftrace;
			}
		      kill_start=focus_red->next;
		      kill_stop=rtrace->prev;
		    }
		}

	      if (kill_start) /* we have a shortcut ready */
		{
		  if (n_fan) /* insert fan */
		    {
		      /* check fan to see if it picks up face 
			 already in a different section of the 
			 redchain. */
		      ftc=fan;
		      for (i=1;i<=n_fan;i++)
			{
			  if (f_util[ftc->face]) /* face already 
						    in redchain */
			    {
			      /* if it's in the part of redchain being 
				 cut out, okay */
			      dum=0;
			      ftrace=kill_start;
			      while (ftrace->next!=rtrace 
				     && !(dum=(ftrace->face==ftc->face)))
				ftrace=ftrace->next;
			      if (!dum) /* didn't find it, so scrap
					   this shortcut */
				{
				  free_redfaces(&fan);
				  goto CONT;
				}
			    }
			  ftc=ftc->next;
			}
		      /* check fan to see if it crosses lifeline */
		      if (lifeline)
			{
			  ftrace=fan;
			  while (ftrace->next!=fan)
			    {
			      indx=nghb_tri(p,ftrace->next->face,ftrace->face);
			      v1=p->faces[ftrace->face].vert[indx];
			      v2=p->faces[ftrace->face].vert[(indx+1)%3];
			      if (ck_in_elist(lifeline,v1,v2))
				{
				  free_redfaces(&fan);
				  goto CONT;
				}
			      ftrace=ftrace->next;
			    }
			}
		      /* okay, so mark these faces in f_util */
		      f_util[fan->face]=1;
		      ftrace=fan->next;
		      while (ftrace!=fan)
			{
			  f_util[ftrace->face]=1;
			  ftrace=ftrace->next;
			}

		      /* now, hook in shortcut */
		      focus_red->next=fan;
		      fan->prev=focus_red;
		      ftrace=fan;
		      while(ftrace->next && ftrace->next!=fan) 
			ftrace=ftrace->next;
		      ftrace->next=rtrace;
		      rtrace->prev=ftrace;
		    }
		  else /* nothing to add, just short out */
		    {
		      focus_red->next=rtrace;
		      rtrace->prev=focus_red;
		    }

		  /* reset f_util for faces no longer in redchain */
		  kill_stop->next=NULL;
		  ftrace=kill_start;
		  while (ftrace)
		    {
		      f_util[ftrace->face]=0;
		      ftrace=ftrace->next;
		    }

		  /* close up and free kill chain */
		  kill_start->prev=kill_stop;
		  kill_stop->next=kill_start;
		  free_redfaces(&kill_start);

		  /* indicate that we did a shortcut */
		  trip_flag++; 

		  /* update info around focus_vert and check
		     settings of flags concerning search. */
		  next_red=downstream_red(p,focus_vert,focus_red,&down_indx);
		  if (next_red==focus_red) /* shouldn't happen; 
					      stop the inner loop. */
		    last_flag=1; 
		  focus_red=next_red;
		  if (focus_red==keep_ptr) /* reset begin_flag because
					      we can continue looping
					      with the same focus */
		    begin_flag=1;
		  /* continue search strictly downstream */
		  rtrace=focus_red;
		  new_vert=red_share_vert(p,focus_red);
		}
	    CONT:
	      continue;
	    } /* end of 'jump' while; focus doesn't change yet,
		 we might find other shortcuts for it. */
	  
	  /* jump to change the focus, stopping if keep_ptr crossed. */
	  focus_vert=jump_down_red(p,focus_red,focus_vert,
				   &focus_red,keep_ptr,&hit_flag);

	  if (focus_vert && !hit_flag) 
	    /* update local info around focus, checking to see
	       if we pass keep_ptr. */
	    {
	      up_red=upstream_red(p,focus_vert,focus_red,&up_indx);
	      next_red=downstream_red(p,focus_vert,focus_red,&down_indx);
	      ftrace=focus_red;
	      while (ftrace!=next_red)
		{
		  if (ftrace==keep_ptr)
		    hit_flag=1;
		  ftrace=ftrace->next;
		}
	      focus_red=next_red;
	    }

	} while (focus_vert && !hit_flag);
	/* keep going until you get back to the beginning or while
	   shortcuts are being found */
    } /* end of outer while; didn't make any changes on last pass. */

  if (f_util) free(f_util); 
  return 1;
} /* simplify_redchain */

struct RedList *upstream_red(struct p_data *p,int vert,
			     struct RedList *redlist,int *up_indx)
     /* Find furthest contiguous upstream face in redlist containing vert.
      Return NULL on error or if redchain encircles single vert. */
{
  int cface,pface,indx,v;
  struct RedList *trace;

  if (!redlist || find_index(p,(cface=redlist->face),vert)<0) 
    return NULL;
  if ((redlist->next->face==redlist->prev->face)) /* blue? */
    {
      v=red_share_vert(p,redlist->prev);
      indx=find_index(p,cface,v);
      if (vert==p->faces[cface].vert[(indx+1)%3])
	{
	  *up_indx=nghb(p,vert,v);
	  return redlist;
	}
      else if (vert==p->faces[cface].vert[(indx+2)%3])
	{
	  *up_indx=nghb(p,vert,p->faces[cface].vert[(indx+1)%3]);
	  return redlist;
	}
    }
  else if (redlist->face==redlist->prev->prev->face
	   && vert==red_share_vert(p,redlist->prev)) /* prev is blue */
    {
      indx=find_index(p,redlist->prev->face,vert);
      *up_indx=nghb(p,vert,p->faces[redlist->prev->face].vert[(indx+2)%3]);
      return redlist->prev;
    }
  trace=redlist;
  pface=trace->prev->face;
  while ((vert==red_share_vert(p,trace->prev)) && pface!=cface)
    {
      trace=trace->prev;
      pface=trace->prev->face;
    }
  if (pface==cface) /* red chain encircles vert */
    return NULL; 
  indx=find_index(p,trace->face,vert);
  *up_indx=nghb(p,vert,p->faces[trace->face].vert[(indx+2)%3]);
  return trace;
} /* upstream_red */

struct RedList *downstream_red(struct p_data *p,int vert,
			       struct RedList *redlist,int *down_indx)
     /* Find furthest contiguous downstream face in redlist containing vert.
      Return NULL on error or if redchain encircles single vert. */
{
  int cface,nface,indx,v;
  struct RedList *trace;

  if (!redlist || find_index(p,(cface=redlist->face),vert)<0)
    return NULL;
  if ((redlist->next->face==redlist->prev->face)) /* blue? */
    {
      v=red_share_vert(p,redlist);
      indx=find_index(p,cface,v);
      if (vert==p->faces[cface].vert[(indx+1)%3])
	{
	  *down_indx=nghb(p,vert,p->faces[cface].vert[(indx+2)%3]);
	  return redlist;
	}
      else if (vert==p->faces[cface].vert[(indx+2)%3])
	{
	  *down_indx=nghb(p,vert,v);
	  return redlist;
	}
    }
  else if (redlist->face==redlist->next->next->face
	   && vert==red_share_vert(p,redlist)) /* next is blue */
    {
      indx=find_index(p,redlist->next->face,vert);
      *down_indx=nghb(p,vert,p->faces[redlist->next->face].vert[(indx+1)%3]);
      return redlist->next;
    }
  trace=redlist;
  nface=trace->next->face;
  while ((vert==red_share_vert(p,trace)) && nface!=cface)
    {
      trace=trace->next;
      nface=trace->next->face;
    }
  if (nface==cface) /* red chain encircles vert */
    return NULL;
  indx=find_index(p,trace->face,vert);
  *down_indx=nghb(p,vert,p->faces[trace->face].vert[(indx+1)%3]);
  return trace;
} /* downstream_red */

int jump_down_red(struct p_data *p,struct RedList *start,int start_vert,
		  struct RedList **new_ptr,struct RedList *keep_ptr,
		  int *hit_flag)
     /* Coming in, start should be face having edge in outside of red
	chain and start_vert should be downstream end of that edge.
	Have to be sure to check blue face situation, where two
	vertices of face can satisfy this.

	Move down redlist until new outside vert is found. Set new_ptr
	to new redlist entry, return vert. Set hit_flag if keep_ptr is 
	encountered; return 0 on error. */
{
  int n,f,indx;
  struct RedList *ptr;

  *hit_flag=0;
  if (start->next->face==start->prev->face) /* at blue face? */
    {
      f=start->face;
      indx=find_index(p,f,start_vert);
      if (start_vert!=red_share_vert(p,start)) /* at isolated vert */
	{
	  n=p->faces[f].vert[(indx+1)%3];
	  *new_ptr=start;
	  if (start==keep_ptr) /* encountered keep_ptr */
	    *hit_flag=1;
	  return n;
	}
    }
  ptr=start;
  while ((n=red_share_vert(p,ptr))==start_vert)
    {
      if (ptr==keep_ptr) *hit_flag=1;
      ptr=ptr->next;
    }
  *new_ptr=ptr;
  if (ptr->next->face==ptr->prev->face) /* new one is blue? return its
					   isolated vert. */
    {
      f=ptr->face;
      n=p->faces[f].vert[(find_index(p,f,n)+2)%3];
    }
  return n;
} /* jump_down_red */

struct RedList *build_fan(struct p_data *p,int vert,int indx1,
			  int indx2,int *flag,int *count)
     /* build fan about vert between indices; petal verts must be 
	interior; return count and set flag if bdry petal encountered. */
{
  int j,num,*face_ptr;
  struct RedList *fan,*rtrace;
  
  *flag=*count=0;
  num=p->packK_ptr[vert].num;
  if (p->packK_ptr[vert].bdry_flag && indx1>indx2) /* bdry, wrong order */
    {
      *flag=1;
      return NULL;
    }
  if (indx1==indx2) return NULL; /* okay, but count is 0 */
  if (indx2<indx1) indx2 += num;
  j=indx1+1;
  while (j<=indx2)
    {
      if (p->packK_ptr[p->packK_ptr[vert].flower[j%num]].bdry_flag)
	*flag=1;
      (*count)++;
      j++;
    }
  
  /* creat fan as closed linked list */
  
  fan=(struct RedList *)calloc((size_t)1,sizeof(struct RedList));
  face_ptr=face_org[vert]+1;
  fan->face=face_ptr[(indx2+num-1)%num];
  rtrace=fan;
  for (j=indx2-1;j>indx1;j--)
    {
      rtrace->next=(struct RedList *)
	calloc((size_t)1,sizeof(struct RedList));
      rtrace->next->prev=rtrace;
      rtrace=rtrace->next;
      rtrace->face=face_ptr[(j+num-1)%num];
    }
  fan->prev=rtrace;
  rtrace->next=fan;
  return fan;
} /* build_fan */

int ck_in_elist(struct Edgelist *elist,int v,int w)
     /* check if edge <v,w> is in given edge-list */
{
  struct Edgelist *etrace;

  if (!elist) return 0;
  etrace=elist;
  while (etrace)
    {
      if ((etrace->v==v && etrace->w==w) 
	  || (etrace->v==w && etrace->w==v) )
	return 1;
      etrace=etrace->next;
    }
  return 0;
} /* ck_in_elist */
      
