/* sph_math.c    generic spherical geometry routines */

/* gcc sph_math.c -o sph_math.o -c -g -lm */

#include <stdio.h>
#include <math.h>
#include "complex_math.h"
#include "hyp_math.h"
#include "eucl_math.h"
#include "sph_math.h"
#define OKERR 0.0000000001

/* sphere is {x^2+y^2+z^2=1}. Store centers as complex with
form (theta,phi), radii are spherical.*/

struct Pathlist
 {
	float x,y;		/* coords of point */
	struct Pathlist *next;
 };

int
tangent(ctr1,ctr2,T) /* Given pts on sphere, return ptr to 3-vector
in tangent space of first, pointing to second. */
complex ctr1,ctr2;
float *T;
{

	float x, y ,z ,D, N, A[3], B[3];

	s_pt_to_vec(ctr1,A);
	s_pt_to_vec(ctr2,B);
	D = (A[0]*B[1]-A[1]*B[0])*(A[0]*B[1]-A[1]*B[0])+
		(A[1]*B[2]-A[2]*B[1])*(A[1]*B[2]-A[2]*B[1])+
		(A[0]*B[2]-A[2]*B[0])*(A[0]*B[2]-A[2]*B[0]);            
               
 /* first, need point in the intersection of the tangent spaces */

  if (fabs((A[0]*B[1]-A[1]*B[0]))>OKERR) 
   {
    z=((A[1]-B[1]+A[0]*(A[0]*B[1]-A[1]*B[0]))*(A[1]*B[2]-A[2]*B[1])+
      (B[0]-A[0]+A[1]*(A[0]*B[1]-A[1]*B[0]))*(A[2]*B[0]-A[0]*B[2])+
      A[2]*(A[0]*B[1]-A[1]*B[0])*(A[0]*B[1]-A[1]*B[0]))/D;

    x=((B[1]-A[1])+(A[1]*B[2]-A[2]*B[1])*z)/(A[0]*B[1]-A[1]*B[0]);

    y=((A[0]-B[0])+(A[2]*B[0]-A[0]*B[2])*z)/(A[0]*B[1]-A[1]*B[0]);

   }
   
   else if (fabs((A[1]*B[2]-A[2]*B[1]))>OKERR)
    {
     x=((A[0]*B[1]-A[1]*B[0])*(A[2]*(A[1]*B[2]-A[2]*B[1])+(B[1]-A[1]))+
       (A[2]*B[0]-A[0]*B[2])*(A[1]*(A[1]*B[2]-A[2]*B[1])+(A[2]-B[2]))+
       A[0]*(A[1]*B[2]-A[2]*B[1])*(A[1]*B[2]-A[2]*B[1]))/D;
 
     y=((B[2]-A[2])+(A[2]*B[0]-A[0]*B[2])*x)/(A[1]*B[2]-A[2]*B[1]);

     z=((A[1]-B[1])+(A[0]*B[1]-B[0]*A[1])*x)/(A[1]*B[2]-A[2]*B[1]);

    }
   else
    {
      y=A[1];

      x=((B[2]-A[2])+(A[2]*B[1]-A[1]*B[2])*y)/(A[0]*B[2]-A[2]*B[0]);

      z=((A[0]-B[0])+(A[1]*B[0]-A[0]*B[1])*y)/(A[0]*B[2]-A[2]*B[0]);
     }
    N=sqrt((x-A[0])*(x-A[0])+(y-A[1])*(y-A[1])+(z-A[2])*(z-A[2]));
          
           /* return a vector giving the required unit tangent vector */

     T[0] = (x-A[0])/N;
     T[1] = (y-A[1])/N;
     T[2] = (z-A[2])/N;
     return 1;
    
} /* tangent */

int 
s_compcenter(z1,z2,z3,r1,r2,r3,o1,o2,o3) /* Find center of third circle
in ordered tangent triple. Note: orientation is clockwise looking at
sphere from outside. Overlaps not yet used. (r3 is ptr only for compat 
reasons.) */ 
float r1,r2,*r3,o1,o2,o3;
complex z1,z2,*z3;
{

  float a,b,c,h,j,k,angle;
  float TV[3], P[3], N[3];
  float mtan[3];

  a=sin(z1.im)*cos(z1.re);
  b=sin(z1.im)*sin(z1.re);
  c=cos(z1.im);
  h=sin(z2.im)*cos(z2.re);
  j=sin(z2.im)*sin(z2.re);
  k=cos(z2.im);
/* angle is how far around from T we will rotate */
  angle=2*M_PI-
	acos(( cos(r2+(*r3))-cos(r1+(*r3))*cos(r1+r2) )/
	( sin(r1+(*r3))*sin(r1+r2) ));
/* TV is a tangent vector at z1 */
  tangent(z1,z2,TV);
/* N is T x z1 */
  N[0]=b*TV[2]-c*TV[1];
  N[1]=c*TV[0]-a*TV[2];
  N[2]=a*TV[1]-b*TV[0];
/* P will point at the new center */
  P[0]=cos(angle)*TV[0]+sin(angle)*N[0];
  P[1]=cos(angle)*TV[1]+sin(angle)*N[1];
  P[2]=cos(angle)*TV[2]+sin(angle)*N[2];
  *mtan=cos(r1+(*r3))*a+sin(r1+(*r3))*P[0];
  *(mtan+1)=cos(r1+(*r3))*b+sin(r1+(*r3))*P[1];
  *(mtan+2)=cos(r1+(*r3))*c+sin(r1+(*r3))*P[2];
  if(mtan[2]>(1.0-OKERR))
    {z3->im=z3->re=0.0; return 1;}
  z3->im=acos(mtan[2]);
  z3->re=aTan2(mtan[1],mtan[0]);
  return 1;
} /* s_compcenter */

float
s_comp_cos(r1,r2,r3) /* given ordered triple of radii, compute
the cosine of the angle at first circle in triangle formed by
mutually tangent circles. */
float r1,r2,r3;
{
	float ans;

	ans=(cos(r2+r3)-(cos(r1+r2)*cos(r1+r3)))/(sin(r1+r2)*sin(r1+r3));
	return (ans);
} /* s_comp_cos */

struct Pathlist *
s_geodesic(z1,z2,num_plot)  /* returns pointer to list of pts on geodesic
which are on front of (apparent) sphere. num_plot is no. of divisions
in plotting curve. */
complex z1,z2;
int num_plot;
{
	int gplot=(num_plot/2);
	float T[3], P[3], angle;
	int n;
	extern int tangent();
	struct Pathlist *trace,*pathlist;
        int i;
	int length=0;

        pathlist=
	   (struct Pathlist *)malloc(sizeof(struct Pathlist));
	pathlist->next=NULL;
  angle=s_dist(z1,z2);
  tangent(z1,z2,T);
  n=0;
  while (n<=gplot)
   {
     P[0]=cos((n*angle)/gplot)*sin(z1.im)*cos(z1.re)+
		sin((n*angle)/gplot)*T[0];
     P[1]=cos((n*angle)/gplot)*sin(z1.im)*sin(z1.re)+
		sin((n*angle)/gplot)*T[1];
     P[2]=cos((n*angle)/gplot)*cos(z1.im)+
		sin((n*angle)/gplot)*T[2];
    
     if (length==0 && P[0]>-OKERR)
       {
         pathlist->x=P[1];pathlist->y=P[2];
         pathlist->next=NULL;
	 length=1;
	 trace=pathlist;
        }
      else if (length>0 && P[0]>-OKERR)
        {
         trace->next=(struct Pathlist *)malloc(sizeof(struct Pathlist)); 
         trace->next->next=NULL;
         trace=trace->next;
         trace->x=P[1];trace->y=P[2];		
         length++;
        }
       n++;
}
	if (length==0) {free(pathlist);return NULL;}
     return pathlist;
  } /* s_geodesic */

struct Pathlist *
full_s_geodesic(z1,z2,num_plot) /* returns pointer to list of sph pts on whole 
geodesic; for use with closed faces. */
complex z1,z2;
int num_plot;
{
	int gplot=num_plot/2;
	float T[3], P[3], angle;
	int n;
	struct Pathlist *trace,*pathlist;
        int i;
	int length=0;
	complex pt;

        pathlist=
	   (struct Pathlist *)malloc(sizeof(struct Pathlist));
	pathlist->next=NULL;
  angle=s_dist(z1,z2);
  tangent(z1,z2,T);
  n=0;
  while (n<gplot)
   {
     P[0]=cos((n*angle)/gplot)*sin(z1.im)*cos(z1.re)+
		sin((n*angle)/gplot)*T[0];
     P[1]=cos((n*angle)/gplot)*sin(z1.im)*sin(z1.re)+
		sin((n*angle)/gplot)*T[1];
     P[2]=cos((n*angle)/gplot)*cos(z1.im)+
		sin((n*angle)/gplot)*T[2];
	pt=proj_vec_to_sph(P[0],P[1],P[2]);
    
     if (length==0)
       {
         pathlist->x=pt.re;pathlist->y=pt.im;
         pathlist->next=NULL;
	 length=1;
	 trace=pathlist;
        }
      else if (length>0)
        {
         trace->next=(struct Pathlist *)malloc(sizeof(struct Pathlist)); 
         trace->next->next=NULL;
         trace=trace->next;
         trace->x=pt.re;trace->y=pt.im;		
         length++;
        }
       n++;
}
	if (length==0) {free(pathlist);return NULL;}
     return pathlist;
  } /* full_s_geodesic */

struct Pathlist *
s_circle_list(ctr,r,num_plot) /* return oriented list of sph pts for circle. */
float r;
complex ctr;
int num_plot;
{
	int i;
	float A[3],B[3],C[3],P[3];
	float step,ss,a11,a12,a21,a22,zr,zi,wr,wi,dist,rad,norm;
	complex pt;
	struct Pathlist *outlist,*trace;

	rad=sin(r);
	dist=cos(r);
	s_pt_to_vec(ctr,C);
		/* find pair of orthog vecs */
	if (fabs(C[0])<0.3)
	 {A[0]=0.0;A[1]=(-1.0)*C[2];A[2]=C[1];}
	else
	 {A[0]=(-1.0)*C[1];A[1]=C[0];A[2]=0.0;}
	cross_prod(A,C,B);
/*
	if (r<= (M_PI/2.0)) cross_prod(A,C,B);
	else cross_prod(C,A,B);
*/ /* non-convex circle */
	C[0]=C[0]*dist;C[1]=C[1]*dist;C[2]=C[2]*dist;
	norm=vec_norm(A);
	A[0]=A[0]/norm;A[1]=A[1]/norm;A[2]=A[2]/norm;
	norm=vec_norm(B);
	B[0]=B[0]/norm;B[1]=B[1]/norm;B[2]=B[2]/norm;
		/* set up matrix to propogate circle */
	step=2.0*M_PI/num_plot;
	ss=step*step/4;
	a11=(1-ss)/(1+ss);a12= (-step)/(1+ss);a21= (-a12);a22=a11; 
	zr=rad;zi=0.0;
	outlist=(struct Pathlist *)malloc(sizeof(struct Pathlist));
	outlist->next=NULL;
	trace=outlist;
	for (i=0;i<=num_plot;i++)
	 {
		P[0]=C[0]+zr*A[0]+zi*B[0];
		P[1]=C[1]+zr*A[1]+zi*B[1];
		P[2]=C[2]+zr*A[2]+zi*B[2];
		pt=proj_vec_to_sph(P[0],P[1],P[2]);
		trace->x=pt.re;trace->y=pt.im;
		trace->next=
			(struct Pathlist *)malloc(sizeof(struct Pathlist));
		trace=trace->next;
		trace->next=NULL;
		wr=zr;
		wi=zi;
		zr=a11*wr+a12*wi;
		zi=a21*wr+a22*wi;
	 }
	trace->x=outlist->x;trace->y=outlist->y;
	return (outlist);
} /* s_circle_list */

struct Pathlist *
sph_tri_list(z0,z1,z2,num_plot) /* clockw' list of geodesics in face */
complex z0,z1,z2;
int num_plot;
{
	struct Pathlist *firstlist,*nlist,*trace;

	if ((firstlist=full_s_geodesic(z0,z1,num_plot))==NULL) return NULL;
	if ((nlist=full_s_geodesic(z1,z2,num_plot))==NULL) 
	 {path_free(&firstlist);return NULL;}
	trace=firstlist;
	while (trace->next!=NULL) trace=trace->next;
	trace=trace->next=(struct Pathlist *)malloc(sizeof(struct Pathlist));
	trace->x=nlist->x;trace->y=nlist->y; 
	trace->next=nlist;/* link in nlist */
	if ((nlist=full_s_geodesic(z2,z0,num_plot))==NULL) 
	 {path_free(&firstlist);return NULL;}
	while (trace->next!=NULL) trace=trace->next;
	trace=trace->next=(struct Pathlist *)malloc(sizeof(struct Pathlist));
	trace->x=nlist->x;trace->y=nlist->y; 
	trace->next=nlist;/* link in nlist */
	while (trace->next!=NULL) trace=trace->next;
	trace=trace->next=(struct Pathlist *)malloc(sizeof(struct Pathlist));;
	trace->x=firstlist->x;trace->y=firstlist->y;
	trace->next=NULL;
	return firstlist;
} /* sph_tri_list */
	
struct Pathlist *
fix_convex_list(inlist) /* given a neg oriented bdry list for
convex region on (apparent) sphere, return pos oriented list (projected
to screen plane) for display with 'fill' potential (i.e., if it 
goes over horizon, portion of horizon replaces that in back). */
struct Pathlist *inlist;
{
	int i,n;
	float alpha,beta,delta;
	float ac,step,ss,a11,a12,a21,a22,zr,zi,wr,wi;
	struct Pathlist *outlist,*trace,*last;
	struct Pathlist *ntrace,*nexttrace,*firsttrace;
	struct Pathlist *begin,*ending,*back;

	/* if (inlist==NULL || (inlist->next==NULL && free(inlist))) */
	if (inlist==NULL || (inlist->next==NULL )) 
		return NULL; /* too short */
/* whole list on front? */
	trace=inlist;
	while (trace!=NULL && cos(trace->x)>=(-1.0)*okerr) /* on front */
		trace=trace->next;
	if (trace==NULL) /* whole list on front, reverse and return */
	 {
		trace=inlist;
		ntrace=(struct Pathlist *)malloc(sizeof(struct Pathlist));
		ntrace->next=NULL;
		ntrace->x=sin(trace->y)*sin(trace->x); /* y coord */
		ntrace->y=cos(trace->y); /* z coord */
		while (trace->next!=NULL)
		 {
			trace=trace->next;
			nexttrace=
			   (struct Pathlist *)malloc(sizeof(struct Pathlist));
			nexttrace->next=ntrace;
			nexttrace->x=sin(trace->y)*sin(trace->x);/* y coord */
			nexttrace->y=cos(trace->y); /* z coord */
			ntrace=nexttrace;
		 }
		path_free(&inlist);
		return (nexttrace); 
	 }
	back=trace; 
/* whole list on back? */
	trace=inlist;
	while (trace!=NULL && cos(trace->x)<=okerr) /* on back */
		trace=trace->next;
	if (trace==NULL) /* all on back */
	 {
		path_free(&inlist);
		return NULL;
	 }
/* close up list temporarily */
	trace=begin=inlist;
	while (trace!=NULL)
	 {
		ending=trace;
		trace=trace->next;
	 }
	ending->next=begin;
/* find first on front */
	nexttrace=back;
	trace=back->next; /* 'back' is on back */
	while (trace!=back && cos(trace->x)<=okerr) 
	 {
		nexttrace=trace;
		trace=trace->next;
	 }
	if (trace==back) /* shouldn't happen */
	 {
		ending->next=NULL;
		path_free(&begin);
		return NULL;
	 }
/* readjust begin, end */
	begin=trace;
	ending=nexttrace;
	ending->next=NULL;
/* now find last on front */
	trace=last=begin;
	while (trace!=NULL && cos(trace->x)>=(-1.0)*okerr) /* on front */
	 {
		last=trace;
		trace=trace->next;
	 }
	path_free(&(last->next)); /* discard after last, on back */
	last->next=NULL;
/* rebuild list in reverse, putting in positive arc of horizon */
	trace=begin;
	ntrace=firsttrace=(struct Pathlist *)malloc(sizeof(struct Pathlist));
	ntrace->next=NULL;
	ntrace->x=sin(trace->y)*sin(trace->x); /* y coord */
	ntrace->y=cos(trace->y); /* z coord */
	while (trace->next!=NULL)  /* go begin to last */
	 {
		trace=trace->next;
		nexttrace=(struct Pathlist *)malloc(sizeof(struct Pathlist));
		nexttrace->next=ntrace;
		nexttrace->x=sin(trace->y)*sin(trace->x); /* y coord */
		nexttrace->y=cos(trace->y); /* z coord */
		ntrace=nexttrace;
	 }
	outlist=nexttrace;
	alpha=begin->y;
	if ((begin->x)>=0) alpha *=(-1.0);
	beta=last->y;
	if ((last->x)>=0) beta *=(-1.0);
	delta=beta-alpha;
	if (delta<0.0) delta +=2.0*M_PI;
		/* angles in (y,z)-plane mes'rd clockwise from south pole */
	n=delta/(2.0*M_PI)*10;
	n=n*10; /* n will be 0,10,20, etc. */
	if (n<10) n=10; /* tailor number of steps to proportion of circle */
	step=delta/n;
	ss=step*step/4;
	a11=(1-ss)/(1+ss);a12= (-step)/(1+ss);a21= (-a12);a22=a11; 
	zr=(-1.0)*sin(alpha);zi=cos(alpha);
	n++;
	ntrace=firsttrace;
	for (i=0;i<n;i++)
	{
		ntrace=ntrace->next=
		   (struct Pathlist *)malloc(sizeof(struct Pathlist));
		ntrace->next=NULL;
		ntrace->x=zr;
		ntrace->y=zi;
		wr=zr;
		wi=zi;
		zr=a11*wr+a12*wi;
		zi=a21*wr+a22*wi;
	} 
	path_free(&begin);
	return (outlist);
} /* fix_convex_list */

float
s_dist(z1,z2) /* spherical distance between pts */
complex z1,z2;
{
	float v1[3],v2[3],dotprod;

 	if ( (fabs(z1.re-z2.re)<OKERR) && (fabs(z1.im-z2.im)<OKERR))
   		return (0);
	s_pt_to_vec(z1,v1);
	s_pt_to_vec(z2,v2);
	dotprod= v1[0]*v2[0]+v1[1]*v2[1]+v1[2]*v2[2];
	if (fabs(dotprod)>(1.0-OKERR)) return M_PI;
	return acos(dotprod);
} /* s_dist */

s_pt_to_vec(z,V) /* convert (theta,phi) pt on sphere to unit 3 vector */
complex z;
float *V;
{
	V[0]=sin(z.im)*cos(z.re);
	V[1]=sin(z.im)*sin(z.re);
	V[2]=cos(z.im);
} /* s_pt_to_vec */

complex
s_pt_to_visual_plane(pt) /* return (y,z) of pt */
complex pt;
{
	complex z;

	z.im=cos(pt.im);
	z.re=sin(pt.im)*sin(pt.re);
	return z;
} /* s_pt_to_visual_plane */

complex
screen_to_s_pt(pt,hit) /* given pt on screen, find pt on front of sphere.
hit=1 if found */
int *hit;
complex pt;
{
	float V[3],xx;
	complex z;

	xx=1.0 - sqrt(pt.re*pt.re + pt.im*pt.im);
	if (xx<0)
	 {
		*hit=0;
		z.re=z.im=0.0;
		return z;
	 }
	V[0]=xx;
	V[1]=pt.re;
	V[2]=pt.im;
	*hit=1;
	return (proj_vec_to_sph(V[0],V[1],V[2]));
} /* screen_to_s_pt */

s_to_e_data(z,r,e,er) /* projects circles from sphere to plane */
complex z,*e;
float r,*er;
{
	float V[3],m,RR,rr,up,down;

	s_pt_to_vec(z,V);
	if (r>(M_PI/2.0)) /* use outer radius */
	 {
		r=M_PI-r;
		V[0] *= (-1.0);
		V[1] *= (-1.0);
		V[2] *= (-1.0);
		z=proj_vec_to_sph(V[0],V[1],V[2]);
	 }
	up=z.im-r;
	down=z.im+r;
	if (up <= 0.0) /* circle contains north pole */
	 {
		if (z.im < okerr)  /* essentially centered at np */
		 {
			if (r < okerr) /* huge eucl circle */
				*er=100000;
			else *er=sin(r)/(1.0-cos(r));
			e->re=e->im=0.0;
			return;
		 }
		RR=sin(up)/(1.0-cos(up));
		rr=sin(down)/(1.0-cos(down));
		*er=(rr-RR)/2.0;
		m=(RR+rr)/2.0;
		e->re=V[0]*m/sin(z.im);
		e->im=V[1]*m/sin(z.im);
		return;	
	 }
	if (down > M_PI) /* circle contains south pole */
	 {
		if (z.im>(M_PI-okerr) && z.im<(M_PI+okerr) )
		 {
			*er=sin(up)/(1.0-cos(up));
			e->re=e->im=0.0;
			return;
		 }
		RR=sin(up)/(1.0-cos(up));
		rr=sin(down)/(1.0-cos(down));
		*er=(RR-rr)/2.0;
		m=(RR+rr)/2.0;
		e->re=V[0]*m/sin(z.im);
		e->im=V[1]*m/sin(z.im);
		return;	
	 }
	RR=sin(up)/(1.0-cos(up));
	rr=sin(down)/(1.0-cos(down));
	*er=(RR-rr)/2.0;
	m=(RR+rr)/2.0;
	e->re=V[0]*m/sin(z.im);
	e->im=V[1]*m/sin(z.im);
	return;
} /* s_to_e_data */
		
complex
proj_vec_to_sph(x,y,z) /* radially project (x,y,z) to sphere,
origin to north pole */
float x,y,z;
{
	float dist;
	complex ans;

	if ((dist=sqrt(x*x+y*y+z*z))< OKERR) 
		{ans.re=ans.im=0.0;return ans;}
		/* default for things near origin */
	x /= dist;
	y /= dist;
	z /= dist;
	ans.re=aTan2(y,x);
	ans.im=acos(z);	
	return ans;
} /* proj_vec_to_sph */

int
e_to_s_data(z,r,s,sr) /* converts circle data via stereo proj 
from plane to sphere */
complex z,*s;
float r,*sr;
{
	extern int tangent();
	complex v,w,u;
	float norm, C[3], P1[3], P2[3],P3[3], T[3], rad,denom;
	float x,y,a,b;

	norm=sqrt(z.re*z.re+z.im*z.im);
	if (norm<OKERR) /* close to origin */
	 {
		x=-r;
		y=0.0;
		a=r;
		b=0.0;
	 }
 	else
	 {
		  /* a point on the circle closest to origin */
		x=((norm-r)*(z.re))/norm;
		y=((norm-r)*(z.im))/norm;
		  /* a point on the circle furthest from the origin */
		a=((norm+r)*(z.re))/norm;
		b=((norm+r)*(z.im))/norm;
	 }
		/* now we project these two points onto the sphere */
	denom=x*x + y*y +1;
	P1[0]=(2*x)/denom;  P1[1]=(2*y)/denom;  P1[2]=(denom-2)/denom;
	denom=a*a + b*b +1;
	P2[0]=(2*a)/denom;  P2[1]=(2*b)/denom;  P2[2]=(denom-2)/denom;
	if (r>(norm+OKERR)) /* use origin as pt along geo */
	 {P3[0]=P3[1]=0.0;P3[2]=-1.0;}
	else 		    /* use projection of center as pt along geo */
	 {
		denom=z.re*z.re+z.im*z.im+1;
		P3[0]=(2*z.re)/denom;  P3[1]=(2*z.im)/denom;  
		P3[2]=(denom-2)/denom;
	 }
		/* here is the spherical radius of the circle */
	rad=(acos(P1[0]*P3[0]+P1[1]*P3[1]+P1[2]*P3[2])
		+acos(P2[0]*P3[0]+P2[1]*P3[1]+P2[2]*P3[2]))/2;
		/* use this to find center along the geo from P1 to P2 */

	v.re=aTan2(P1[1],P1[0]);
	v.im=acos(P1[2]);
	w.re=aTan2(P2[1],P2[0]);
	w.im=acos(P2[2]);
	if (r>(norm+OKERR))
	 {
		if (rad>M_PI/2.0) 
		 {
			w.re=aTan2(P3[1],P3[0]);
			w.im=acos(P3[2]);
		 }
		tangent(v,w,T);
	 }
	else
	 {
		u.re=aTan2(P3[1],P3[0]);
		u.im=acos(P3[2]);
		tangent(v,u,T);
	 }
		/* C will be the rectangular coordinates of the center */

	C[0]=P1[0]*cos(rad)+T[0]*sin(rad);
	C[1]=P1[1]*cos(rad)+T[1]*sin(rad);
	C[2]=P1[2]*cos(rad)+T[2]*sin(rad);
  
	*sr=rad;
	if(C[2]>(1.0-OKERR))
		{s->re=s->im=0.0;return 1;}
	if (C[2]<(OKERR-1.0))
		{s->re=0.0;s->im=M_PI;return 1;}
	s->im=acos(C[2]);
	s->re=aTan2(C[1],C[0]);
	return 1;
}/* e_to_s_data */

float
aTan2(y,x) /* patched version of atan2 to error-check */
float y,x;
{
	if ((fabs(y)+fabs(x))<OKERR) return 0;
	return atan2(y,x);
} /* aTan2 */

complex
sph_tri_center(z1,z2,z3) /* given points on sphere, finds barycenter of
triangle they form; inside determined by orientation. */
complex z1,z2,z3;
{
	float X[3],Y[3],Z[3],M[3],C[3],D[3],vn;

	s_pt_to_vec(z1,X);
	s_pt_to_vec(z2,Y);
	s_pt_to_vec(z3,Z);
	M[0]=(X[0]+Y[0]+Z[0])/3.0;
	M[1]=(X[1]+Y[1]+Z[1])/3.0;
	M[2]=(X[2]+Y[2]+Z[2])/3.0;
	cross_prod(Y,X,C);
	vn=vec_norm(C);
	if (vn<okerr) /* almost parallel */
	 {
		cross_prod(Z,Y,D);
		vn=vec_norm(D);
		if (vn<okerr || dot_prod(D,X)>0) /* M should be good. */
		 {
			return (proj_vec_to_sph(M[0],M[1],M[2]));
		 }
		return (proj_vec_to_sph((-1.0)*M[0],(-1.0)*M[1],(-1.0)*M[2]));
	 }
	if (vec_norm(M)<okerr) /* almost coplanar */
		return (proj_vec_to_sph(C[0],C[1],C[2]));
	if (dot_prod(C,Z)>0) return (proj_vec_to_sph(M[0],M[1],M[2]));
	return (proj_vec_to_sph((-1.0)*M[0],(-1.0)*M[1],(-1.0)*M[2]));
} /* sph_tri_center */

int
pt_in_sph_tri(pt,z1,z2,z3) /* true if pt in given tri, assuming corners
in order for sph CONVEX tri; may fail for non-convex tri. */
complex pt,z1,z2,z3;
{
	float X[3],Y[3],Z[3],P[3],C[3];

	s_pt_to_vec(pt,P);
	s_pt_to_vec(z1,X);
	s_pt_to_vec(z2,Y);
	s_pt_to_vec(z3,Z);
	cross_prod(Y,X,C);
	if (dot_prod(P,C)<0) return 0;
	cross_prod(Z,Y,C);
	if (dot_prod(P,C)<0) return 0;
	cross_prod(X,Z,C);
	if (dot_prod(P,C)<0) return 0;
	return 1;
} /* pt_in_sph_tri */


/* ======================= matrix/vector operations =============== */

int 
sph_rotation(alpha, beta, gama, M, invM) /* Apply new rotations to 
matrix and its inverse; order of application = z, y, then x */ 
float alpha, beta, gama, M[3][3], invM[3][3]; 
{
	int i,j;
 	float X[3][3],Y[3][3],Z[3][3],S[3][3],nM[3][3],ninvM[3][3];

	for (i=0;i<3;i++) 
	 {
		for (j=0;j<3;j++)
			X[i][j]=Y[i][j]=Z[i][j]=nM[i][j]=ninvM[i][j]=0.0;
		X[i][i]=Y[i][i]=Z[i][i]=nM[i][i]=ninvM[i][i]=1.0;
	 }
		/*  individual rotation matrices */

	X[1][1]=X[2][2]=cos(alpha);
	X[2][1]=sin(alpha);
	X[1][2]=(-1.0)*X[2][1]; 

	Y[0][0]=Y[2][2]=cos(beta);
	Y[2][0]=sin(beta);
	Y[0][2]= (-1.0)*Y[2][0];
	
	Z[0][0]=Z[1][1]=cos(gama);
	Z[1][0]=sin(gama);
	Z[0][1]=(-1.0)*Z[1][0];

		/* multiply together */
	mat_mult(X,Y,S);
	mat_mult(S,Z,nM); /* nM = new rotation matrix */
		/* for inverse, only adjustments needed are these */
	X[2][1] *= (-1.0);
	X[1][2] *= (-1.0);
	Y[2][0] *= (-1.0);
	Y[0][2] *= (-1.0);
	Z[0][1] *= (-1.0);
	Z[1][0] *= (-1.0);
		/* multiply in correct order for 'left' inverse */
	mat_mult(Z, Y, S);
	mat_mult(S, X, ninvM);
		/* apply to given matrices */
	mat_mult(nM,M,S);
	for (i=0;i<3;i++) for (j=0;j<3;j++) M[i][j]=S[i][j];
	mat_mult(invM,ninvM,S);
	for (i=0;i<3;i++) for (j=0;j<3;j++) invM[i][j]=S[i][j];
	return 1;
} /*rot_mat */

int
mat_mult(A, B, C) /* C=AB */
float A[3][3], C[3][3], B[3][3];
{
	int i, j;
	for(i=0;i<3;i++) for(j=0;j<3;j++)
		C[i][j]=A[i][0]*B[0][j]+A[i][1]*B[1][j]+A[i][2]*B[2][j];
 } /* mat_mult */

float
dot_prod(V,W) /* (V,W) */
float V[3],W[3];
{
	int i, j;
	return (V[0]*W[0]+V[1]*W[1]+V[2]*W[2]);
} /* dot_prod */

int
cross_prod(X,Y,Z) /* Z=XxY */
float X[3],Y[3],Z[3];
{
	Z[0]=X[1]*Y[2]-X[2]*Y[1];
	Z[1]=X[2]*Y[0]-X[0]*Y[2];
	Z[2]=X[0]*Y[1]-X[1]*Y[0];
} /* cross_prod */

float
vec_norm(X) /* length of 3-vector */
float X[3];
{
	return (sqrt(X[0]*X[0]+X[1]*X[1]+X[2]*X[2]));
} /* vec_norm */

/* the following routines will each calculate a rotation matrix that fixes */
/* a particular axis (noted in routine's name)*/


/* ============= conversion routines ======================= */
/* NOTE: See book on Geometry of Complex Numbers, by Hans Swertfager.
Circle/line may be represented as 2x2 complex matrix. */

s_to_matrix_data(center,rad,C) /* given sph radius/center, find 2x2
matrix form of circle. */
complex center;
float rad;
Mobius *C;
{
	float m,x,y,R,D,z;
	complex ez;

	if (fabs(center.im-rad)<okerr)    /* if we project to a line */
	 {
		C->a.re=C->a.im=0;
		if (fabs(center.im+rad-M_PI)<okerr)   /* line thru origin */
		 {
			C->b.re=C->b.im=0;
			C->d.re=C->d.im=0;
			if (fabs(center.re)<okerr)   /* vertical line */
				C->b.re=0.5;
			else if (fabs(center.re-M_PI)<okerr) /* vertical,neg orient*/
				C->b.re=-0.5;
			else if (fabs(center.re-(M_PI/2))<okerr) /* horizontal line*/
				C->b.im=0.5;
			else if (fabs(center.re-(3*M_PI/2))<okerr) /* hor, neg ornt*/
				C->b.im=-0.5;
			else if (center.re<M_PI)                /* line */
			 {
				m=-1.0/tan(center.re);
				C->b.re=m/2.0;C->b.im=0.05;
			 }
			else	/* center.re>=M_PI */ /*line, neg orientation */
			 {
				m=-1.0/tan(center.re);				
				C->b.re=(-1.0)*M_PI/2.0; C->b.im=-0.5;
			 }
		 }
		else				/* line not thru origin */
		 {
/* CHECK */		R=sin(center.im+rad)/(1-cos(center.im+rad));
			if (fabs(center.re)<okerr)   /* vertical line */
			 {
				C->b.re=0.5; C->b.im=0;
				C->d.re=R; C->d.im=0;
			 }
			else if (fabs(center.re-M_PI)<okerr) /* vertical,neg orient*/
			 {
				C->b.re=-0.5; C->b.im=0;
				C->d.re=-R; C->d.im=0;
			 }
			else if (fabs(center.re-(M_PI/2))<okerr) /* horizontal line*/
			 {
				C->b.re=0; C->b.im=0.5;
				C->d.re=R; C->d.im=0;
			 }
			else if (fabs(center.re-(3*M_PI/2))<okerr) /* hor, neg ornt*/
			 {
				C->b.re=0; C->b.im=-0.5;
				C->d.re=-R; C->d.im=0;
			 }
			else if (rad<(M_PI/2.0))                /* line not special*/
			 {
				m=-1.0/tan(center.re);
				x=R*cos(center.re);
				y=R*sin(center.re);
				
				if ((D=y-m*x)>0)
				 {
					C->b.re=m/2.0;C->b.im=0.5;
					C->d.re=D; C->d.im=0;
				 }
				else	/* adjust so sgn C->d matches orient */
				 {
					C->b.re=-m/2.0; C->b.im=-0.5;
					C->d.re=-D; C->d.im=0;
				 }
			 }

			else if (rad>(M_PI/2.0))       /*line, neg orientation */
			 {
				m=-1.0/tan(center.re);
				x=R*cos(center.re);
				y=R*sin(center.re);

				if ((D=y-m*x)<0)
				 {
					C->b.re=m/2.0;  C->b.im=0.5;
					C->d.re=D;  C->d.im=0;
				 }
				else     /* adjust so sgn C->d matches orient */
			 	 {
					C->b.re=-m/2.0; C->b.im=-0.5;
					C->d.re=-D; C->d.im=0;
				 }
			 }
		 }
	 }
	else                                    /* project to circle */
	 {
		s_to_e_data(center,rad,&ez,&R);
		if (rad<center.im)
		 {
			C->a.re=1; C->a.im=0;
			C->b.re=-ez.re;  C->b.im=ez.im;   /* B= - conj(z) */
			C->d.re=cAbs(ez)*cAbs(ez) - R*R; C->d.im=0;
		 }
		else  /* negative orientation */
		 {
			C->a.re=-1; C->a.im=0;
			C->b=cconj(ez);

			C->d.re=-cAbs(ez)*cAbs(ez)+R*R; C->d.im=0;
		 }
	 }
	C->c=cconj(C->b);
	return;
} /* s_to_matrix_data */

matrix_to_s_data(C,center,rad) /* given 2x2 matrix form of circle,
find sph radius/center. */
Mobius C;
complex *center;
float *rad;
{
	float x,y,R,m;
	complex z;
	
	if (fabs(C.a.re)<okerr)		/* line */
	 {
		if (fabs(C.d.re)<okerr)		/* line thru origin */
		 {
			if (C.b.im=0)		/* vertical line */
			 {
				if (C.b.re>0)   /* pos orient */
				 {
					center->re=0;
					center->im=M_PI/2.0;
					*rad=M_PI/2.0;
					return;
				 }
				else		/* neg orient */
				 {
					center->re=M_PI;
					center->im=M_PI/2.0;
					*rad=M_PI/2.0;
					return;
				 }
			 }
			if (C.b.re=0)		/* horizonal line */
			 {
/* CHECK COMPATIBILITY WITH MOBIUS */
				if (C.b.im>0)	/* pos orient */
				 {
					center->re=M_PI/2.0;
					center->im=M_PI/2.0;
					*rad=M_PI/2.0;
					return;
				 }
				else		/* neg orient */
				 {
					center->re=3*M_PI/2.0;
					center->im=M_PI/2.0;
					*rad=M_PI/2.0;
					return;
				 }
			 }
			else		/* just a line thru origin */		
			 {
				if (C.b.im>0)	/* pos orient */
				 {
					m=C.b.re/C.b.im;
					center->re=aTan2(-1.0,m);
					center->im=M_PI/2.0;
					*rad=M_PI/2.0;
					return;
				 }
				else		/* neg orient */
				 {
					m=C.b.re/C.b.im;
					center->re=aTan2(-1.0,m) + M_PI;
					center->im=M_PI/2.0;
					*rad=M_PI/2.0;
					return;
				 }
			 }
		 }
		else	/* line not thru origin */
		 {
			/* Let R= d(origin,line) */
			if (fabs(C.b.im)>okerr)	/* not vertical */
				R=(C.b.im<0)? 0.5*(-C.d.re)/C.b.im :
				  0.5*C.d.re/C.b.im;
			else 	/* vertical */
				R=(C.b.re<0) ? 0.5*(-C.d.re)/C.b.re :
				  0.5*C.d.re/C.b.re;

			if (fabs(C.b.im)<okerr)		/* vertical */
			 {
				z=proj_pt_to_sph(R,0);
	
				if (C.b.re>0)		/* pos orient */
				 {
					center->re=z.re;
					center->im=z.im/2.0;
					*rad=center->im;
					return;
				 }
				else			/* neg orient */
				 {	
					center->re=(z.re<M_PI)? z.re+M_PI :
						   z.re-M_PI;
					center->im=M_PI-z.im/2.0;
					*rad=center->im;
					return;
				 }
			 }
			if (fabs(C.b.re)<okerr)		/* horizontal */
			 {
				z=proj_pt_to_sph(0,R);
				
				if (C.b.im>0)		/* pos orient */
				 {
					center->re=z.re;
					center->im=z.im/2.0;
					*rad=center->im;
					return;
				 }
				else			/* neg orient */
				 {
					center->re=(z.re<M_PI)? z.re+M_PI :
						   z.re-M_PI;
					center->im=M_PI-z.im/2.0;
					*rad=center->im;
					return;
				 }
			 }
			else				/* just a line */
			 {
				m=C.b.re/C.b.im;
				x=(-R*m)/(m*m + 1); /*x,y coord nearest origin*/
				y=R/(m*m+1);
				z=proj_pt_to_sph(x,y); /* corresp pt on sph */

				if (C.b.im>0)		/* pos orient */
				 {
					center->re=z.re;
					center->im=z.im/2;
					*rad=center->im;
					return;
				 }
				else			/* neg orient */
				 {
					center->re=(z.re<M_PI)? z.re+M_PI :
						  z.re-M_PI;
					center->im=M_PI-z.im/2.0;
					*rad=center->im;
					return;
				 }
			 }				
		 }
	 }
	else	/* circle */
	 {
		z.re=-C.c.re/C.a.re;
		z.im=-C.c.im/C.a.re;		
			/* z=eucl. center of circle */
		R=sqrt(cAbs(z)*cAbs(z)-(C.d.re/C.a.re)); 
			/* R= eucl. rad of circle */

		e_to_s_data(z,R,center,rad);
		
		if (C.a.re<0)	/* neg orientation */
		 {
			center->re=(center->re<M_PI) ? 
				center->re+M_PI : center->re-M_PI;
			center->im=fabs(M_PI-center->im);
			*rad = M_PI-*rad;
			return;
		 }
	 }
} /* matrix_to_s_data */

complex
proj_pt_to_sph(x,y)
float x,y;
{
	float x1,x2,x3,denom;

	denom=x*x+y*y+1.0;
	x1=(2.0*x)/denom;
	x2=(2.0*y)/denom;
	x3=(denom-2.0)/denom;
	return (proj_vec_to_sph(x1,x2,x3));
} /* proj_pt_to_sph */

		 
       
 

  
 




