/*
 * Copyright (c) 1999,2001 Robert Woodcock <rcw@debian.org>
 * This code is hereby licensed for public consumption under either the
 * GNU GPL v2 or greater, or Larry Wall's Artistic license - your choice.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/ioctl.h>

/* Porting credits:
 * Solaris: David Champion <dgc@uchicago.edu>
 * FreeBSD: Niels Bakker <niels@bakker.net>
 * OpenBSD: Marcus Daniel <danielm@uni-muenster.de>
 * NetBSD: Chris Gilbert <chris@NetBSD.org>
 */

#if defined(linux)
# include <linux/cdrom.h>

#elif defined(sun) && defined(unix) && defined(__SVR4)
# include <sys/cdio.h>
# define CD_MSF_OFFSET	150
# define CD_FRAMES	75

#elif defined(__FreeBSD__)
#include <sys/cdio.h>
#define        CDROM_LBA       CD_LBA_FORMAT   /* first frame is 0 */
#define        CD_MSF_OFFSET   150     /* MSF offset of first frame */
#define        CD_FRAMES       75      /* per second */
#define        CDROM_LEADOUT   0xAA    /* leadout track */
#define        CDROMREADTOCHDR         CDIOREADTOCHEADER
#define        CDROMREADTOCENTRY       CDIOREADTOCENTRY
#define        cdrom_tochdr    ioc_toc_header
#define        cdth_trk0       starting_track
#define        cdth_trk1       ending_track
#define        cdrom_tocentry  ioc_read_toc_single_entry
#define        cdte_track      track
#define        cdte_format     address_format
#define        cdte_addr       entry.addr

#elif defined(__OpenBSD__) || defined(__NetBSD__)
#include <sys/cdio.h>
#define        CDROM_LBA       CD_LBA_FORMAT   /* first frame is 0 */
#define        CD_MSF_OFFSET   150     /* MSF offset of first frame */
#define        CD_FRAMES       75      /* per second */
#define        CDROM_LEADOUT   0xAA    /* leadout track */
#define        CDROMREADTOCHDR         CDIOREADTOCHEADER
#define        cdrom_tochdr    ioc_toc_header
#define        cdth_trk0       starting_track
#define        cdth_trk1       ending_track
#define        cdrom_tocentry  cd_toc_entry
#define        cdte_track      track
#define        cdte_addr       addr


#else
# error "Your OS isn't supported yet."
#endif	/* os selection */

struct cdrom_tocentry *TocEntry;

int cddb_sum (int n)
{
	/* a number like 2344 becomes 2+3+4+4 (13) */
	int ret=0;

	while (n > 0) {
		ret = ret + (n % 10);
		n = n / 10;
	}

	return ret;
}

int main(int argc, char *argv[])
{
#if defined(__OpenBSD__) || defined(__NetBSD__)
	struct ioc_read_toc_entry t;
#endif
	int len;
	int drive, i, totaltime;
	long int cksum=0;
	unsigned char first=1, last=1;
	struct cdrom_tochdr hdr;
	
	if (argc < 2) {
		fprintf(stderr, "Usage: %s <devicename>\n", argv[0]);
		exit(1);
	}

	drive = open(argv[1], O_RDONLY | O_NONBLOCK);
	if (drive < 0) {
		fprintf(stderr, "cd-discid: %s: ", argv[1]);
		perror("open");
		exit(1);
	}

	if (ioctl(drive,CDROMREADTOCHDR,&hdr) < 0) {
		fprintf(stderr, "cd-discid: %s: ", argv[1]);
		perror("CDROMREADTOCHDR");
		exit(1);
	}

	first=hdr.cdth_trk0;
	last=hdr.cdth_trk1;

	len = (last + 1) * sizeof (struct cdrom_tocentry);
	if (TocEntry) {
		free(TocEntry);
		TocEntry = 0;
	}
	TocEntry = malloc(len);
	if (!TocEntry) {
		fprintf(stderr, "cd-discid: %s: Can't allocate memory for TOC entries\n", argv[1]);
		exit(1);
	}

#if defined(__OpenBSD__) 
	t.address_format = CDROM_LBA;
	t.starting_track = 0;
	t.data_len = len;
	t.data = TocEntry;

	if (ioctl(drive, CDIOREADTOCENTRYS, (char *) &t) < 0) {
		fprintf(stderr, "cd-discid: %s: ", argv[1]);
		perror("CDIOREADTOCENTRYS");
	}	
#elif defined(__NetBSD__)
	t.address_format = CDROM_LBA;
	t.starting_track = 1;
	t.data_len = len;
	t.data = TocEntry;
	memset(TocEntry, 0, len);
	
	if(ioctl(drive, CDIOREADTOCENTRYS, (char *) &t) < 0) {
	    fprintf(stderr, "cd-discid: %s: ", argv[1]);
	    perror("CDIOREADTOCENTRYS");
	}
	
#else

	for (i=0; i < last; i++) {
		TocEntry[i].cdte_track = i + 1; /* tracks start with 1, but i must start with 0 on OpenBSD */
		TocEntry[i].cdte_format = CDROM_LBA;
		if (ioctl(drive, CDROMREADTOCENTRY, &TocEntry[i]) < 0) {
			fprintf(stderr, "cd-discid: %s: ", argv[1]);
			perror("CDROMREADTOCENTRY");
		}
	}

	TocEntry[last].cdte_track = CDROM_LEADOUT;
	TocEntry[last].cdte_format = CDROM_LBA;
	if (ioctl(drive, CDROMREADTOCENTRY, &TocEntry[i]) < 0) {
		fprintf(stderr, "cd-discid: %s: ", argv[1]);
		perror("CDROMREADTOCENTRY");
	}
#endif

#if defined(__FreeBSD__)
	TocEntry[i].cdte_addr.lba = ntohl(TocEntry[i].cdte_addr.lba);
#endif

	for (i=0; i < last; i++) {
#if defined(__FreeBSD__)
		TocEntry[i].cdte_addr.lba = ntohl(TocEntry[i].cdte_addr.lba);
#endif
		cksum += cddb_sum((TocEntry[i].cdte_addr.lba + CD_MSF_OFFSET) / CD_FRAMES);
	}

	totaltime = ((TocEntry[last].cdte_addr.lba + CD_MSF_OFFSET) / CD_FRAMES) -
		    ((TocEntry[0].cdte_addr.lba + CD_MSF_OFFSET) / CD_FRAMES);

	/* print discid */
	printf("%08lx", (cksum % 0xff) << 24 | totaltime << 8 | last);

	/* print number of tracks */
	printf(" %d", last);

	/* print frame offsets of all tracks */
	for (i = 0; i < last; i++) {
		printf(" %d", TocEntry[i].cdte_addr.lba + CD_MSF_OFFSET);
	}
	
	/* print length of disc in seconds */
	printf(" %d\n", (TocEntry[last].cdte_addr.lba + CD_MSF_OFFSET) / CD_FRAMES);

        return 0;
}
