/*
 * KON2 - Kanji ON Console -
 * Copyright (C) 1992-1996 Takashi MANABE (manabe@papilio.tutics.tut.ac.jp)
 *
 * CCE - Console Chinese Environment -
 * Copyright (C) 1998-1999 Rui He (herui@cs.duke.edu)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY TAKASHI MANABE ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * 
 */

/*
  This code is based on vgalib.
  
  Thanks to frandsen@diku.dk (Tommy Frandsen).
  */

#include	<stdio.h>
#include	<fcntl.h>
#include	<termios.h>
#include	<string.h>
#include	<unistd.h>
#include	<sys/mman.h>
#if defined(linux)
/* #include	<linux/mm.h> */
#include	<sys/kd.h>
#elif defined(__FreeBSD__)
#include      <vm/vm_param.h>
#include      <sys/ioctl.h>
#include      <machine/console.h>
vm_size_t page_size;
#endif
#undef free
#include	<stdlib.h>

#include	<getcap.h>
#include	<defs.h>
#include	<errors.h>
#include	<vc.h>
#include	<vt.h>
#include	<vga.h>
#include	<font.h>

#ifdef __Use_lrmi__
#include        <lrmi.h>
#endif

static struct pelRegs grapPels, textPels;

struct vgaRegs
    regText,
    regGraph = {
	{	/* CRT */  /* CRT 25, ATT 21, GRA 9, SEQ 5, MISC 1, Total 61 */
	    0x5F,0x4F,0x50,0x82,0x54,0x80,0x0B,0x3E,
	    0x00,0x40,0x00,0x00,0x00,0x00,0x00,0x00,
	    0xEA,0x2C,0xDF,0x28,0x00,0xE7,0x04,0xE3,
	    0xFF 
	}, {	/* ATT OK, reg 0x11 is the overscan register */
	    0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,
	    0x08,0x09,0x0A,0x0B,0x0C,0x0D,0x0E,0x0F,
	    0x01,0x00,0x0F,0x00,0x00
	}, {	/* GRA */
	    0x00,0x0F,0x00,0x20,0x00,0x00,0x05,0x0F, 
	    0xFF
	}, {	/*SEQ svgalib's value is 02, 02/03 all OK */
	    0x03,0x01,0x0F,0x00,0x06
	}, /* MIS */
	    0xE3
    };


static struct videoTimings
{
        int hDot, hStart, hEnd, hTotal;
        int vLine, vStart, vEnd, vTotal;
        int clock,txmax, tymax;
}
video = 
{
        640, 680, 768, 800,
        480, 491, 493, 525,
        1,   80,  25
};


int LineComp9, LineComp8; //, gramHead;
 /* gramHead is the relative current starting address, gramMem is the absolute
    display address */

static u_int vgaCrtAddr = 0x3D4;
static u_int vgaCrtData = 0x3D5;
static u_int vgaSt1Addr = 0x3DA;   /* State Register */

char	*gramMem;		/* dummy buffer for mmapping grahics memory */

u_int   writeAddr;               /* address to write next character, (x,y)in svgalib mode */
bool    kanjiCursor;
u_char  cursorTop, cursorBtm;

static	char	*fontBuff1;		/* saved font data - plane 2 */
static	char	*fontBuff2;		/* saved font data - plane 3 */
static	bool	savePlane3;

//#define GRAPH_MEM  (80*(25*18+30))
//#define TOTAL_TEXT_MEM  (80*30*18)
//#define INIT_START_MEM   0x960
//#define INIT_START_MEM   0x0
   /* 0x960 = 2400 = 80 * 30, input area */

/*  /usr/src/linux/drivers/video/fbcmap.c  */

u_short red16[] = {
    0x0000, 0x0000, 0x0000, 0x0000, 0xaaaa, 0xaaaa, 0xaaaa, 0xaaaa,
    0x5555, 0x5555, 0x5555, 0x5555, 0xffff, 0xffff, 0xffff, 0xffff
};

u_short green16[] = {
    0x0000, 0x0000, 0xaaaa, 0xaaaa, 0x0000, 0x0000, 0x5555, 0xaaaa,
    0x5555, 0x5555, 0xffff, 0xffff, 0x5555, 0x5555, 0xffff, 0xffff
};

u_short blue16[] = {
    0x0000, 0xaaaa, 0x0000, 0xaaaa, 0x0000, 0xaaaa, 0x0000, 0xaaaa,
    0x5555, 0xffff, 0x5555, 0xffff, 0x5555, 0xffff, 0x5555, 0xffff
};

/*
u_short red16[] = {
    0x0000, 0x0000, 0x0000, 0x0000, 0x7f7f, 0x7f7f, 0x7f7f, 0x7f7f,
    0xc0c0, 0xc0c0, 0xc0c0, 0xc0c0, 0xffff, 0xffff, 0xffff, 0xffff
};

u_short green16[] = {
    0x0000, 0x0000, 0x7f7f, 0x7f7f, 0x0000, 0x0000, 0xc0c0, 0x7f7f,
    0xc0c0, 0xc0c0, 0xffff, 0xffff, 0xc0c0, 0xc0c0, 0xffff, 0xffff
};

u_short blue16[] = {
    0x0000, 0x7f7f, 0x0000, 0x7f7f, 0x0000, 0x7f7f, 0x0000, 0x7f7f,
    0xc0c0, 0xffff, 0xc0c0, 0xffff, 0xc0c0, 0xffff, 0xc0c0, 0xffff
};
*/
/***************************************************************************
 *                                Vga routines                             *
 ***************************************************************************/

void VgaSetRegisters(struct vgaRegs *regs)
{
    int	i;
    
    /* disable video */
    PortInb(vgaSt1Addr);	
    PortOutb(0x00, VGAATTR_A_O);
    
    /* update misc output register */
    PortOutb((regs->mis&~0x1)|(PortInb(VGAMISC_IN)&0x01), VGAMISC_OUT);
    
    /* synchronous reset on */
    PortOutb(0x00,VGASEQ_ADDR);
    PortOutb(0x01,VGASEQ_DATA);	
    
    /* write sequencer registers */
    for (i = 1; i < VGASEQ_CNT; i++) 
    {
	PortOutb(i, VGASEQ_ADDR);
	PortOutb(regs->seq[i], VGASEQ_DATA);
    }
    
    /* synchronous reset off */
    PortOutb(0x00, VGASEQ_ADDR);
    PortOutb(0x03, VGASEQ_DATA);
    
    /* deprotect CRT registers 0-7 */
    PortOutb(0x11, vgaCrtAddr);
    PortOutb(PortInb(vgaCrtData)&0x7F, vgaCrtData);
    
    /* write CRT registers */
    for (i = 0; i < VGACRT_CNT; i++) 
    {
	PortOutb(i, vgaCrtAddr);
	PortOutb(regs->crt[i], vgaCrtData);
    }
    
    /* write graphics controller registers */
    for (i = 0; i < VGAGRP_CNT; i++) 
    {
	PortOutb(i, VGAGRP_ADDR);
	PortOutb(regs->gra[i], VGAGRP_DATA);
    }
    
    /* write attribute controller registers */
    for (i = 0; i < VGAATTR_CNT; i++) 
    {
	/* reset flip-flop */
	PortInb(vgaSt1Addr);
	PortOutb(i, VGAATTR_A_O);
	PortOutb(regs->att[i],VGAATTR_A_O);
    }
}

/***********************************************************************
 *                        Palette Operations                           *
 ***********************************************************************/

/* DAC Palette
#define VGAPAL_OADR     0x3C8
#define VGAPAL_IADR     0x3C7
#define VGAPAL_DATA     0x3C9
#define MAX_PELS        16
*/

static void VgaSetPels(struct pelRegs *pels)
{
    int i;
    
    PortOutb(0, VGAPAL_OADR);

    for (i = 0; i < MAX_PELS; i++) {
	PortOutb(pels->red[i], VGAPAL_DATA);
	PortOutb(pels->grn[i], VGAPAL_DATA);
	PortOutb(pels->blu[i], VGAPAL_DATA);
    }
}

static void VgaGetPels(struct pelRegs *pels)
{
    int	i;

    PortOutb(0, VGAPAL_IADR);

    for (i = 0; i < MAX_PELS; i++)
    {
	pels->red[i] = PortInb(VGAPAL_DATA);
	pels->grn[i] = PortInb(VGAPAL_DATA);
	pels->blu[i] = PortInb(VGAPAL_DATA);
    }
}

/*
int VgaReadPels(const char *str)
{
    int i, red, grn, blu;

    for (i = 0; i < MAX_PELS; i ++) 
    {
        sscanf(str, "%d %d %d", &red, &grn, &blu);
        if ((str = strchr(str, '\n')) == NULL) {
            error("PELS entry too short\r\n");
            return FAILURE;
        }
        str++;                  // skip '\n' 
        grapPels.red[i] = red;
        grapPels.grn[i] = grn;
        grapPels.blu[i] = blu;
    }
    return SUCCESS;
}
*/

static inline void VgaSetColor(u_char col)
{
    static int old;
    
    if (old == col) return;
    PortOutw(col << 8, VGAGRP_ADDR);
    old = col;
}

// -------------------------------------------------------------
// VgaStart
//    1. Save palettes into textPels
//    2. Allocate Font buffer
//    3. Save Textmode VGA controller Registers
//    4. Set Registers to Graphics mode
//    5. Save FONT into plane 2 or 3
// Use when start graphics mode.
// ------------------------------------------------------------------
void VgaStart(void)
{
#ifdef __Use_lrmi__
    struct LRMI_regs r;
    memset(&r, 0, sizeof(struct LRMI_regs));

    r.eax = 0x12;
    LRMI_int(0x10, &r);

    // Set Set/Reset Register and Data Recycle Register
    PortOutw(0x0F01, VGAGRP_ADDR);
    PortOutw(0x2003, VGAGRP_ADDR);
#elif defined(__FreeBSD__)
    ioctl(0, SW_CG640x480, 0);

    // Set Set/Reset Register and Data Recycle Register
    PortOutw(0x0F01, VGAGRP_ADDR);
    PortOutw(0x2003, VGAGRP_ADDR);
#else
    int	i;

    VgaGetPels(&textPels);

    if ((fontBuff1 = malloc(FONT_SIZE)) == NULL ||
       (savePlane3 && (fontBuff2 = malloc(FONT_SIZE)) == NULL))
    {
        fatal("malloc fontBuff failed!");
    }
    
    /* disable video */
    PortInb(vgaSt1Addr);   /* State Register 0x3DA */	
    PortOutb(0x00, VGAATTR_A_O);  /* Attribute Register 0x3C0 */

    /* save text mode VGA registers */
    for (i = 0; i < VGACRT_CNT; i++) 
    {
	PortOutb(i, vgaCrtAddr);
	regText.crt[i] = PortInb(vgaCrtData);
    }
    for (i = 0; i < VGAATTR_CNT; i++) {
	PortInb(vgaSt1Addr);
	PortOutb(i, VGAATTR_A_O);
	regText.att[i] = PortInb(VGAATTR_DATA);
    }
    for (i = 0; i < VGAGRP_CNT; i++) {
	PortOutb(i, VGAGRP_ADDR);
	regText.gra[i] = PortInb(VGAGRP_DATA);
    }
    for (i = 0; i < VGASEQ_CNT; i++) {
	PortOutb(i, VGASEQ_ADDR);
	regText.seq[i] = PortInb(VGASEQ_DATA);
    }
    regText.mis = PortInb(VGAMISC_IN);
    
    PortOutb(PortInb(VGAMISC_IN)|0x01, VGAMISC_OUT);
    VgaSetRegisters(&regGraph);
    
    /* save font data in plane 2 */
    PortOutw(0x0204, VGAGRP_ADDR);
    memcpy(fontBuff1, gramMem, FONT_SIZE);  /* FONT_SIZE=0x2000, 8K */

    if (savePlane3) 
    {
	/* save font data in plane 3 */
	PortOutw(0x0304, VGAGRP_ADDR);
	memcpy(fontBuff2, gramMem, FONT_SIZE);
    }
#endif

    VgaClearInput(1);
}

// VgaTextMode is called when VC switches to text mode
void VgaTextMode(void)
{
#ifdef __Use_lrmi__
    struct LRMI_regs r;
    memset(&r, 0, sizeof(struct LRMI_regs));

    ioperm(0, 0x400, 1);
    iopl(3);
    
    r.eax = 0x3;
    LRMI_int(0x10, &r);
#elif defined(__FreeBSD__)
    ioctl(0, SW_TEXT_80x25, 0);
#else
    /* disable video */
    PortInb(vgaSt1Addr);
    PortOutb(0x00, VGAATTR_A_O);

    /* restore font data - first select a 16 color graphics mode */
    VgaSetRegisters(&regGraph);

    /* disable Set/Reset Register */
    PortOutb(0x01, VGAGRP_ADDR );
    PortOutb(0x00, VGAGRP_DATA );
    
    /* restore font data in plane 2 - necessary for all VGA's */
    PortOutb(0x02, VGASEQ_ADDR );
    PortOutb(0x04, VGASEQ_DATA );
    memcpy(gramMem, fontBuff1, FONT_SIZE);
    
    if (savePlane3) 
    {
	/* restore font data in plane 3 - necessary for Trident VGA's */
	PortOutb(0x02, VGASEQ_ADDR );
	PortOutb(0x08, VGASEQ_DATA );
	memcpy(gramMem, fontBuff2, FONT_SIZE);
    }
    
    /* restore text mode VGA registers */
    VgaSetRegisters(&regText);
    
    /* set text palette */
    
    VgaSetPels(&textPels);
    
    /* enable video */
    PortInb(vgaSt1Addr);
    PortOutb(0x20, VGAATTR_A_O);
#endif
}

// Enter Graphics mode when VC switches back - as VgaTextMode
void VgaGraphMode(void)
{
#ifdef __Use_lrmi__
    struct LRMI_regs r;
    memset(&r, 0, sizeof(struct LRMI_regs));

    r.eax = 0x12;
    LRMI_int(0x10, &r);
    
    // Set Set/Reset Register and Data Recycle Register
    PortOutw(0x0F01, VGAGRP_ADDR);
    PortOutw(0x2003, VGAGRP_ADDR);
#elif defined(__FreeBSD__)
    ioctl(0, SW_CG640x480, 0);

    // Set Set/Reset Register and Data Recycle Register
    PortOutw(0x0F01, VGAGRP_ADDR);
    PortOutw(0x2003, VGAGRP_ADDR);
#else
    /* disable video */
    PortInb(vgaSt1Addr); 		
    PortOutb(0x00, VGAATTR_A_O);	
    
    VgaSetRegisters(&regGraph);

    /* set default palette */
    
    VgaSetPels(&grapPels);
    
    /* enable video */
    PortInb(vgaSt1Addr);
    PortOutb(0x20, VGAATTR_A_O);
    VgaSetStartAddress();
#endif
}

/***********************************************************************
 *                Single/Double Character Output                       *
 ***********************************************************************/

/* After set the writing address(writeAddr), call this to output the 
  value gramMem is the absolute starting address */

void VgaWput(u_char *code, u_char fc, u_char bc)
{
// #ifndef __Use_lrmi__
    volatile char	*gram, *vcls;
    u_char *til;
    u_char	x;

    VgaSetColor( bc&7);
    vcls = gram = gramMem + writeAddr;
    for (x = 0;x < dispInfo.glineChar;x ++, vcls += dispInfo.glineByte)
	*vcls = *(vcls + 1) = 0;
    VgaSetColor(fc);
    if (bc & 0x8) 
    {
	vcls -= dispInfo.glineByte;
	*vcls = *(vcls + 1) = 0;
    }
    til = code + (dbFReg->high << 1);
    for (;code < til; code ++, gram += dispInfo.glineByte) 
    {
	if (*code) 
        {
	    VgaOutByte(*code);
	    *gram = *gram;
//	    *gram = 0xFF;
	}
	code ++;
	if (*code) 
        {
	    VgaOutByte(*code);
	    *(gram + 1) = *(gram + 1);
//	    *(gram + 1) = 0xFF;
	}
    }
    VgaOutByte(0xFF);
// #endif
}

void VgaSput(u_char *code, u_char fc, u_char bc)
{
    volatile char *gram, *vcls;
    u_char *til;
    u_char	x;

    vcls = gram = gramMem + writeAddr;
    VgaSetColor(bc&7 );
    for (x = 0;x < dispInfo.glineChar;x ++, vcls += dispInfo.glineByte)
	*vcls = 0;
    if (!code) return;
    VgaSetColor(fc);
    if (bc & 0x8) *(vcls - dispInfo.glineByte) = 0;
    til = code + sbFReg->high;
    for (;code < til;code ++, gram += dispInfo.glineByte) {
	if (*code) {
	    VgaOutByte(*code);
	    *gram = *gram;
	}
    }
    VgaOutByte(0xFF);
}

// only one line, input area
void VgaInputSput(int x,u_char ch,int fg,int bg)
{
   sbFReg = &fSRegs[0];
   VgaSetInputAddress(x);
   VgaSput(sbFReg->bitmap + (ch << 4), fg, bg);
}

void VgaInputWput(int x,u_char ch1,u_char ch2,int fg,int bg)
{
    u_int fnt;

    dbFReg = &fDRegs[0];  // hard-coded here, GB here
    VgaSetInputAddress(x);
    fnt = dbFReg->addr(ch2, ch1);
    if (fnt < dbFReg->size)
       VgaWput(dbFReg->bitmap + fnt, fg, bg);
}

/**************************************************************************
 *              Hardware(display start address) scroll up/down            * 
 **************************************************************************/

/* no hardware scroll support */

#if 0
void VgaResetHardScroll(void)
{
    gramHead =  INIT_START_MEM;
    VgaSetStartAddress(); 
    // reset the video buffer start address
}

void VgaHardScrollUp(int line, int color)
{
    int	oldhead;
    
    VgaSetColor(color);
    
    if (line > dispInfo.tymax) 
    {
	line %= dispInfo.tymax + 1;
	lzero(gramMem+INIT_START_MEM, dispInfo.tsize); //TOTAL_TEXT_MEM);
    }
    
    oldhead = gramHead;
    gramHead += line * dispInfo.tlineByte;
    lzero(gramMem + oldhead + dispInfo.gsize, gramHead - oldhead);
    VgaSetStartAddress();
}

void VgaHardScrollDown(int line, int color)
{
    int	oldhead;
    
    VgaSetColor(color);
   
    if (line > dispInfo.tymax) 
    {
	line %= dispInfo.tymax + 1;
	lzero(gramMem+INIT_START_MEM, dispInfo.tsize); //TOTAL_TEXT_MEM);
    }
    
    oldhead = gramHead;
    gramHead -= line * dispInfo.tlineByte;
    lzero(gramMem + gramHead, oldhead-gramHead);
    VgaSetStartAddress();
}

#endif

void VgaSetCursorAddress(CursorInfo *ci, u_int x, u_int y)
{
    ci->addr = /* gramHead +*/ y * dispInfo.tlineByte + cursorTop
		* dispInfo.glineByte + x;
}

/* p is index, 0-4095  80x25 = 2000 -  p == the Character Index in screen - 0 - 1999 */
void VgaSetAddress(u_int p)
{
    writeAddr = /* gramHead + */ (p%dispInfo.glineByte) + (p/dispInfo.glineByte) * dispInfo.tlineByte;
}

void VgaSetInputAddress(u_int p)
{
  writeAddr = (dispInfo.gydim - dispInfo.input) * dispInfo.glineByte +
       (p % dispInfo.tx_avail) + 6 * dispInfo.glineByte ;
}

/* draw the cursor using XOR operation */

void VgaCursor(CursorInfo *ci)
{
//#ifndef __Use_lrmi__
    volatile char	*gram;
    u_char	x;
    int	bottom = cursorBtm + 1 <= dispInfo.glineChar ?
	cursorBtm + 1 : dispInfo.glineChar;
    
    VgaSetColor(15);
    gram = gramMem + ci->addr;
    
    PortOutw(0x0F00, VGAGRP_ADDR);	/* color white */
    PortOutw(0x1803, VGAGRP_ADDR);	/* XOR mode */
    x = cursorTop;
    if (kanjiCursor && ci->kanji) 
    {
	for (;x < bottom;x ++, gram += dispInfo.glineByte) 
        {
	    *gram = *gram;
	    *(gram + 1)= *(gram + 1);
	}
    } 
    else
	for (;x < bottom;x ++, gram += dispInfo.glineByte)
	    *gram = *gram;
    
    PortOutw(0x0003, VGAGRP_ADDR);	/* unmodify mode */
// #endif
}

/* clear the display area */
void VgaClearAll(int color)
{
#ifndef __Use_lrmi__
    VgaSetColor(color);
    lzero(gramMem /*+INIT_START_MEM */, dispInfo.tsize); //TOTAL_TEXT_MEM);
#endif
}

/* clear the input area */
void VgaClearInput(int color)
{
   VgaSetColor(color);  /* color = 1 */
   lzero(gramMem + dispInfo.tsize, dispInfo.input * dispInfo.glineByte);
   /* such operation is very slow, if you put the VgaClearInput in
   VgaSput, then it will be dramatically slower */
}

void VgaScreenSaver(bool blank)
{
#ifndef __Use_lrmi__
    if (blank) {
	PortOutb(0x01, VGASEQ_ADDR);
	PortOutb(PortInb(VGASEQ_DATA) | 0x20, VGASEQ_DATA);
    } else {
	PortOutb(0x01, VGASEQ_ADDR);
	PortOutb(PortInb(VGASEQ_DATA) & 0xDF, VGASEQ_DATA);
    }
#endif
}

/*************************************************************************
 *              Read the Palette/Register Values from cce.cfg            *
 *************************************************************************/

/* Standard VGA mode, 640 x 480 x 16 color */

static void VgaInitRegs(void)
{
    int i;
    int fontwidth = 8, fontheight = 16;

    dispInfo.gxdim = video.hDot;
    dispInfo.gydim = video.vLine;
    dispInfo.input = 30;

    dispInfo.tx_avail = dispInfo.gxdim / fontwidth;
    if (dispInfo.linegap < 0 || dispInfo.linegap > 10)
         dispInfo.linegap = 2;
    dispInfo.glineChar = fontheight + dispInfo.linegap;
    dispInfo.ty_avail = (dispInfo.gydim - dispInfo.input) / dispInfo.glineChar;

    if (dispInfo.txmax <= 0 || dispInfo.txmax > dispInfo.tx_avail)
           dispInfo.txmax = dispInfo.tx_avail;
    if (dispInfo.tymax <= 0 || dispInfo.tymax > dispInfo.ty_avail)
           dispInfo.tymax = dispInfo.ty_avail;

    dispInfo.glineByte = dispInfo.gxdim >> 3;
    dispInfo.gsize = dispInfo.glineByte * dispInfo.gydim;
    dispInfo.tlineByte = dispInfo.glineChar * dispInfo.glineByte;
    dispInfo.tsize = dispInfo.tlineByte * dispInfo.tymax;

    dispInfo.bpp = 1;

    for(i = 0; i < MAX_PELS; i++)
    {
       grapPels.red[i] = red16[i] >> 8;
       grapPels.grn[i] = green16[i] >> 8;
       grapPels.blu[i] = blue16[i] >> 8;
    };

/* What? We don't need to set these registers? */
    if (video.vLine < 480) 
    {
	regGraph.crt[23] = 0xE3;   //23=0x17
	regGraph.mis = 0xE3;
    } 
    else 
    {
	if (video.vLine < 768) regGraph.mis = 0xE3;
	else regGraph.mis = 0x23;
	regGraph.crt[23] = 0xC3;
    }
    regGraph.mis |= (video.clock & 3) << 2;
    regGraph.crt[0] = (video.hTotal>>3) - 5;
    regGraph.crt[1] = (video.hDot>>3) - 1;
    regGraph.crt[2] = (video.hStart>>3) - 1;
    regGraph.crt[3] = ((video.hEnd>>3) & 0x1F) | 0x80;
    regGraph.crt[4] = video.hStart>>3;
    regGraph.crt[5] = (((video.hEnd>>3) & 0x20) << 2)
	| ((video.hEnd>>3) & 0x1F);
    regGraph.crt[6] = (video.vTotal - 2) & 0xFF;

   /* crt[7]   ABCDEFGH
      A:  vStart bit 9 
      B:  vLine bit 9
      C:  vTotal bit 9   crt[6] = vTotal
      D:  1
      E:  vStart bit 8
      F:  ?
      G:  vLine bit 8
      H:  vTotal bit 8
 */
    regGraph.crt[7] = 0x10;

    regGraph.crt[7] |= (((video.vLine - 1) & 0x100) >> 7)
	| (((video.vLine - 1) & 0x200) >> 3);
        /* 0Y0000X0 X bit 8, Y bit 9 */
    regGraph.crt[7] |= ((video.vStart & 0x100) >> 6)
	| ((video.vStart & 0x100) >> 5);
         /* 000XX000 */
    regGraph.crt[7] |= (((video.vTotal - 2) & 0x100) >> 8)
	| (((video.vTotal - 2) & 0x200) >> 4);
         /* 00Y0000X, X bit 8, Y bit 9 */
    regGraph.crt[7] |= ((video.vStart & 0x200) >> 2);
         /* X0000000 */

    regGraph.crt[9] = ((video.vStart & 0x200) >>4) | 0x40;
             /* vStart bit 9, 00X00000  | 01000000 = 01X00000 */

    regGraph.crt[16] = video.vStart & 0xFF;
    regGraph.crt[17] = (video.vEnd & 0x0F) | 0x20;
    regGraph.crt[18] = (video.vLine - 1) & 0xFF;
    regGraph.crt[19] = video.hDot >> 4;
    regGraph.crt[21] = video.vStart & 0xFF;
    regGraph.crt[22] = (video.vStart + 1) & 0xFF;

    LineComp8 = ((regGraph.crt[7] & 0xEF) << 8) + 0x07;
            /* 11101111    Port 07 */
    LineComp9 = ((regGraph.crt[9] & 0xBF) << 8) + 0x09;
            /* 10111111    Port 09 */
}

/* VGA initialize & uninitialize */

int VgaAttach(void)
{
    int	devMem;
    

    if (ioctl(0, KDSETMODE,KD_GRAPHICS) < 0) 
    {
	Perror("ioctl CONSOLE_IO_ENABLE");
	return FAILURE;
    }
#if defined(linux)
    ioperm(VGAMISC_IN, 1, 1);
    if (!(PortInb(VGAMISC_IN)&0x01)) 
    { /* monochrome VGA */
	vgaCrtAddr = 0x3B4;
	vgaCrtData = 0x3B5;
	vgaSt1Addr = 0x3BA;
    }

    /* get I/O permissions for VGA registers */
    ioperm(vgaCrtAddr, 1, 1);
    ioperm(VGAATTR_A_O, 1, 1);
    ioperm(VGAGRP_ADDR, 1, 1);
    ioperm(VGASEQ_ADDR, 1, 1);
    ioperm(VGAPAL_OADR, 1, 1);
    ioperm(VGAPAL_IADR, 1, 1);
    ioperm(vgaCrtData, 1, 1);
    ioperm(VGAATTR_DATA, 1, 1);
    ioperm(VGAGRP_DATA, 1, 1);
    ioperm(VGASEQ_DATA, 1, 1);
    ioperm(VGAMISC_IN, 1, 1);
    ioperm(VGAMISC_OUT, 1, 1);
    ioperm(vgaSt1Addr, 1, 1);
    ioperm(VGAPAL_DATA, 1, 1);

    if ((devMem = open("/dev/mem", O_RDWR) ) < 0) 
    {
	Perror("/dev/mem");
	return FAILURE;
    }

#elif defined(__FreeBSD__)

    if (ioctl(0, KDENABIO,0) < 0) 
    {
	Perror("ioctl CONSOLE_IO_ENABLE");
	return FAILURE;
    }

    if (ioctl(0, SW_CG640x480, 0) < 0 ) {
       PerrorExit("SW_CG640x480");
       return FAILURE;
    }

    if ((devMem = open("/dev/vga", O_RDWR|O_NDELAY) ) < 0) 
    {
	Perror("/dev/mem");
	return FAILURE;
    }

#endif

    gramMem = (unsigned char *)mmap(
#if defined(linux)
				    (__ptr_t)0,
#else
				    0,
#endif
				    dispInfo.gsize, //  GRAPH_MEM,
				    PROT_READ|PROT_WRITE,
#if defined(linux)
				    MAP_SHARED,
#elif defined(__FreeBSD__)
/*				    MAP_FILE|MAP_SHARED,
*/				    MAP_SHARED,
#endif
				    devMem,
				    GRAPH_BASE
				    );
    close(devMem);

    if ((long)gramMem < 0) 
    {
	Perror("mmap");
	return FAILURE;
    }
#ifdef __Use_lrmi__
    if ( LRMI_init() == -1 ) {
       Perror("LRMI_init");
       return FAILURE;
    }

    ioperm(0, 0x400, 1);
    iopl(3);
#endif
    //VgaClearInput();
    return SUCCESS;
}

void VgaDetach(void)
{
#if defined(linux)
    ioperm(vgaCrtAddr, 1, 0);
    ioperm(VGAATTR_A_O, 1, 0);
    ioperm(VGAGRP_ADDR, 1, 0);
    ioperm(VGASEQ_ADDR, 1, 0);
    ioperm(VGAPAL_OADR, 1, 0);
    ioperm(VGAPAL_IADR, 1, 0);
    ioperm(vgaCrtData, 1, 0);
    ioperm(VGAATTR_DATA, 1, 0);
    ioperm(VGAGRP_DATA, 1, 0);
    ioperm(VGASEQ_DATA, 1, 0);
    ioperm(VGAMISC_IN, 1, 0);
    ioperm(VGAMISC_OUT, 1, 0);
    ioperm(vgaSt1Addr, 1, 0);
    ioperm(VGAPAL_DATA, 1, 0);
#endif

    munmap(gramMem, dispInfo.gsize);
    
    ioctl(0, KDSETMODE, KD_TEXT);

   // SafeFree((void **)&gramMem);
    /* Can't Free gramMem because it's not allocated, but mmap ,
       so I comment it */

    SafeFree((void **)&fontBuff1);
    if (savePlane3)
	SafeFree((void **)&fontBuff2);
}

/***********************************************************************
 *                           Configure                                 *
 ***********************************************************************/

static int ConfigPlane3(const char *confstr)
{
    savePlane3 = BoolConf(confstr);
    return SUCCESS;
}

int ConfigKanjiCursor(const char *confstr)
{
    kanjiCursor = BoolConf(confstr);
    return SUCCESS;
}

int ConfigCursorTop(const char *confstr)
{
    cursorTop = atoi(confstr);
    return SUCCESS;
}

int ConfigCursorBottom(const char *confstr)
{
    cursorBtm = atoi(confstr);
    return SUCCESS;
}

int ConfigScreen(const char *confstr)
{
    /* Rows Columns LineGap */
    sscanf(confstr, "%d %d %d", &dispInfo.tymax, &dispInfo.txmax,
           &dispInfo.linegap);

    if (pVideoInfo == NULL)
  	fatal("error reading" CONFIG_NAME);
    else pVideoInfo->init();

    /* we need to know the user-specified tymax & txmax, so I put init here */

    return SUCCESS;
}

void VgaSetStartAddress()
{
#ifndef __Use_lrmi__
    int	til;
    int gramHead = 0;
 
    PortOutw((gramHead  & 0xff00) | 0x0c, vgaCrtAddr);
    PortOutw((gramHead << 8) | 0x0d, vgaCrtAddr);

    til = (dispInfo.gydim - 1  /* - (gramHead / dispInfo.glineByte) */) << 4;
   // til = (18*25 - 1) << 4;

    /* Split screen bit 0-9, 10 bit 0-1023 */
    PortOutw((til << 4) | 0x18, vgaCrtAddr);

    PortOutw((til & 0x1000) | LineComp8, vgaCrtAddr);

             /* LineComp8's Low 8 bits is 0x07, split bit 7 X
             LineComp8 = ((regGraph.crt[7] & 0xEF) << 8 ) + 0x07
                 YYYXYYYY00000111 */

    PortOutw(((til & 0x2000) << 1) | LineComp9, vgaCrtAddr);

             /* LineComp9's Low 8 bits is 0x09, split bit 8 X
             LineComp9 = ((regGraph.crt[9] & 0xBF) << 8) + 0x09  
                YXYYYYYYY00001001 */
#endif
}

void DisplayDefaultCaps(void)
{
    DefineCap("SavePlane3", ConfigPlane3, "Off");
    DefineCap("Screen", ConfigScreen, "25 80 2");

    DefineCap("KanjiCursor", ConfigKanjiCursor, "On");
    DefineCap("CursorTop", ConfigCursorTop, "14");
    DefineCap("CursorBottom", ConfigCursorBottom, "15");

    DefineCap("SavePlane3", ConfigPlane3, "Off");
}

void VgaInit(void)
{
    VgaInitRegs();

    VgaAttach();    
}


VideoInfo VgaInfo =
{
 //   TRUE,
    VgaInit,
    VgaStart,
    VgaTextMode,
    VgaGraphMode,
    VgaWput,
    VgaInputWput,
    VgaSput,
    VgaInputSput,
    VgaSetCursorAddress,
    VgaSetAddress,
    VgaSetInputAddress,
    VgaCursor,
    VgaClearAll,
    VgaClearInput,
    VgaScreenSaver,
    VgaDetach,
    /*
    VgaSetStartAddress,
    VgaHardScrollUp,
    VgaHardScrollDown,
    VgaResetHardScroll
    */
};


/****************************************************************************
 *                  Special Handling for VgaS3 Video Cards                     *
 ****************************************************************************/

#define R5x_MASK	0x7737

union s3Regs 
{
    struct 
    {
	u_char r3[10];/* Video Atribute (CR30-34, CR38-3C) */
	u_char rx[33];/* Video Atribute (CR40-65) */
    } x;
    struct 
    {
	u_char
	    x30, x31, x32, x33, x34,
	    x38, x39, x3a, x3b, x3c;
	u_char
	    x40, x41, x42, x43, x44, x45, x46, x47,
	    x48, x49, x4a, x4b, x4c, x4d, x4e, x4f,
	    x50, x51, x53, x54, x55, x58, x59, x5a, x5c, x5d, x5e,
	    x60, x61, x62, x63, x64, x65;
    } r;
} s3Text, s3Graph;

static void VgaS3SetRegisters(union s3Regs *regs)
{
    int i, n;

    PortOutw(0xa539, vgaCrtAddr); /* unlock system control regs */
    for (i = 0; i < 5; i ++) 
    {
	PortOutb(0x30 + i, vgaCrtAddr);
	PortOutb(regs->x.r3[i], vgaCrtData);
	PortOutb(0x38 + i, vgaCrtAddr);
	PortOutb(regs->x.r3[5+i], vgaCrtData);
    }

    for (i = 0; i < 16; i ++) 
    {
	PortOutb(0x40 + i, vgaCrtAddr);
	PortOutb(regs->x.rx[i], vgaCrtData);
    }
    for (n = 16, i = 0; i < 16; i ++) 
    {
	if ((1 << i) & R5x_MASK) 
        {
	    PortOutb(0x50 + i, vgaCrtAddr);
	    PortOutb(regs->x.rx[n], vgaCrtData);
	    n ++;
	}
    }
    for (i = 0; i < 6; i ++, n ++) 
    {
	PortOutb(0x60 + i, vgaCrtAddr);
	PortOutb(s3Text.x.rx[n], vgaCrtData);
    }
}

static void VgaS3SetStartAddress(void)
{
    u_int til;
    int gramHead = 0;

    PortOutb(0x31, vgaCrtAddr);
    PortOutb(((gramHead & 0x030000) >> 12) | s3Graph.r.x31, vgaCrtData);

    s3Graph.r.x51 &= ~0x03;
    s3Graph.r.x51 |= ((gramHead & 0x040000) >> 18);
    PortOutb(0x51, vgaCrtAddr);
    /* Don't override current bank selection */
    PortOutb((PortInb(vgaCrtData) & ~0x03)
	     | ((gramHead & 0x40000) >> 18), vgaCrtData);

    PortOutw((gramHead & 0xFF00) | 0x0c, vgaCrtAddr);
    PortOutw(((gramHead & 0x00FF) << 8) | 0x0d, vgaCrtAddr);
    
    til = dispInfo.gydim - 1 /* - (gramHead / dispInfo.glineByte) */;
 //   til = 18 * 25 -1 ;
    PortOutw((til << 8) | 0x18, vgaCrtAddr);
    PortOutw(((til & 0x100) << 4) | LineComp8, vgaCrtAddr);
    PortOutw(((til & 0x200) << 5) | LineComp9, vgaCrtAddr);
    PortOutw(0x8d31, vgaCrtAddr); /* unlock system control regs */
}

static void VgaS3TextMode(void)
{
    VgaTextMode();
    VgaS3SetRegisters(&s3Text);
}

static void VgaS3InitRegs(void)
{
    regGraph.mis |= 0x0D;
/*    regGraph.crt[19] = 0xA0;*/
/*    regGraph.crt[20] = 0xA0;*/
    regGraph.crt[23] = 0xE3;
/*    regGraph.crt[24] = 0;*/

    s3Graph.r.x5e = (((video.vTotal - 2) & 0x400) >> 10)
	| (((video.vLine - 1) & 0x400) >> 9)
	    | ((video.vStart & 0x400) >> 8)
		| ((video.vStart & 0x400) >> 6) | 0x40;
    s3Graph.r.x5d = ((video.hTotal & 0x800) >> 11)
	| ((video.hDot & 0x800) >> 10)
	    | ((video.hStart & 0x800) >> 9)
		| ((video.hStart & 0x800) >> 7);
}

static void VgaS3GraphMode(void)
{
/*    s3Graph.r.x35 = s3Text.r.x35 & 0xF0;*/

    s3Graph.r.x5c = 0x20;
    s3Graph.r.x31 = 0x8D;
    s3Graph.r.x32 = 0;
    s3Graph.r.x33 = 0x20;
    s3Graph.r.x34 = 0x10;
/*    s3Graph.r.x35 = 0;*/
/*    s3Graph.r.x3a = 0x95;*/
    s3Graph.r.x3b = (regGraph.crt[0] + regGraph.crt[4] + 1) / 2;
    s3Graph.r.x3c = regGraph.crt[0] / 2;
    s3Graph.r.x40 = (s3Text.r.x40 & 0xF6) | 1;
    s3Graph.r.x43 = s3Text.r.x44 = 0;
    s3Graph.r.x45 = s3Text.r.x45 & 1;

    s3Graph.r.x50 = s3Text.r.x50 & ~0xC1;
    s3Graph.r.x51 = (s3Text.r.x51 & 0xC0) | ((dispInfo.gxdim >> 7) & 0x30);
    s3Graph.r.x53 = s3Text.r.x53 & ~0x30;
    s3Graph.r.x54 = 0xA0;
    s3Graph.r.x55 = (s3Text.r.x55 & 8) | 0x40;
    s3Graph.r.x58 = 0;
    s3Graph.r.x5d |= s3Graph.r.x5d & ~0x17;
    s3Graph.r.x60 = 0x3F;
    s3Graph.r.x61 = 0x81;
    s3Graph.r.x62 = 0;
    if (dispInfo.gxdim < 800) {
	s3Graph.r.x50 |= 0x40;
	s3Graph.r.x42 = 0xb;
    } else if (dispInfo.gxdim < 1024) {
	s3Graph.r.x50 |= 0x80;
	s3Graph.r.x42 = 2;
    } else {
	s3Graph.r.x42 = 0xE;
    }
    VgaGraphMode();

    VgaS3SetRegisters(&s3Graph);
}

static void VgaS3Start()
{
    int i, n;

    PortOutw(0xa539, vgaCrtAddr); /* unlock system control regs */
//    PortOutw(0x483b, vgaCrtAddr); /* unlock system control regs */

    for (i = 0; i < 5; i ++)
    {
        PortOutb(0x30 + i, vgaCrtAddr);
        s3Text.x.r3[i] = PortInb(vgaCrtData);
        PortOutb(0x38 + i, vgaCrtAddr);
        s3Text.x.r3[i+5] = PortInb(vgaCrtData);
    }
    for (i = 0; i < 16; i ++)
    {
        PortOutb(0x40 + i, vgaCrtAddr);
        s3Text.x.rx[i] = PortInb(vgaCrtData);
    }
    for (n = 16, i = 0; i < 16; i ++)
    {
        if ((1 << i) & R5x_MASK)
        {
            PortOutb(0x50 + i, vgaCrtAddr);
            s3Text.x.rx[n] = PortInb(vgaCrtData);
            n ++;
        }
    }
    for (i = 0; i < 6; i ++, n ++)
    {
        PortOutb(0x60 + i, vgaCrtAddr);
        s3Text.x.rx[n] = PortInb(vgaCrtData);
    }
    s3Graph = s3Text;
    s3Graph.r.x39 = 0xA5;

    VgaStart();
}

static void VgaS3Init(void)
{
    VgaInitRegs();

    VgaS3InitRegs();

    VgaAttach();
}


VideoInfo VgaS3Info =
{
   // TRUE,
    VgaS3Init,
    VgaS3Start,
    VgaS3TextMode,
    VgaS3GraphMode,
    VgaWput,
    VgaInputWput,
    VgaSput,
    VgaInputSput,
    VgaSetCursorAddress,
    VgaSetAddress,
    VgaSetInputAddress,
    VgaCursor,
    VgaClearAll,
    VgaClearInput,
    VgaScreenSaver,
    VgaDetach
//    VgaS3SetStartAddress,
};

