;;  Filename : test-number-literal.scm
;;  About    : unit test for R5RS number literals
;;
;;  Copyright (C) 2006 YAMAMOTO Kengo <yamaken AT bp.iij4u.or.jp>
;;  Copyright (c) 2007-2008 SigScheme Project <uim-en AT googlegroups.com>
;;
;;  All rights reserved.
;;
;;  Redistribution and use in source and binary forms, with or without
;;  modification, are permitted provided that the following conditions
;;  are met:
;;
;;  1. Redistributions of source code must retain the above copyright
;;     notice, this list of conditions and the following disclaimer.
;;  2. Redistributions in binary form must reproduce the above copyright
;;     notice, this list of conditions and the following disclaimer in the
;;     documentation and/or other materials provided with the distribution.
;;  3. Neither the name of authors nor the names of its contributors
;;     may be used to endorse or promote products derived from this software
;;     without specific prior written permission.
;;
;;  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
;;  IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
;;  THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
;;  PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
;;  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
;;  EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
;;  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
;;  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
;;  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
;;  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
;;  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

(require-extension (unittest))

(if (not (symbol-bound? 'number?))
    (test-skip "R5RS numbers is not enabled"))

(define tn test-name)


(tn "integer implicit decimal")
(assert-equal? (tn) 0  (string-read   "-0"))
(assert-equal? (tn) 0  (string-read    "0"))
(assert-equal? (tn) 0  (string-read   "+0"))
(assert-equal? (tn) 0  (string-read  "-00"))
(assert-equal? (tn) 0  (string-read   "00"))
(assert-equal? (tn) 0  (string-read  "+00"))
(assert-equal? (tn) -9 (string-read   "-9"))
(assert-equal? (tn) 9  (string-read    "9"))
(assert-equal? (tn) 9  (string-read   "+9"))
(assert-equal? (tn) -9 (string-read  "-09"))
(assert-equal? (tn) 9  (string-read   "09"))
(assert-equal? (tn) 9  (string-read  "+09"))
(assert-equal? (tn) 0  (string-eval  "'-0"))
(assert-equal? (tn) 0  (string-eval   "'0"))
(assert-equal? (tn) 0  (string-eval  "'+0"))
(assert-equal? (tn) 0  (string-eval "'-00"))
(assert-equal? (tn) 0  (string-eval  "'00"))
(assert-equal? (tn) 0  (string-eval "'+00"))
(assert-equal? (tn) -9 (string-eval  "'-9"))
(assert-equal? (tn) 9  (string-eval   "'9"))
(assert-equal? (tn) 9  (string-eval  "'+9"))
(assert-equal? (tn) -9 (string-eval "'-09"))
(assert-equal? (tn) 9  (string-eval  "'09"))
(assert-equal? (tn) 9  (string-eval "'+09"))
(assert-equal? (tn)  238975 (string-read  "0238975"))
(assert-equal? (tn)  238975 (string-read   "238975"))
(assert-equal? (tn)  238975 (string-read  "+238975"))
(assert-equal? (tn)  238975 (string-read "+0238975"))
(assert-equal? (tn) -238975 (string-read  "-238975"))
(assert-equal? (tn) -238975 (string-read "-0238975"))
(if (>= fixnum-bits 60)
    (begin
      (tn "integer implicit decimal 64-bit")
      (string-eval "(assert-equal? (tn)  956397711204 (string-read  \"0956397711204\"))")
      (string-eval "(assert-equal? (tn)  956397711204 (string-read   \"956397711204\"))")
      (string-eval "(assert-equal? (tn)  956397711204 (string-read  \"+956397711204\"))")
      (string-eval "(assert-equal? (tn)  956397711204 (string-read \"+0956397711204\"))")
      (string-eval "(assert-equal? (tn) -956397711204 (string-read  \"-956397711204\"))")
      (string-eval "(assert-equal? (tn) -956397711204 (string-read \"-0956397711204\"))")))
(tn "integer explicit decimal")
(assert-equal? (tn) 0  (string-read   "#d-0"))
(assert-equal? (tn) 0  (string-read    "#d0"))
(assert-equal? (tn) 0  (string-read   "#d+0"))
(assert-equal? (tn) 0  (string-read  "#d-00"))
(assert-equal? (tn) 0  (string-read   "#d00"))
(assert-equal? (tn) 0  (string-read  "#d+00"))
(assert-equal? (tn) -9 (string-read   "#d-9"))
(assert-equal? (tn) 9  (string-read    "#d9"))
(assert-equal? (tn) 9  (string-read   "#d+9"))
(assert-equal? (tn) -9 (string-read  "#d-09"))
(assert-equal? (tn) 9  (string-read   "#d09"))
(assert-equal? (tn) 9  (string-read  "#d+09"))
(assert-equal? (tn) 0  (string-eval  "'#d-0"))
(assert-equal? (tn) 0  (string-eval   "'#d0"))
(assert-equal? (tn) 0  (string-eval  "'#d+0"))
(assert-equal? (tn) 0  (string-eval "'#d-00"))
(assert-equal? (tn) 0  (string-eval  "'#d00"))
(assert-equal? (tn) 0  (string-eval "'#d+00"))
(assert-equal? (tn) -9 (string-eval  "'#d-9"))
(assert-equal? (tn) 9  (string-eval   "'#d9"))
(assert-equal? (tn) 9  (string-eval  "'#d+9"))
(assert-equal? (tn) -9 (string-eval "'#d-09"))
(assert-equal? (tn) 9  (string-eval  "'#d09"))
(assert-equal? (tn) 9  (string-eval "'#d+09"))
(assert-equal? (tn)  238975 (string-read  "#d0238975"))
(assert-equal? (tn)  238975 (string-read   "#d238975"))
(assert-equal? (tn)  238975 (string-read  "#d+238975"))
(assert-equal? (tn)  238975 (string-read "#d+0238975"))
(assert-equal? (tn) -238975 (string-read  "#d-238975"))
(assert-equal? (tn) -238975 (string-read "#d-0238975"))
(if (>= fixnum-bits 60)
    (begin
      (tn "integer explicit decimal 64-bit")
      (string-eval "(assert-equal? (tn)  956397711204 (string-read  \"#d0956397711204\"))")
      (string-eval "(assert-equal? (tn)  956397711204 (string-read   \"#d956397711204\"))")
      (string-eval "(assert-equal? (tn)  956397711204 (string-read  \"#d+956397711204\"))")
      (string-eval "(assert-equal? (tn)  956397711204 (string-read \"#d+0956397711204\"))")
      (string-eval "(assert-equal? (tn) -956397711204 (string-read  \"#d-956397711204\"))")
      (string-eval "(assert-equal? (tn) -956397711204 (string-read \"#d-0956397711204\"))")))
(tn "integer hexadecimal")
(assert-equal? (tn) 0   (string-read   "#x-0"))
(assert-equal? (tn) 0   (string-read    "#x0"))
(assert-equal? (tn) 0   (string-read   "#x+0"))
(assert-equal? (tn) 0   (string-read  "#x-00"))
(assert-equal? (tn) 0   (string-read   "#x00"))
(assert-equal? (tn) 0   (string-read  "#x+00"))
(assert-equal? (tn) -15 (string-read   "#x-f"))
(assert-equal? (tn) 15  (string-read    "#xf"))
(assert-equal? (tn) 15  (string-read   "#x+f"))
(assert-equal? (tn) -15 (string-read  "#x-0f"))
(assert-equal? (tn) 15  (string-read   "#x0f"))
(assert-equal? (tn) 15  (string-read  "#x+0f"))
(assert-equal? (tn) -15 (string-read   "#x-F"))
(assert-equal? (tn) 15  (string-read    "#xF"))
(assert-equal? (tn) 15  (string-read   "#x+F"))
(assert-equal? (tn) -15 (string-read  "#x-0F"))
(assert-equal? (tn) 15  (string-read   "#x0F"))
(assert-equal? (tn) 15  (string-read  "#x+0F"))
(assert-equal? (tn) 0   (string-eval  "'#x-0"))
(assert-equal? (tn) 0   (string-eval   "'#x0"))
(assert-equal? (tn) 0   (string-eval  "'#x+0"))
(assert-equal? (tn) 0   (string-eval "'#x-00"))
(assert-equal? (tn) 0   (string-eval  "'#x00"))
(assert-equal? (tn) 0   (string-eval "'#x+00"))
(assert-equal? (tn) -15 (string-eval  "'#x-f"))
(assert-equal? (tn) 15  (string-eval   "'#xf"))
(assert-equal? (tn) 15  (string-eval  "'#x+f"))
(assert-equal? (tn) -15 (string-eval "'#x-0f"))
(assert-equal? (tn) 15  (string-eval  "'#x0f"))
(assert-equal? (tn) 15  (string-eval "'#x+0f"))
(assert-equal? (tn) -15 (string-eval  "'#x-F"))
(assert-equal? (tn) 15  (string-eval   "'#xF"))
(assert-equal? (tn) 15  (string-eval  "'#x+F"))
(assert-equal? (tn) -15 (string-eval "'#x-0F"))
(assert-equal? (tn) 15  (string-eval  "'#x0F"))
(assert-equal? (tn) 15  (string-eval "'#x+0F"))
(assert-equal? (tn)  14593330 (string-read  "#x0deAd32"))
(assert-equal? (tn)  14593330 (string-read   "#xdeAd32"))
(assert-equal? (tn)  14593330 (string-read  "#x+dEad32"))
(assert-equal? (tn)  14593330 (string-read "#x+0dEad32"))
(assert-equal? (tn) -14593330 (string-read  "#x-deaD32"))
(assert-equal? (tn) -14593330 (string-read "#x-0deaD32"))
(assert-equal? (tn)   3333805 (string-read   "#x32deAd"))
(assert-equal? (tn)   3333805 (string-read  "#x+32dEad"))
(assert-equal? (tn)  -3333805 (string-read  "#x-32deaD"))
(if (>= fixnum-bits 60)
    (begin
      (tn "integer hexadecimal 64-bit")
      (string-eval "(assert-equal? (tn)  956397711204 (string-read  \"#x0deadbeef64\"))")
      (string-eval "(assert-equal? (tn)  956397711204 (string-read   \"#xdeadbeef64\"))")
      (string-eval "(assert-equal? (tn)  956397711204 (string-read  \"#x+deadbeef64\"))")
      (string-eval "(assert-equal? (tn)  956397711204 (string-read \"#x+0deadbeef64\"))")
      (string-eval "(assert-equal? (tn) -956397711204 (string-read  \"#x-deadbeef64\"))")
      (string-eval "(assert-equal? (tn) -956397711204 (string-read \"#x-0deadbeef64\"))")))
(tn "integer octal")
(assert-equal? (tn) 0  (string-read   "#o-0"))
(assert-equal? (tn) 0  (string-read    "#o0"))
(assert-equal? (tn) 0  (string-read   "#o+0"))
(assert-equal? (tn) 0  (string-read  "#o-00"))
(assert-equal? (tn) 0  (string-read   "#o00"))
(assert-equal? (tn) 0  (string-read  "#o+00"))
(assert-equal? (tn) -7 (string-read   "#o-7"))
(assert-equal? (tn) 7  (string-read    "#o7"))
(assert-equal? (tn) 7  (string-read   "#o+7"))
(assert-equal? (tn) -7 (string-read  "#o-07"))
(assert-equal? (tn) 7  (string-read   "#o07"))
(assert-equal? (tn) 7  (string-read  "#o+07"))
(assert-equal? (tn) 0  (string-eval  "'#o-0"))
(assert-equal? (tn) 0  (string-eval   "'#o0"))
(assert-equal? (tn) 0  (string-eval  "'#o+0"))
(assert-equal? (tn) 0  (string-eval "'#o-00"))
(assert-equal? (tn) 0  (string-eval  "'#o00"))
(assert-equal? (tn) 0  (string-eval "'#o+00"))
(assert-equal? (tn) -7 (string-eval  "'#o-7"))
(assert-equal? (tn) 7  (string-eval   "'#o7"))
(assert-equal? (tn) 7  (string-eval  "'#o+7"))
(assert-equal? (tn) -7 (string-eval "'#o-07"))
(assert-equal? (tn) 7  (string-eval  "'#o07"))
(assert-equal? (tn) 7  (string-eval "'#o+07"))
(assert-equal? (tn)  1556392 (string-read  "#o05737650"))
(assert-equal? (tn)  1556392 (string-read   "#o5737650"))
(assert-equal? (tn)  1556392 (string-read  "#o+5737650"))
(assert-equal? (tn)  1556392 (string-read "#o+05737650"))
(assert-equal? (tn) -1556392 (string-read  "#o-5737650"))
(assert-equal? (tn) -1556392 (string-read "#o-05737650"))
(if (>= fixnum-bits 60)
    (begin
      (tn "integer octal 64-bit")
      (string-eval "(assert-equal? (tn)  956397711204 (string-read  \"#o015725557567544\"))")
      (string-eval "(assert-equal? (tn)  956397711204 (string-read   \"#o15725557567544\"))")
      (string-eval "(assert-equal? (tn)  956397711204 (string-read  \"#o+15725557567544\"))")
      (string-eval "(assert-equal? (tn)  956397711204 (string-read \"#o+015725557567544\"))")
      (string-eval "(assert-equal? (tn) -956397711204 (string-read  \"#o-15725557567544\"))")
      (string-eval "(assert-equal? (tn) -956397711204 (string-read \"#o-015725557567544\"))")))
(tn "integer binary")
(assert-equal? (tn) 0  (string-read   "#b-0"))
(assert-equal? (tn) 0  (string-read    "#b0"))
(assert-equal? (tn) 0  (string-read   "#b+0"))
(assert-equal? (tn) 0  (string-read  "#b-00"))
(assert-equal? (tn) 0  (string-read   "#b00"))
(assert-equal? (tn) 0  (string-read  "#b+00"))
(assert-equal? (tn) -1 (string-read   "#b-1"))
(assert-equal? (tn) 1  (string-read    "#b1"))
(assert-equal? (tn) 1  (string-read   "#b+1"))
(assert-equal? (tn) -1 (string-read  "#b-01"))
(assert-equal? (tn) 1  (string-read   "#b01"))
(assert-equal? (tn) 1  (string-read  "#b+01"))
(assert-equal? (tn) 0  (string-eval  "'#b-0"))
(assert-equal? (tn) 0  (string-eval   "'#b0"))
(assert-equal? (tn) 0  (string-eval  "'#b+0"))
(assert-equal? (tn) 0  (string-eval "'#b-00"))
(assert-equal? (tn) 0  (string-eval  "'#b00"))
(assert-equal? (tn) 0  (string-eval "'#b+00"))
(assert-equal? (tn) -1 (string-eval  "'#b-1"))
(assert-equal? (tn) 1  (string-eval   "'#b1"))
(assert-equal? (tn) 1  (string-eval  "'#b+1"))
(assert-equal? (tn) -1 (string-eval "'#b-01"))
(assert-equal? (tn) 1  (string-eval  "'#b01"))
(assert-equal? (tn) 1  (string-eval "'#b+01"))
(assert-equal? (tn)  2990842 (string-read  "#b01011011010001011111010"))
(assert-equal? (tn)  2990842 (string-read   "#b1011011010001011111010"))
(assert-equal? (tn)  2990842 (string-read  "#b+1011011010001011111010"))
(assert-equal? (tn)  2990842 (string-read "#b+01011011010001011111010"))
(assert-equal? (tn) -2990842 (string-read  "#b-1011011010001011111010"))
(assert-equal? (tn) -2990842 (string-read "#b-01011011010001011111010"))
(if (>= fixnum-bits 60)
    (begin
      (tn "integer binary 64-bit")
      (string-eval "(assert-equal? (tn)  956397711204 (string-read  \"#b01101111010101101101111101110111101100100\"))")
      (string-eval "(assert-equal? (tn)  956397711204 (string-read   \"#b1101111010101101101111101110111101100100\"))")
      (string-eval "(assert-equal? (tn)  956397711204 (string-read  \"#b+1101111010101101101111101110111101100100\"))")
      (string-eval "(assert-equal? (tn)  956397711204 (string-read \"#b+01101111010101101101111101110111101100100\"))")
      (string-eval "(assert-equal? (tn) -956397711204 (string-read  \"#b-1101111010101101101111101110111101100100\"))")
      (string-eval "(assert-equal? (tn) -956397711204 (string-read \"#b-01101111010101101101111101110111101100100\"))")))
(case fixnum-bits
  ((28)
   (tn "integer limit implicit decimal")
   (assert-error  (tn) (lambda () (string-read     "-134217729")))
   (assert-error  (tn) (lambda () (string-read      "134217728")))
   (assert-error  (tn) (lambda () (string-read     "+134217728")))
   (assert-error  (tn) (lambda () (string-read    "-0134217729")))
   (assert-error  (tn) (lambda () (string-read     "0134217728")))
   (assert-error  (tn) (lambda () (string-read    "+0134217728")))
   (assert-error  (tn) (lambda () (string-read    "'-134217729")))
   (assert-error  (tn) (lambda () (string-read     "'134217728")))
   (assert-error  (tn) (lambda () (string-read    "'+134217728")))
   (assert-error  (tn) (lambda () (string-read   "'-0134217729")))
   (assert-error  (tn) (lambda () (string-read    "'0134217728")))
   (assert-error  (tn) (lambda () (string-read   "'+0134217728")))
   (assert-true   (tn) (integer?  (string-read     "-134217728")))
   (assert-true   (tn) (integer?  (string-read      "134217727")))
   (assert-true   (tn) (integer?  (string-read     "+134217727")))
   (assert-true   (tn) (integer?  (string-read    "-0134217728")))
   (assert-true   (tn) (integer?  (string-read     "0134217727")))
   (assert-true   (tn) (integer?  (string-read    "+0134217727")))
   (assert-true   (tn) (integer?  (string-eval    "'-134217728")))
   (assert-true   (tn) (integer?  (string-eval     "'134217727")))
   (assert-true   (tn) (integer?  (string-eval    "'+134217727")))
   (assert-true   (tn) (integer?  (string-eval   "'-0134217728")))
   (assert-true   (tn) (integer?  (string-eval    "'0134217727")))
   (assert-true   (tn) (integer?  (string-eval   "'+0134217727")))
   (tn "integer limit explicit decimal")
   (assert-error  (tn) (lambda () (string-read   "#d-134217729")))
   (assert-error  (tn) (lambda () (string-read    "#d134217728")))
   (assert-error  (tn) (lambda () (string-read   "#d+134217728")))
   (assert-error  (tn) (lambda () (string-read  "#d-0134217729")))
   (assert-error  (tn) (lambda () (string-read   "#d0134217728")))
   (assert-error  (tn) (lambda () (string-read  "#d+0134217728")))
   (assert-error  (tn) (lambda () (string-read  "'#d-134217729")))
   (assert-error  (tn) (lambda () (string-read   "'#d134217728")))
   (assert-error  (tn) (lambda () (string-read  "'#d+134217728")))
   (assert-error  (tn) (lambda () (string-read "'#d-0134217729")))
   (assert-error  (tn) (lambda () (string-read  "'#d0134217728")))
   (assert-error  (tn) (lambda () (string-read "'#d+0134217728")))
   (assert-true   (tn) (integer?  (string-read   "#d-134217728")))
   (assert-true   (tn) (integer?  (string-read    "#d134217727")))
   (assert-true   (tn) (integer?  (string-read   "#d+134217727")))
   (assert-true   (tn) (integer?  (string-read  "#d-0134217728")))
   (assert-true   (tn) (integer?  (string-read   "#d0134217727")))
   (assert-true   (tn) (integer?  (string-read  "#d+0134217727")))
   (assert-true   (tn) (integer?  (string-eval  "'#d-134217728")))
   (assert-true   (tn) (integer?  (string-eval   "'#d134217727")))
   (assert-true   (tn) (integer?  (string-eval  "'#d+134217727")))
   (assert-true   (tn) (integer?  (string-eval "'#d-0134217728")))
   (assert-true   (tn) (integer?  (string-eval  "'#d0134217727")))
   (assert-true   (tn) (integer?  (string-eval "'#d+0134217727")))
   (tn "integer limit hexadecimal")
   (assert-error  (tn) (lambda () (string-read   "#x-8000001")))
   (assert-error  (tn) (lambda () (string-read    "#x8000000")))
   (assert-error  (tn) (lambda () (string-read   "#x+8000000")))
   (assert-error  (tn) (lambda () (string-read  "#x-08000001")))
   (assert-error  (tn) (lambda () (string-read   "#x08000000")))
   (assert-error  (tn) (lambda () (string-read  "#x+08000000")))
   (assert-error  (tn) (lambda () (string-read  "'#x-8000001")))
   (assert-error  (tn) (lambda () (string-read   "'#x8000000")))
   (assert-error  (tn) (lambda () (string-read  "'#x+8000000")))
   (assert-error  (tn) (lambda () (string-read "'#x-08000001")))
   (assert-error  (tn) (lambda () (string-read  "'#x08000000")))
   (assert-error  (tn) (lambda () (string-read "'#x+08000000")))
   (assert-true   (tn) (integer?  (string-read   "#x-8000000")))
   (assert-true   (tn) (integer?  (string-read    "#x7ffffff")))
   (assert-true   (tn) (integer?  (string-read   "#x+7ffffff")))
   (assert-true   (tn) (integer?  (string-read  "#x-08000000")))
   (assert-true   (tn) (integer?  (string-read   "#x07ffffff")))
   (assert-true   (tn) (integer?  (string-read  "#x+07ffffff")))
   (assert-true   (tn) (integer?  (string-eval  "'#x-8000000")))
   (assert-true   (tn) (integer?  (string-eval   "'#x7ffffff")))
   (assert-true   (tn) (integer?  (string-eval  "'#x+7ffffff")))
   (assert-true   (tn) (integer?  (string-eval "'#x-08000000")))
   (assert-true   (tn) (integer?  (string-eval  "'#x07ffffff")))
   (assert-true   (tn) (integer?  (string-eval "'#x+07ffffff")))
   (tn "integer limit octal")
   (assert-error  (tn) (lambda () (string-read   "#o-1000000001")))
   (assert-error  (tn) (lambda () (string-read    "#o1000000000")))
   (assert-error  (tn) (lambda () (string-read   "#o+1000000000")))
   (assert-error  (tn) (lambda () (string-read  "#o-01000000001")))
   (assert-error  (tn) (lambda () (string-read   "#o01000000000")))
   (assert-error  (tn) (lambda () (string-read  "#o+01000000000")))
   (assert-error  (tn) (lambda () (string-read  "'#o-1000000001")))
   (assert-error  (tn) (lambda () (string-read   "'#o1000000000")))
   (assert-error  (tn) (lambda () (string-read  "'#o+1000000000")))
   (assert-error  (tn) (lambda () (string-read "'#o-01000000001")))
   (assert-error  (tn) (lambda () (string-read  "'#o01000000000")))
   (assert-error  (tn) (lambda () (string-read "'#o+01000000000")))
   (assert-true   (tn) (integer?  (string-read   "#o-1000000000")))
   (assert-true   (tn) (integer?  (string-read     "#o777777777")))
   (assert-true   (tn) (integer?  (string-read    "#o+777777777")))
   (assert-true   (tn) (integer?  (string-read  "#o-01000000000")))
   (assert-true   (tn) (integer?  (string-read    "#o0777777777")))
   (assert-true   (tn) (integer?  (string-read   "#o+0777777777")))
   (assert-true   (tn) (integer?  (string-eval  "'#o-1000000000")))
   (assert-true   (tn) (integer?  (string-eval    "'#o777777777")))
   (assert-true   (tn) (integer?  (string-eval   "'#o+777777777")))
   (assert-true   (tn) (integer?  (string-eval "'#o-01000000000")))
   (assert-true   (tn) (integer?  (string-eval   "'#o0777777777")))
   (assert-true   (tn) (integer?  (string-eval  "'#o+0777777777")))
   (tn "integer limit binary")
   (assert-error  (tn) (lambda () (string-read   "#b-1000000000000000000000000001")))
   (assert-error  (tn) (lambda () (string-read    "#b1000000000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read   "#b+1000000000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read  "#b-01000000000000000000000000001")))
   (assert-error  (tn) (lambda () (string-read   "#b01000000000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read  "#b+01000000000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read  "'#b-1000000000000000000000000001")))
   (assert-error  (tn) (lambda () (string-read   "'#b1000000000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read  "'#b+1000000000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read "'#b-01000000000000000000000000001")))
   (assert-error  (tn) (lambda () (string-read  "'#b01000000000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read "'#b+01000000000000000000000000000")))
   (assert-true   (tn) (integer?  (string-read   "#b-1000000000000000000000000000")))
   (assert-true   (tn) (integer?  (string-read     "#b111111111111111111111111111")))
   (assert-true   (tn) (integer?  (string-read    "#b+111111111111111111111111111")))
   (assert-true   (tn) (integer?  (string-read  "#b-01000000000000000000000000000")))
   (assert-true   (tn) (integer?  (string-read    "#b0111111111111111111111111111")))
   (assert-true   (tn) (integer?  (string-read   "#b+0111111111111111111111111111")))
   (assert-true   (tn) (integer?  (string-eval  "'#b-1000000000000000000000000000")))
   (assert-true   (tn) (integer?  (string-eval    "'#b111111111111111111111111111")))
   (assert-true   (tn) (integer?  (string-eval   "'#b+111111111111111111111111111")))
   (assert-true   (tn) (integer?  (string-eval "'#b-01000000000000000000000000000")))
   (assert-true   (tn) (integer?  (string-eval   "'#b0111111111111111111111111111")))
   (assert-true   (tn) (integer?  (string-eval  "'#b+0111111111111111111111111111"))))
  ((32)
   (tn "integer limit implicit decimal")
   (assert-error  (tn) (lambda () (string-read     "-2147483649")))
   (assert-error  (tn) (lambda () (string-read      "2147483648")))
   (assert-error  (tn) (lambda () (string-read     "+2147483648")))
   (assert-error  (tn) (lambda () (string-read    "-02147483649")))
   (assert-error  (tn) (lambda () (string-read     "02147483648")))
   (assert-error  (tn) (lambda () (string-read    "+02147483648")))
   (assert-error  (tn) (lambda () (string-read    "'-2147483649")))
   (assert-error  (tn) (lambda () (string-read     "'2147483648")))
   (assert-error  (tn) (lambda () (string-read    "'+2147483648")))
   (assert-error  (tn) (lambda () (string-read   "'-02147483649")))
   (assert-error  (tn) (lambda () (string-read    "'02147483648")))
   (assert-error  (tn) (lambda () (string-read   "'+02147483648")))
   (assert-true   (tn) (integer?  (string-read     "-2147483648")))
   (assert-true   (tn) (integer?  (string-read      "2147483647")))
   (assert-true   (tn) (integer?  (string-read     "+2147483647")))
   (assert-true   (tn) (integer?  (string-read    "-02147483648")))
   (assert-true   (tn) (integer?  (string-read     "02147483647")))
   (assert-true   (tn) (integer?  (string-read    "+02147483647")))
   (assert-true   (tn) (integer?  (string-eval    "'-2147483648")))
   (assert-true   (tn) (integer?  (string-eval     "'2147483647")))
   (assert-true   (tn) (integer?  (string-eval    "'+2147483647")))
   (assert-true   (tn) (integer?  (string-eval   "'-02147483648")))
   (assert-true   (tn) (integer?  (string-eval    "'02147483647")))
   (assert-true   (tn) (integer?  (string-eval   "'+02147483647")))
   (tn "integer limit explicit decimal")
   (assert-error  (tn) (lambda () (string-read   "#d-2147483649")))
   (assert-error  (tn) (lambda () (string-read    "#d2147483648")))
   (assert-error  (tn) (lambda () (string-read   "#d+2147483648")))
   (assert-error  (tn) (lambda () (string-read  "#d-02147483649")))
   (assert-error  (tn) (lambda () (string-read   "#d02147483648")))
   (assert-error  (tn) (lambda () (string-read  "#d+02147483648")))
   (assert-error  (tn) (lambda () (string-read  "'#d-2147483649")))
   (assert-error  (tn) (lambda () (string-read   "'#d2147483648")))
   (assert-error  (tn) (lambda () (string-read  "'#d+2147483648")))
   (assert-error  (tn) (lambda () (string-read "'#d-02147483649")))
   (assert-error  (tn) (lambda () (string-read  "'#d02147483648")))
   (assert-error  (tn) (lambda () (string-read "'#d+02147483648")))
   (assert-true   (tn) (integer?  (string-read   "#d-2147483648")))
   (assert-true   (tn) (integer?  (string-read    "#d2147483647")))
   (assert-true   (tn) (integer?  (string-read   "#d+2147483647")))
   (assert-true   (tn) (integer?  (string-read  "#d-02147483648")))
   (assert-true   (tn) (integer?  (string-read   "#d02147483647")))
   (assert-true   (tn) (integer?  (string-read  "#d+02147483647")))
   (assert-true   (tn) (integer?  (string-eval  "'#d-2147483648")))
   (assert-true   (tn) (integer?  (string-eval   "'#d2147483647")))
   (assert-true   (tn) (integer?  (string-eval  "'#d+2147483647")))
   (assert-true   (tn) (integer?  (string-eval "'#d-02147483648")))
   (assert-true   (tn) (integer?  (string-eval  "'#d02147483647")))
   (assert-true   (tn) (integer?  (string-eval "'#d+02147483647")))
   (tn "integer limit hexadecimal")
   (assert-error  (tn) (lambda () (string-read   "#x-80000001")))
   (assert-error  (tn) (lambda () (string-read    "#x80000000")))
   (assert-error  (tn) (lambda () (string-read   "#x+80000000")))
   (assert-error  (tn) (lambda () (string-read  "#x-080000001")))
   (assert-error  (tn) (lambda () (string-read   "#x080000000")))
   (assert-error  (tn) (lambda () (string-read  "#x+080000000")))
   (assert-error  (tn) (lambda () (string-read  "'#x-80000001")))
   (assert-error  (tn) (lambda () (string-read   "'#x80000000")))
   (assert-error  (tn) (lambda () (string-read  "'#x+80000000")))
   (assert-error  (tn) (lambda () (string-read "'#x-080000001")))
   (assert-error  (tn) (lambda () (string-read  "'#x080000000")))
   (assert-error  (tn) (lambda () (string-read "'#x+080000000")))
   (assert-true   (tn) (integer?  (string-read   "#x-80000000")))
   (assert-true   (tn) (integer?  (string-read    "#x7fffffff")))
   (assert-true   (tn) (integer?  (string-read   "#x+7fffffff")))
   (assert-true   (tn) (integer?  (string-read  "#x-080000000")))
   (assert-true   (tn) (integer?  (string-read   "#x07fffffff")))
   (assert-true   (tn) (integer?  (string-read  "#x+07fffffff")))
   (assert-true   (tn) (integer?  (string-eval  "'#x-80000000")))
   (assert-true   (tn) (integer?  (string-eval   "'#x7fffffff")))
   (assert-true   (tn) (integer?  (string-eval  "'#x+7fffffff")))
   (assert-true   (tn) (integer?  (string-eval "'#x-080000000")))
   (assert-true   (tn) (integer?  (string-eval  "'#x07fffffff")))
   (assert-true   (tn) (integer?  (string-eval "'#x+07fffffff")))
   (tn "integer limit octal")
   (assert-error  (tn) (lambda () (string-read   "#o-20000000001")))
   (assert-error  (tn) (lambda () (string-read    "#o20000000000")))
   (assert-error  (tn) (lambda () (string-read   "#o+20000000000")))
   (assert-error  (tn) (lambda () (string-read  "#o-020000000001")))
   (assert-error  (tn) (lambda () (string-read   "#o020000000000")))
   (assert-error  (tn) (lambda () (string-read  "#o+020000000000")))
   (assert-error  (tn) (lambda () (string-read  "'#o-20000000001")))
   (assert-error  (tn) (lambda () (string-read   "'#o20000000000")))
   (assert-error  (tn) (lambda () (string-read  "'#o+20000000000")))
   (assert-error  (tn) (lambda () (string-read "'#o-020000000001")))
   (assert-error  (tn) (lambda () (string-read  "'#o020000000000")))
   (assert-error  (tn) (lambda () (string-read "'#o+020000000000")))
   (assert-true   (tn) (integer?  (string-read   "#o-20000000000")))
   (assert-true   (tn) (integer?  (string-read    "#o17777777777")))
   (assert-true   (tn) (integer?  (string-read   "#o+17777777777")))
   (assert-true   (tn) (integer?  (string-read  "#o-020000000000")))
   (assert-true   (tn) (integer?  (string-read   "#o017777777777")))
   (assert-true   (tn) (integer?  (string-read  "#o+017777777777")))
   (assert-true   (tn) (integer?  (string-eval  "'#o-20000000000")))
   (assert-true   (tn) (integer?  (string-eval   "'#o17777777777")))
   (assert-true   (tn) (integer?  (string-eval  "'#o+17777777777")))
   (assert-true   (tn) (integer?  (string-eval "'#o-020000000000")))
   (assert-true   (tn) (integer?  (string-eval  "'#o017777777777")))
   (assert-true   (tn) (integer?  (string-eval "'#o+017777777777")))
   (tn "integer limit binary")
   (assert-error  (tn) (lambda () (string-read   "#b-10000000000000000000000000000001")))
   (assert-error  (tn) (lambda () (string-read    "#b10000000000000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read   "#b+10000000000000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read  "#b-010000000000000000000000000000001")))
   (assert-error  (tn) (lambda () (string-read   "#b010000000000000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read  "#b+010000000000000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read  "'#b-10000000000000000000000000000001")))
   (assert-error  (tn) (lambda () (string-read   "'#b10000000000000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read  "'#b+10000000000000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read "'#b-010000000000000000000000000000001")))
   (assert-error  (tn) (lambda () (string-read  "'#b010000000000000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read "'#b+010000000000000000000000000000000")))
   (assert-true   (tn) (integer?  (string-read   "#b-10000000000000000000000000000000")))
   (assert-true   (tn) (integer?  (string-read     "#b1111111111111111111111111111111")))
   (assert-true   (tn) (integer?  (string-read    "#b+1111111111111111111111111111111")))
   (assert-true   (tn) (integer?  (string-read  "#b-010000000000000000000000000000000")))
   (assert-true   (tn) (integer?  (string-read    "#b01111111111111111111111111111111")))
   (assert-true   (tn) (integer?  (string-read   "#b+01111111111111111111111111111111")))
   (assert-true   (tn) (integer?  (string-eval  "'#b-10000000000000000000000000000000")))
   (assert-true   (tn) (integer?  (string-eval    "'#b1111111111111111111111111111111")))
   (assert-true   (tn) (integer?  (string-eval   "'#b+1111111111111111111111111111111")))
   (assert-true   (tn) (integer?  (string-eval "'#b-010000000000000000000000000000000")))
   (assert-true   (tn) (integer?  (string-eval   "'#b01111111111111111111111111111111")))
   (assert-true   (tn) (integer?  (string-eval  "'#b+01111111111111111111111111111111"))))
  ((60)
   (tn "integer limit implicit decimal")
   (assert-error  (tn) (lambda () (string-read     "-576460752303423489")))
   (assert-error  (tn) (lambda () (string-read      "576460752303423488")))
   (assert-error  (tn) (lambda () (string-read     "+576460752303423488")))
   (assert-error  (tn) (lambda () (string-read    "-0576460752303423489")))
   (assert-error  (tn) (lambda () (string-read     "0576460752303423488")))
   (assert-error  (tn) (lambda () (string-read    "+0576460752303423488")))
   (assert-error  (tn) (lambda () (string-read    "'-576460752303423489")))
   (assert-error  (tn) (lambda () (string-read     "'576460752303423488")))
   (assert-error  (tn) (lambda () (string-read    "'+576460752303423488")))
   (assert-error  (tn) (lambda () (string-read   "'-0576460752303423489")))
   (assert-error  (tn) (lambda () (string-read    "'0576460752303423488")))
   (assert-error  (tn) (lambda () (string-read   "'+0576460752303423488")))
   (assert-true   (tn) (integer?  (string-read     "-576460752303423488")))
   (assert-true   (tn) (integer?  (string-read      "576460752303423487")))
   (assert-true   (tn) (integer?  (string-read     "+576460752303423487")))
   (assert-true   (tn) (integer?  (string-read    "-0576460752303423488")))
   (assert-true   (tn) (integer?  (string-read     "0576460752303423487")))
   (assert-true   (tn) (integer?  (string-read    "+0576460752303423487")))
   (assert-true   (tn) (integer?  (string-eval    "'-576460752303423488")))
   (assert-true   (tn) (integer?  (string-eval     "'576460752303423487")))
   (assert-true   (tn) (integer?  (string-eval    "'+576460752303423487")))
   (assert-true   (tn) (integer?  (string-eval   "'-0576460752303423488")))
   (assert-true   (tn) (integer?  (string-eval    "'0576460752303423487")))
   (assert-true   (tn) (integer?  (string-eval   "'+0576460752303423487")))
   (tn "integer limit explicit decimal")
   (assert-error  (tn) (lambda () (string-read   "#d-576460752303423489")))
   (assert-error  (tn) (lambda () (string-read    "#d576460752303423488")))
   (assert-error  (tn) (lambda () (string-read   "#d+576460752303423488")))
   (assert-error  (tn) (lambda () (string-read  "#d-0576460752303423489")))
   (assert-error  (tn) (lambda () (string-read   "#d0576460752303423488")))
   (assert-error  (tn) (lambda () (string-read  "#d+0576460752303423488")))
   (assert-error  (tn) (lambda () (string-read  "'#d-576460752303423489")))
   (assert-error  (tn) (lambda () (string-read   "'#d576460752303423488")))
   (assert-error  (tn) (lambda () (string-read  "'#d+576460752303423488")))
   (assert-error  (tn) (lambda () (string-read "'#d-0576460752303423489")))
   (assert-error  (tn) (lambda () (string-read  "'#d0576460752303423488")))
   (assert-error  (tn) (lambda () (string-read "'#d+0576460752303423488")))
   (assert-true   (tn) (integer?  (string-read   "#d-576460752303423488")))
   (assert-true   (tn) (integer?  (string-read    "#d576460752303423487")))
   (assert-true   (tn) (integer?  (string-read   "#d+576460752303423487")))
   (assert-true   (tn) (integer?  (string-read  "#d-0576460752303423488")))
   (assert-true   (tn) (integer?  (string-read   "#d0576460752303423487")))
   (assert-true   (tn) (integer?  (string-read  "#d+0576460752303423487")))
   (assert-true   (tn) (integer?  (string-eval  "'#d-576460752303423488")))
   (assert-true   (tn) (integer?  (string-eval   "'#d576460752303423487")))
   (assert-true   (tn) (integer?  (string-eval  "'#d+576460752303423487")))
   (assert-true   (tn) (integer?  (string-eval "'#d-0576460752303423488")))
   (assert-true   (tn) (integer?  (string-eval  "'#d0576460752303423487")))
   (assert-true   (tn) (integer?  (string-eval "'#d+0576460752303423487")))
   (tn "integer limit hexadecimal")
   (assert-error  (tn) (lambda () (string-read   "#x-800000000000001")))
   (assert-error  (tn) (lambda () (string-read    "#x800000000000000")))
   (assert-error  (tn) (lambda () (string-read   "#x+800000000000000")))
   (assert-error  (tn) (lambda () (string-read  "#x-0800000000000001")))
   (assert-error  (tn) (lambda () (string-read   "#x0800000000000000")))
   (assert-error  (tn) (lambda () (string-read  "#x+0800000000000000")))
   (assert-error  (tn) (lambda () (string-read  "'#x-800000000000001")))
   (assert-error  (tn) (lambda () (string-read   "'#x800000000000000")))
   (assert-error  (tn) (lambda () (string-read  "'#x+800000000000000")))
   (assert-error  (tn) (lambda () (string-read "'#x-0800000000000001")))
   (assert-error  (tn) (lambda () (string-read  "'#x0800000000000000")))
   (assert-error  (tn) (lambda () (string-read "'#x+0800000000000000")))
   (assert-true   (tn) (integer?  (string-read   "#x-800000000000000")))
   (assert-true   (tn) (integer?  (string-read    "#x7ffffffffffffff")))
   (assert-true   (tn) (integer?  (string-read   "#x+7ffffffffffffff")))
   (assert-true   (tn) (integer?  (string-read  "#x-0800000000000000")))
   (assert-true   (tn) (integer?  (string-read   "#x07ffffffffffffff")))
   (assert-true   (tn) (integer?  (string-read  "#x+07ffffffffffffff")))
   (assert-true   (tn) (integer?  (string-eval  "'#x-800000000000000")))
   (assert-true   (tn) (integer?  (string-eval   "'#x7ffffffffffffff")))
   (assert-true   (tn) (integer?  (string-eval  "'#x+7ffffffffffffff")))
   (assert-true   (tn) (integer?  (string-eval "'#x-0800000000000000")))
   (assert-true   (tn) (integer?  (string-eval  "'#x07ffffffffffffff")))
   (assert-true   (tn) (integer?  (string-eval "'#x+07ffffffffffffff")))
   (tn "integer limit octal")
   (assert-error  (tn) (lambda () (string-read   "#o-40000000000000000001")))
   (assert-error  (tn) (lambda () (string-read    "#o40000000000000000000")))
   (assert-error  (tn) (lambda () (string-read   "#o+40000000000000000000")))
   (assert-error  (tn) (lambda () (string-read  "#o-040000000000000000001")))
   (assert-error  (tn) (lambda () (string-read   "#o040000000000000000000")))
   (assert-error  (tn) (lambda () (string-read  "#o+040000000000000000000")))
   (assert-error  (tn) (lambda () (string-read  "'#o-40000000000000000001")))
   (assert-error  (tn) (lambda () (string-read   "'#o40000000000000000000")))
   (assert-error  (tn) (lambda () (string-read  "'#o+40000000000000000000")))
   (assert-error  (tn) (lambda () (string-read "'#o-040000000000000000001")))
   (assert-error  (tn) (lambda () (string-read  "'#o040000000000000000000")))
   (assert-error  (tn) (lambda () (string-read "'#o+040000000000000000000")))
   (assert-true   (tn) (integer?  (string-read   "#o-40000000000000000000")))
   (assert-true   (tn) (integer?  (string-read    "#o37777777777777777777")))
   (assert-true   (tn) (integer?  (string-read   "#o+37777777777777777777")))
   (assert-true   (tn) (integer?  (string-read  "#o-040000000000000000000")))
   (assert-true   (tn) (integer?  (string-read   "#o037777777777777777777")))
   (assert-true   (tn) (integer?  (string-read  "#o+037777777777777777777")))
   (assert-true   (tn) (integer?  (string-eval  "'#o-40000000000000000000")))
   (assert-true   (tn) (integer?  (string-eval   "'#o37777777777777777777")))
   (assert-true   (tn) (integer?  (string-eval  "'#o+37777777777777777777")))
   (assert-true   (tn) (integer?  (string-eval "'#o-040000000000000000000")))
   (assert-true   (tn) (integer?  (string-eval  "'#o037777777777777777777")))
   (assert-true   (tn) (integer?  (string-eval "'#o+037777777777777777777")))
   (tn "integer limit binary")
   (assert-error  (tn) (lambda () (string-read   "#b-100000000000000000000000000000000000000000000000000000000001")))
   (assert-error  (tn) (lambda () (string-read    "#b100000000000000000000000000000000000000000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read   "#b+100000000000000000000000000000000000000000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read  "#b-0100000000000000000000000000000000000000000000000000000000001")))
   (assert-error  (tn) (lambda () (string-read   "#b0100000000000000000000000000000000000000000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read  "#b+0100000000000000000000000000000000000000000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read  "'#b-100000000000000000000000000000000000000000000000000000000001")))
   (assert-error  (tn) (lambda () (string-read   "'#b100000000000000000000000000000000000000000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read  "'#b+100000000000000000000000000000000000000000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read "'#b-0100000000000000000000000000000000000000000000000000000000001")))
   (assert-error  (tn) (lambda () (string-read  "'#b0100000000000000000000000000000000000000000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read "'#b+0100000000000000000000000000000000000000000000000000000000000")))
   (assert-true   (tn) (integer?  (string-read   "#b-100000000000000000000000000000000000000000000000000000000000")))
   (assert-true   (tn) (integer?  (string-read     "#b11111111111111111111111111111111111111111111111111111111111")))
   (assert-true   (tn) (integer?  (string-read    "#b+11111111111111111111111111111111111111111111111111111111111")))
   (assert-true   (tn) (integer?  (string-read  "#b-0100000000000000000000000000000000000000000000000000000000000")))
   (assert-true   (tn) (integer?  (string-read    "#b011111111111111111111111111111111111111111111111111111111111")))
   (assert-true   (tn) (integer?  (string-read   "#b+011111111111111111111111111111111111111111111111111111111111")))
   (assert-true   (tn) (integer?  (string-eval  "'#b-100000000000000000000000000000000000000000000000000000000000")))
   (assert-true   (tn) (integer?  (string-eval    "'#b11111111111111111111111111111111111111111111111111111111111")))
   (assert-true   (tn) (integer?  (string-eval   "'#b+11111111111111111111111111111111111111111111111111111111111")))
   (assert-true   (tn) (integer?  (string-eval "'#b-0100000000000000000000000000000000000000000000000000000000000")))
   (assert-true   (tn) (integer?  (string-eval   "'#b011111111111111111111111111111111111111111111111111111111111")))
   (assert-true   (tn) (integer?  (string-eval  "'#b+011111111111111111111111111111111111111111111111111111111111"))))
  ((64)
   (tn "integer limit implicit decimal")
   (assert-error  (tn) (lambda () (string-read     "-9223372036854775809")))
   (assert-error  (tn) (lambda () (string-read      "9223372036854775808")))
   (assert-error  (tn) (lambda () (string-read     "+9223372036854775808")))
   (assert-error  (tn) (lambda () (string-read    "-09223372036854775809")))
   (assert-error  (tn) (lambda () (string-read     "09223372036854775808")))
   (assert-error  (tn) (lambda () (string-read    "+09223372036854775808")))
   (assert-error  (tn) (lambda () (string-read    "'-9223372036854775809")))
   (assert-error  (tn) (lambda () (string-read     "'9223372036854775808")))
   (assert-error  (tn) (lambda () (string-read    "'+9223372036854775808")))
   (assert-error  (tn) (lambda () (string-read   "'-09223372036854775809")))
   (assert-error  (tn) (lambda () (string-read    "'09223372036854775808")))
   (assert-error  (tn) (lambda () (string-read   "'+09223372036854775808")))
   (assert-true   (tn) (integer?  (string-read     "-9223372036854775808")))
   (assert-true   (tn) (integer?  (string-read      "9223372036854775807")))
   (assert-true   (tn) (integer?  (string-read     "+9223372036854775807")))
   (assert-true   (tn) (integer?  (string-read    "-09223372036854775808")))
   (assert-true   (tn) (integer?  (string-read     "09223372036854775807")))
   (assert-true   (tn) (integer?  (string-read    "+09223372036854775807")))
   (assert-true   (tn) (integer?  (string-eval    "'-9223372036854775808")))
   (assert-true   (tn) (integer?  (string-eval     "'9223372036854775807")))
   (assert-true   (tn) (integer?  (string-eval    "'+9223372036854775807")))
   (assert-true   (tn) (integer?  (string-eval   "'-09223372036854775808")))
   (assert-true   (tn) (integer?  (string-eval    "'09223372036854775807")))
   (assert-true   (tn) (integer?  (string-eval   "'+09223372036854775807")))
   (tn "integer limit explicit decimal")
   (assert-error  (tn) (lambda () (string-read   "#d-9223372036854775809")))
   (assert-error  (tn) (lambda () (string-read    "#d9223372036854775808")))
   (assert-error  (tn) (lambda () (string-read   "#d+9223372036854775808")))
   (assert-error  (tn) (lambda () (string-read  "#d-09223372036854775809")))
   (assert-error  (tn) (lambda () (string-read   "#d09223372036854775808")))
   (assert-error  (tn) (lambda () (string-read  "#d+09223372036854775808")))
   (assert-error  (tn) (lambda () (string-read  "'#d-9223372036854775809")))
   (assert-error  (tn) (lambda () (string-read   "'#d9223372036854775808")))
   (assert-error  (tn) (lambda () (string-read  "'#d+9223372036854775808")))
   (assert-error  (tn) (lambda () (string-read "'#d-09223372036854775809")))
   (assert-error  (tn) (lambda () (string-read  "'#d09223372036854775808")))
   (assert-error  (tn) (lambda () (string-read "'#d+09223372036854775808")))
   (assert-true   (tn) (integer?  (string-read   "#d-9223372036854775808")))
   (assert-true   (tn) (integer?  (string-read    "#d9223372036854775807")))
   (assert-true   (tn) (integer?  (string-read   "#d+9223372036854775807")))
   (assert-true   (tn) (integer?  (string-read  "#d-09223372036854775808")))
   (assert-true   (tn) (integer?  (string-read   "#d09223372036854775807")))
   (assert-true   (tn) (integer?  (string-read  "#d+09223372036854775807")))
   (assert-true   (tn) (integer?  (string-eval  "'#d-9223372036854775808")))
   (assert-true   (tn) (integer?  (string-eval   "'#d9223372036854775807")))
   (assert-true   (tn) (integer?  (string-eval  "'#d+9223372036854775807")))
   (assert-true   (tn) (integer?  (string-eval "'#d-09223372036854775808")))
   (assert-true   (tn) (integer?  (string-eval  "'#d09223372036854775807")))
   (assert-true   (tn) (integer?  (string-eval "'#d+09223372036854775807")))
   (tn "integer limit hexadecimal")
   (assert-error  (tn) (lambda () (string-read   "#x-8000000000000001")))
   (assert-error  (tn) (lambda () (string-read    "#x8000000000000000")))
   (assert-error  (tn) (lambda () (string-read   "#x+8000000000000000")))
   (assert-error  (tn) (lambda () (string-read  "#x-08000000000000001")))
   (assert-error  (tn) (lambda () (string-read   "#x08000000000000000")))
   (assert-error  (tn) (lambda () (string-read  "#x+08000000000000000")))
   (assert-error  (tn) (lambda () (string-read  "'#x-8000000000000001")))
   (assert-error  (tn) (lambda () (string-read   "'#x8000000000000000")))
   (assert-error  (tn) (lambda () (string-read  "'#x+8000000000000000")))
   (assert-error  (tn) (lambda () (string-read "'#x-08000000000000001")))
   (assert-error  (tn) (lambda () (string-read  "'#x08000000000000000")))
   (assert-error  (tn) (lambda () (string-read "'#x+08000000000000000")))
   (assert-true   (tn) (integer?  (string-read   "#x-8000000000000000")))
   (assert-true   (tn) (integer?  (string-read    "#x7fffffffffffffff")))
   (assert-true   (tn) (integer?  (string-read   "#x+7fffffffffffffff")))
   (assert-true   (tn) (integer?  (string-read  "#x-08000000000000000")))
   (assert-true   (tn) (integer?  (string-read   "#x07fffffffffffffff")))
   (assert-true   (tn) (integer?  (string-read  "#x+07fffffffffffffff")))
   (assert-true   (tn) (integer?  (string-eval  "'#x-8000000000000000")))
   (assert-true   (tn) (integer?  (string-eval   "'#x7fffffffffffffff")))
   (assert-true   (tn) (integer?  (string-eval  "'#x+7fffffffffffffff")))
   (assert-true   (tn) (integer?  (string-eval "'#x-08000000000000000")))
   (assert-true   (tn) (integer?  (string-eval  "'#x07fffffffffffffff")))
   (assert-true   (tn) (integer?  (string-eval "'#x+07fffffffffffffff")))
   (tn "integer limit octal")
   (assert-error  (tn) (lambda () (string-read   "#o-1000000000000000000001")))
   (assert-error  (tn) (lambda () (string-read    "#o1000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read   "#o+1000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read  "#o-01000000000000000000001")))
   (assert-error  (tn) (lambda () (string-read   "#o01000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read  "#o+01000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read  "'#o-1000000000000000000001")))
   (assert-error  (tn) (lambda () (string-read   "'#o1000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read  "'#o+1000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read "'#o-01000000000000000000001")))
   (assert-error  (tn) (lambda () (string-read  "'#o01000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read "'#o+01000000000000000000000")))
   (assert-true   (tn) (integer?  (string-read   "#o-1000000000000000000000")))
   (assert-true   (tn) (integer?  (string-read     "#o777777777777777777777")))
   (assert-true   (tn) (integer?  (string-read    "#o+777777777777777777777")))
   (assert-true   (tn) (integer?  (string-read  "#o-01000000000000000000000")))
   (assert-true   (tn) (integer?  (string-read    "#o0777777777777777777777")))
   (assert-true   (tn) (integer?  (string-read   "#o+0777777777777777777777")))
   (assert-true   (tn) (integer?  (string-eval  "'#o-1000000000000000000000")))
   (assert-true   (tn) (integer?  (string-eval    "'#o777777777777777777777")))
   (assert-true   (tn) (integer?  (string-eval   "'#o+777777777777777777777")))
   (assert-true   (tn) (integer?  (string-eval "'#o-01000000000000000000000")))
   (assert-true   (tn) (integer?  (string-eval   "'#o0777777777777777777777")))
   (assert-true   (tn) (integer?  (string-eval  "'#o+0777777777777777777777")))
   (tn "integer limit binary")
   (assert-error  (tn) (lambda () (string-read   "#b-1000000000000000000000000000000000000000000000000000000000000001")))
   (assert-error  (tn) (lambda () (string-read    "#b1000000000000000000000000000000000000000000000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read   "#b+1000000000000000000000000000000000000000000000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read  "#b-01000000000000000000000000000000000000000000000000000000000000001")))
   (assert-error  (tn) (lambda () (string-read   "#b01000000000000000000000000000000000000000000000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read  "#b+01000000000000000000000000000000000000000000000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read  "'#b-1000000000000000000000000000000000000000000000000000000000000001")))
   (assert-error  (tn) (lambda () (string-read   "'#b1000000000000000000000000000000000000000000000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read  "'#b+1000000000000000000000000000000000000000000000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read "'#b-01000000000000000000000000000000000000000000000000000000000000001")))
   (assert-error  (tn) (lambda () (string-read  "'#b01000000000000000000000000000000000000000000000000000000000000000")))
   (assert-error  (tn) (lambda () (string-read "'#b+01000000000000000000000000000000000000000000000000000000000000000")))
   (assert-true   (tn) (integer?  (string-read   "#b-1000000000000000000000000000000000000000000000000000000000000000")))
   (assert-true   (tn) (integer?  (string-read     "#b111111111111111111111111111111111111111111111111111111111111111")))
   (assert-true   (tn) (integer?  (string-read    "#b+111111111111111111111111111111111111111111111111111111111111111")))
   (assert-true   (tn) (integer?  (string-read  "#b-01000000000000000000000000000000000000000000000000000000000000000")))
   (assert-true   (tn) (integer?  (string-read    "#b0111111111111111111111111111111111111111111111111111111111111111")))
   (assert-true   (tn) (integer?  (string-read   "#b+0111111111111111111111111111111111111111111111111111111111111111")))
   (assert-true   (tn) (integer?  (string-eval  "'#b-1000000000000000000000000000000000000000000000000000000000000000")))
   (assert-true   (tn) (integer?  (string-eval    "'#b111111111111111111111111111111111111111111111111111111111111111")))
   (assert-true   (tn) (integer?  (string-eval   "'#b+111111111111111111111111111111111111111111111111111111111111111")))
   (assert-true   (tn) (integer?  (string-eval "'#b-01000000000000000000000000000000000000000000000000000000000000000")))
   (assert-true   (tn) (integer?  (string-eval   "'#b0111111111111111111111111111111111111111111111111111111111111111")))
   (assert-true   (tn) (integer?  (string-eval  "'#b+0111111111111111111111111111111111111111111111111111111111111111")))))


(total-report)
