/*
  DB Fourier Synthesis Daemon 
  ===========================
  
  Description:
  -----------
  This deamon creates input "channels" using pipes and network sockets,
  and then combines their input for writing to /dev/dsp.

  The daemon assumes that the data is 16 bit 44.1 kHz stereo.  It *is*not*
  the task of this daemon to perform audio format conversions.  Such 
  calculation would add delay to channels that do not require it.

  The number of pipe channels is a constant number set at startup.  Due to 
  connection oriented nature of socket streams, the number of socket channels
  in existence at any given time is variable.  The maximum number of 
  connections is also set at startup.

  This software is brought to you by the letter A and the number 4

  Version: see VRESION_STR declared below
  Author:  Bob Dean
  Copyright (c) 1999, 2000


  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public Licensse as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

*/

/*

 source code used to write wave files copied from the 
 XMMS Disk Writer Plugin, released under the GNU General Public License
 Copyright (C) 1998-2000  Peter Alm, Mikael Alm, Olle Hallnas, 
 Thomas Nilsson and 4Front Technologies

 */

/*
  Design notes:

  Use of signals:
  SIGINT  - shut down
  SIGHUP  - close and reopen audio devices
  SIGUSR1 - start recording to disk using filename stored in sysdata structure 
            (filled in by dbmixer).
  SIGUSR2 - stop recording to disk.

 */


#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <limits.h>
#include <stdarg.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/shm.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <sys/stat.h>
#include <sys/msg.h>

#include <dbsoundcard.h>

#include "dbaudio.h"
#include <dbchannel.h>
#include "fsadders.h"
#include <dbdebug.h>

/* local defines */
#define DBFSD_VERSION_STR   "A.12"

/* timeout values for the select call. */
#define MIN_TIMEOUT_VALUE_SEC   0
#define MAX_TIMEOUT_VALUE_SEC   60
#define MIN_TIMEOUT_VALUE_USEC  25000
#define MAX_TIMEOUT_VALUE_USEC  0

/* local global variables */
extern int errno;
extern int debug_level;

/* local prototypes */
void shutdown_dbfsd();

/***************************************************************************/
/*  Global variables                                                       */
/***************************************************************************/

/* daemon system data*/
dbfsd_data * sysdata;

int loop_flag;   /*main loop flag*/
int free_channels_flag;
int cue_flag;
int single_output_flag;
int reset_audio_flag;

int shmid;     /* shared memory id for channels*/
int sysshmid;  /* shared memory id for dbfsd_data*/
int semid;      /* channel semaphore id  */

FILE * record_fid;
gint record_count;
wav_header header;

/*channel arrays*/
local_channel *local_channels, *cue_channels;
/* socket_channel **socket_channels; */

char tempdirstr[512], cue_tempdirstr[512];

/*flags to hold cmdline switches*/
int stdout_flag;
int quiet_flag;

fd_set read_ch_set; /*fd set for select*/

signed short output_buf[DB_BUFSIZE_SHORT], cue_output_buf[DB_BUFSIZE_SHORT], empty_buf[DB_BUFSIZE_SHORT];

/* signed short *output_buf, *cue_output_buf, *empty_buf; */

oss_control_struct * main_audio, * cue_audio;

/***************************************************************************/
/*  Initialization functions                                               */
/***************************************************************************/

/*
  print_version - outputs version info to stdout
 */
void print_version()
{
	printf("DB Fourier Synthesis Daemon for OSS\n");
	printf("%s\n",DBMIX_COPYRIGHT);
	printf("DBfsd Version: %s\n",DBFSD_VERSION_STR);	
	printf("DBMix Version: %s\n",DBMIX_VERSION);
}

/*
  print_greeting - outputs to stdout a greeting
  
  inputs: none
  outputs: none
*/
void print_help()
{
	print_version();

	printf("\n");
	printf("Supported options (argument type in parens) [defaults in brackets]:    \n");
	printf("   -d   turn on debug messages to console          (none)   [Off]      \n");
	printf("   -c   set op mode to cue                         (none)   [no cue]   \n");
	printf("   -e   output cue as left channel of master out   (none)   [Off]      \n");
	printf("   -n   set total number of input channels         (int)    [4]        \n");
	printf("   -s   set number of input channels to be sockets (int)    [0]        \n");
	printf("   -a   set master output audio device             (string) [/dev/dsp0]\n");
	printf("   -b   set cue output audio device                (string) [/dev/dsp2]\n");
	printf("   -r   set number of OSS output fragments         (int)    [128]      \n");
	printf("   -o   write mixed output to stdout               (none)   [Off]      \n");
	printf("   -v   print version information                  (none)              \n");
	printf("   -h   print this message                         (none)              \n");
	printf("\n");
	printf("For example, to run the server with cueing, 5 input channels, and sending\n");
	printf("master output to /dev/dsp15 you would type:\n");
	printf("dbfsd -c -n 5 -a /dev/dsp15\n");
	printf("\n");
	printf("WARNING: if you use the -r option, the minimum number of allowable fragments\n");
	printf("         is %d. Any fewer, and audio corruption may occur withing the driver.\n",DB_MIN_AUDIO_BUFFS);
	printf("         Depending on your system, if you expeience audio corruption, \n");
	printf("         you should increase the number of fragments.\n");
  
	printf("\n");
}


/*
  parse_cmdline - steps through argv and sets all the variables

  inputs - argc argv
  outputs - sets system variables
 */
void parse_cmdline(int argc, char* argv[])
{
	int opt;

	/* init arguments... */
	quiet_flag = FALSE;
	stdout_flag = FALSE;
	cue_flag = FALSE;
	single_output_flag = FALSE;

	record_fid = NULL;

	sprintf(tempdirstr,"/tmp/ch");
	sysdata->cue_audio_device[0] = '\0';
	cue_tempdirstr[0] = '\0';

	sysdata->pid = getpid();
	strcpy(sysdata->main_audio_device,DEFAULT_AUDIO_DEVICE);

	/* parse argv */
	while ((opt = getopt(argc,argv,"n:s:a:b:ovqhdcer:")) != -1)
	{
		switch (opt)
		{
			case 'h':
				print_help();
				shutdown_dbfsd();
				break;
			case 'a':
				strcpy(sysdata->main_audio_device,optarg);
				Debug("Set master audio device to be: \"%s\"",sysdata->main_audio_device);
				break;
			case 'b':
				strcpy(sysdata->cue_audio_device,optarg);
				Debug("Set cue audio device to be: \"%s\"",sysdata->cue_audio_device);
				break;
			case 'e':
				Debug("Set single output flag.");
				single_output_flag = TRUE;
				sysdata->single_output = TRUE;
				sysdata->cue_split = TRUE;
			    /* fail through to enable cue */
			case 'c':
				cue_flag = TRUE;
				sysdata->cue_enabled = TRUE;
				Debug("set cue flag.");
				break;
			case 'v':
				print_version();
				shutdown_dbfsd();
			case 'n':
				sysdata->num_channels = atoi(optarg);
				printf("sysdata->num_channels set %d\n",sysdata->num_channels);
				break;
			case 'o':
				stdout_flag = TRUE;
				Debug("output to stdout");
				break;
			case 'q':   /* I wish I could remember what this was for... */
				quiet_flag = TRUE;
				Debug("quiet flag set");
				break;
			case 'r':
				sysdata->num_main_buffs = atoi(optarg);

				if (sysdata->num_main_buffs < DB_MIN_AUDIO_BUFFS)
				{
					printf("The requested number of audio buffs is less than the allowable minimum. Setting the number of buffs to be the minimum of: %d buffers.",DB_MIN_AUDIO_BUFFS);
					sysdata->num_main_buffs = DB_MIN_AUDIO_BUFFS;
				}

				sysdata->num_cue_buffs = sysdata->num_main_buffs;
				Debug("num audio buffs set %d\n",sysdata->num_main_buffs);
			case 's':
				sysdata->num_sockets = atoi(optarg);
				Debug("numsockets set %d\n",sysdata->num_sockets);
				break;
			case 'd':
				sysdata->debug_level = debug_level = 1;
				Debug("debug level set\n");
				break;
			case ':':
				printf("option needs a value\n");
				break;
			case '?':
				printf("unknown option: %c\n",optopt);
				exit(0);
				break;
			default: break;
		}
	} /*end while*/

	if (cue_flag)
	{
		sprintf(cue_tempdirstr,"/tmp/cue");

		if (sysdata->cue_audio_device[0] == '\0')
		{
			strcpy(sysdata->cue_audio_device,DEFAULT_CUE_AUDIO_DEVICE);	
		}

		sysdata->cue_pid = getpid();
	}

}



/*
  init_sysdata - allocates shared memory to hold the system dbfsd_data struct

      inputs - none
      outputs - none
      side effects - allocates a chunk of shared memory and point to it with sysdata
 */
void init_sysdata()
{
	/*allocate ssytem data struct*/
	sysshmid = shmget((key_t) DB_SYSTEM_SM_KEY, sizeof(dbfsd_data), 
					  0666 | O_RDWR | IPC_CREAT);

	if (sysshmid == -1) 
	{
		Error("init_sysdata: error creating shared memory for system data.");

		exit(EXIT_FAILURE); 
	}

	sysdata = shmat(sysshmid,(void *)0, 0);
   
	if ((int)sysdata == -1)
	{
		Error("init_sysdata: error attaching system data shared memory.");
       
		if (shmdt(sysdata) == -1)
		{
			Error("init_sysdata: could not detach system data memory segment.");
		}
	
		if (shmctl(sysshmid,IPC_RMID, 0) == -1)
		{
			Error("init_sysdata: could not delete system data shared memory segment.");
		}

		exit(EXIT_FAILURE);
	}

	Debug("init_sysdata: system data shared memory attached successfully.");
   
	memset(sysdata->filename,0,FILENAME_MAX);

	sysdata->num_channels = 4;
	sysdata->num_sockets = 0;
	sysdata->debug_level = 0;
	sysdata->ready_count = 0;
	sysdata->free_channel_index = 0;
	sysdata->clipping_threshold = 20;
	sysdata->main_audio_device[0] = '\0';
	sysdata->cue_audio_device[0] = '\0';
	sysdata->main_mixer_device[0] = '\0';
	sysdata->cue_mixer_device[0] = '\0';
	sysdata->clipping = 0;
	sysdata->num_main_buffs = DB_NUM_FRAGMENTS;
	sysdata->num_cue_buffs = DB_NUM_FRAGMENTS;
	sysdata->left_balance = 100;
	sysdata->right_balance = 100;
	sysdata->cue_split = 0;
	sysdata->single_output = 0;
	sysdata->talkover_enabled = 0;
	sysdata->skip_max = DEFAULT_SKIP_MAX;
    sysdata->default_sampler_time = DB_SAMPLER_DEFAULT_TIME;
	sysdata->mixer_pid = 0;
	sysdata->sampler_op_flag = 0;
	sysdata->recording = 0;
}



int reset_channel(local_channel * ch, int reset)
{	
	char      tempstr[256];

	if (reset)
	{
		Debug("Reseting channel %d...",ch->index);
		Debug("  Closing and unlinking pipes...");
		
		close(ch->server_comm_fd);
		unlink(ch->comm_filename);
		close(ch->server_cue_fd);
		unlink(ch->cue_filename);
	   
		Debug("  closing message queue...");

		if (msgctl(ch->msg_q_id,IPC_RMID,0) != SUCCESS)
		{
			Error("Failed to delete message queue on channel %d",ch->index);
		}
	}
	else
	{
		Debug("  Creating channel %d...",ch->index);
	}

	/* intialize the channel name to "Channel x" */
	sprintf(ch->channel_name,"Channel - %d",ch->index+1);

	Debug("  Set channel name to: %s",ch->channel_name);

	/* create message queue */
	ch->msg_q_id = msgget(DB_CHANNELS_Q_KEY_BASE + ch->index, 0666 | IPC_CREAT);

	if (ch->msg_q_id == -1)
	{
		Error("Failed to create message queue on channel %d",ch->index);
	}
	
	Debug("  reset_channel: Opened message queue.");

	/* create comm pipe */
	if (mkfifo(ch->comm_filename,0777) != 0) 
	{ 
		if (errno != EEXIST)
		{
			sprintf(tempstr,"reset_channel: Error creating %s",ch->comm_filename);
			perror(tempstr); return FAILURE;
		}
	}
       
	/* open comm pipe */
	if ((ch->server_comm_fd = open(ch->comm_filename,
								   O_RDONLY | O_NONBLOCK )) == -1)
	{ 
		sprintf(tempstr,"reset_channel: Error opening %s for read:",ch->comm_filename);
		perror(tempstr); 
		return FAILURE; 
	}

	Debug("  reset_channel: opened pipe %s",ch->comm_filename);
      
	if (cue_flag)
	{
		/* create cue pipe */
		Debug("  reset_channel: creating cue pipe...");

		if (mkfifo(ch->cue_filename,0777) != 0)
		{ 
			if (errno != EEXIST)
			{
				sprintf(tempstr,"reset_channel: Error creating %s",
						ch->cue_filename);
				perror(tempstr); 
				return FAILURE;
			}
		}
		
		/* open cue pipe */
		Debug("  reset_channel: Opening cue pipe...");
		if ((ch->server_cue_fd = open(ch->cue_filename,O_RDONLY | O_NONBLOCK )) == -1)
		{ 
			sprintf(tempstr,"reset_channel: Error opening %s for read:",
					ch->cue_filename);
			perror(tempstr); 
			return FAILURE; 
		}
		
		ch->channel_flags |= CUE_FLAG;

		Debug("  reset_channel: opened pipe %s",ch->cue_filename);
	}
	else
	{
		ch->channel_flags &= ~CUE_FLAG;
		ch->cue_filename[0] = '\0';
		ch->server_cue_fd = -1;
	}

	/*add to select set*/
	FD_SET(ch->server_comm_fd,&read_ch_set);

	ch->msg_flags = 0;
	ch->free_channel = TRUE;
	ch->is_symlink = FALSE;
	ch->left_gain = DEFAULT_GAIN;
	ch->right_gain = DEFAULT_GAIN;
	ch->cue_left_gain = DEFAULT_GAIN;
	ch->cue_right_gain = DEFAULT_GAIN;
	ch->cue = FALSE;
	ch->base_pitch = DEFAULT_PITCH;
	ch->user_pitch = DEFAULT_PITCH;
	ch->message_handler = NULL;
	ch->pause = FALSE;
	ch->mute = FALSE;
	ch->skip_count = 0;

	ch->writing = 0;

	/* init sampler variables */
	ch->sampler_state = SAMPLER_OFF;
	ch->sampler_time = sysdata->default_sampler_time;
	ch->sampler_bufsize = 0;
	ch->sampler_size = 0;
	ch->sampler_readoffset = 0;
	ch->sampler_startoffset = 0;
	ch->sampler_endoffset = 0;
	ch->sampler_buf = NULL;

	/* init default channel flags */
	ch->channel_flags |= (PITCH_FLAG | PAUSE_FLAG);

	Debug("  reset_channel: channel %d reset.",ch->index);

	return SUCCESS;
}


/*
*  init_channels - creates and initializes the input channels
*  channels 1 2 and 3 are named pipes.
*	 channel 4 is the soundcard linein.
*
*	 inputs:  str - filesystem location for FIFO's
*	          num - number of channels
*	 outputs: error codes...
*/
int init_channels()
{
	int len,i;
	local_channel* temp_ch;
	key_t channels_key;

	len = strlen(tempdirstr) + 20;

	/* 	output_buf = (signed short*) malloc(DB_BUFSIZE_CHAR); */
	/* 	cue_output_buf = (signed short*) malloc(DB_BUFSIZE_CHAR); */
	/* 	empty_buf = (signed short *) malloc(DB_BUFSIZE_CHAR); */

	memset(empty_buf,0,DB_BUFSIZE_CHAR);

  	channels_key = DB_CHANNELS_SM_KEY;
	
	Debug("init_channels: samples per channel buffer %d",DB_BUFSIZE_SHORT);

	/* allocate input channel control structs */
	shmid = shmget(channels_key, ((sysdata->num_channels) * sizeof(local_channel)), 0666 | O_RDWR | IPC_CREAT);

	if (shmid == -1) 
	{
		Error("init_channels: error creating shared memory.");

	    shutdown_dbfsd(); 
	}
   
	local_channels = shmat(shmid,(void *)0, 0);
   
	if ((int)local_channels == -1)
	{
		Error("init_channels: error attaching shared memory.");
       
		if (shmdt(local_channels) == -1)
		{
			Error("init_channels: could not detach memory segment.");
		}
	
		if (shmctl(shmid,IPC_RMID, 0) == -1)
		{
			Error("init_channels: could not delete shared memory segment.");
		}

		shutdown_dbfsd();
	}

	Debug("init_channels: shared memory attached successfully.");

	FD_ZERO(&read_ch_set);
	
	temp_ch = local_channels;

	/*initialize channels*/
	for (i = 0; i < sysdata->num_channels; i++)
	{
		local_channels[i].channel_flags = 0;

		/*create comm pipe filename*/      
		sprintf(local_channels[i].comm_filename,"%s%d_comm",tempdirstr,i+1);
		/*create cue filename*/
		if (cue_flag)
		{
			sprintf(local_channels[i].cue_filename,"%s%d_cue",tempdirstr,i+1);
		}
		else
		{
			local_channels[i].cue_filename[0] = '\0';
		}

		local_channels[i].index = i;
		reset_channel(temp_ch,FALSE);
		temp_ch++;
	}

	return SUCCESS;
}


/*
 sigexit - changes system variables to force clean shutdown
 inputs - none
 outputs - none
 */
void sigexit(int signum)
{
	loop_flag = FALSE;
	Debug("Sigexit has been called.");
}


void sigcont(int signum)
{
	Debug("Sigcont has been called.");
}


/*
  sighup - callback when a SIGHUP signal is recieved, resets audio devices
 */
void sighup(int signum)
{
	Debug("sighup has been called.");

	reset_audio_flag = 1;
}


/*
  sigusr - callback when a SIGUSR signal is recieved, starts recording to disk.
 */
void sigusr1(int signum)
{
	Debug("sigusr1 has been called.");
	
	/* if not recording, start recording */
	if (record_fid == NULL)
	{
		sysdata->recording = 1;

		/* open file */
		record_fid = fopen(sysdata->filename,"wb");

		if (!record_fid)
		{
			Error("DBFSD: Could not open output file for recording.\n");
			return;
		}
		
		/* initialize wav header */
		memcpy(&header.main_chunk, "RIFF", 4);
		header.length = GUINT32_TO_LE(0);
		memcpy(&header.chunk_type, "WAVE", 4);
		memcpy(&header.sub_chunk, "fmt ", 4);
		header.sc_len = GUINT32_TO_LE(16);
		header.format = GUINT16_TO_LE(1);
		header.modus = GUINT16_TO_LE(DB_CHANNEL_NUM_CHANNELS);
		header.sample_fq = GUINT32_TO_LE(DB_SAMPLE_RATE);
		header.bit_p_spl = GUINT16_TO_LE(DB_SAMPLE_SIZE * 8);
		header.byte_p_sec = GUINT32_TO_LE(DB_SAMPLE_RATE * DB_CHANNEL_NUM_CHANNELS * DB_SAMPLE_SIZE);
		header.byte_p_spl = GUINT16_TO_LE((GUINT16_FROM_LE(header.bit_p_spl) / (8 / DB_CHANNEL_NUM_CHANNELS)));
		memcpy(&header.data_chunk, "data", 4);
		header.data_length = GUINT32_TO_LE(0);

		/* init data count to zero */
		record_count = 0;

		/* write wav header */
		fwrite(&header, sizeof(wav_header), 1, record_fid);
	}
}


/*
  sigusr2 - callback for SIGUSR2 signal, stops recording to disk
 */
void sigusr2(int signum)
{
	Debug("sigusr2 has been called");

	if (record_fid != NULL)
	{
		sysdata->recording = 0;

		/* recording, stop recording */
		header.length = GUINT32_TO_LE(record_count + sizeof(wav_header) - 8);
		header.data_length = GUINT32_TO_LE(record_count);

		fseek(record_fid, 0, SEEK_SET);
		fwrite(&header, sizeof(wav_header), 1, record_fid);
		fclose(record_fid);

		record_count = 0;
		record_fid = NULL;
	}
}


/*
 shutdown - frees all resources before exiting. 
 * 
 No need to free the global malloc'd bits here since the system process
 cleanup will take care of it...
 * 
  inputs - none
  outputs - none
 */
void shutdown_dbfsd()
{
	int i;

	Debug("closing...");

	Debug("removing channels:");

	unlink(tempdirstr);
	
	if(local_channels != NULL)
	{
		for (i = 0; i < sysdata->num_channels;i++)
		{
			Debug("  removing channel %d of %d...",i+1,sysdata->num_channels);
			close(local_channels[i].server_comm_fd);
			close(local_channels[i].server_cue_fd);
			unlink(local_channels[i].comm_filename);
			unlink(local_channels[i].cue_filename);
			if (msgctl(local_channels[i].msg_q_id,IPC_RMID,0) == -1)
			{
				Error("shutdown_dbfsd: could not delete message queue.");
			}
		}

		Debug("deleteing shared memory segment for channel data... ");
		
		if (shmdt(local_channels) == -1)
		{
			Error("shutdown_dbfsd: could not detach memory segment.");
		}
		
		if (shmctl(shmid,IPC_RMID, 0) == -1)
		{
			Error("shutdown_dbfsd: could not delete shared memory segment.");
		}
	}

	Debug("deleteing system data shared memory segment... ");

	if (shmdt(sysdata) == -1)
	{
		Error("shutdown_dbfsd: could not detach system data memory segment.");
	}

	if (shmctl(sysshmid,IPC_RMID, 0) == -1)
	{
		Error("shutdown_dbfsd: could not delete system data shared memory segment.");
	}

	Debug("closing audio devices...");

	close_audio(cue_audio);
	close_audio(main_audio);

	Debug("done.\nthank you for using DBfsd.");
	/*I can never remember if this should be a 0 or a 1...*/

	exit(0);
}


/*************************************************************************/
/*       Execution functions                                             */
/*************************************************************************/


/*
 make_ch_symlink -
  
 Loops through the available looking for a channel that is not in use. 
 If one is found, a link to /tmp/ch otherwise /tmp/ch does not exist.
 When complete, ch_link is the index of the linked channel, otherwise
 is is greater than data->num_channels.
 */

void make_ch_symlink()
{
	int i, flag,notdone;

	unlink(tempdirstr);

	/*find first unused channel*/
	i=0;
	flag = FALSE;
	notdone = TRUE;

	while ((i < sysdata->num_channels) && notdone) 
    {
		if (local_channels[i].free_channel)
		{
			symlink(local_channels[i].comm_filename,tempdirstr);
			local_channels[i].is_symlink = TRUE;
			flag = TRUE;
			notdone = FALSE;
			sysdata->free_channel_index = i;
		}
		else
		{i++;}
    }

	if (flag)
    {
		Debug("MAKE_SYMLINK: made link to channel %d",i);  
    }
	else 
    { 
		sysdata->free_channel_index = -1;
		free_channels_flag = FALSE;
		Debug("MAKE_SYMLINK: All channels are in use."); 
    }
}


/* run_it debug functions */
void PrintData(char* str)
{
	Debug("%s  ready_count: %d  free_channel_index: %d",str,
		  sysdata->ready_count,sysdata->free_channel_index);
}


void PrintChannelSettings(local_channel * ch)
{
	printf("====================\n");
    printf("Current Channel Settings:\n");
	printf("    Comm Name:  %s\n",ch->comm_filename);

	if ((ch->channel_flags & CUE_FLAG) == CUE_FLAG)
		printf("    Cue  Name:  %s\n",ch->cue_filename);

    switch (ch->channel_type)
	{
		case PIPE_CHANNEL:   printf("    Type:       PIPE\n");break;
		case SOCKET_CHANNEL: printf("    Type:       SOCKET\n");break;
		default:             printf("    Type:       UNKNOWN\n");
	}

	switch (ch->cue)
	{
		case 0:  printf("    Cue?        NO\n"); break;
		default: printf("    Cue?        YES\n"); break;
	}

	switch (ch->free_channel)
	{
		case 0:  printf("    Free?       NO\n"); break;
		default: printf("    Free?       YES\n"); break;
	}

	switch (ch->is_symlink)
	{
		case 0:  printf("    Symlink?    NO\n"); break;
		default: printf("    Symlink?    YES\n");
	}

	printf("    Left Gain:  %d\n",ch->left_gain);
	printf("    Right Gain: %d\n",ch->right_gain);

	printf("====================\n");
}


signed short * add_main_audio(signed short * out_buf,local_channel ** buf_pointers, 
							  int channel_count)
{
	switch (channel_count)
	{
		case 0: return NULL;
		case 1:
			memcpy(out_buf,buf_pointers[0]->buffer.buf,DB_BUFSIZE_CHAR);
			return out_buf;
			break;
		case 2: 
			fs_add2channels(out_buf,&(buf_pointers[0]->buffer),
							&(buf_pointers[1]->buffer));
			return out_buf;
			break;
		case 3: 
			fs_add3channels(out_buf,&(buf_pointers[0]->buffer),
							&(buf_pointers[1]->buffer),&(buf_pointers[2]->buffer));
			return out_buf;
			break;
		case 4: 
			fs_add4channels(out_buf,      &(buf_pointers[0]->buffer),
							&(buf_pointers[1]->buffer), &(buf_pointers[2]->buffer),
							&(buf_pointers[3]->buffer));
			return out_buf;
			break;
		case 5: 
			fs_add5channels(out_buf,      &(buf_pointers[0]->buffer),
							&(buf_pointers[1]->buffer), &(buf_pointers[2]->buffer),
							&(buf_pointers[3]->buffer), &(buf_pointers[4]->buffer));
			return out_buf;
			break;
		case 6: 
			fs_add6channels(out_buf,      &(buf_pointers[0]->buffer),
							&(buf_pointers[1]->buffer), &(buf_pointers[2]->buffer),
							&(buf_pointers[3]->buffer), &(buf_pointers[4]->buffer),
							&(buf_pointers[5]->buffer));
			return out_buf;
			break;
		case 7: 
			fs_add7channels(out_buf,      &(buf_pointers[0]->buffer),
							&(buf_pointers[1]->buffer), &(buf_pointers[2]->buffer),
							&(buf_pointers[3]->buffer), &(buf_pointers[4]->buffer),
							&(buf_pointers[5]->buffer), &(buf_pointers[6]->buffer));
			return out_buf;
			break;
		case 8: 
			fs_add8channels(out_buf,      &(buf_pointers[0]->buffer),
							&(buf_pointers[1]->buffer), &(buf_pointers[2]->buffer),
							&(buf_pointers[3]->buffer), &(buf_pointers[4]->buffer),
							&(buf_pointers[5]->buffer), &(buf_pointers[6]->buffer),
							&(buf_pointers[7]->buffer));
			return out_buf;
			break;  
		default:
			Error("add_main_audio: fs_addNchannels NOT IMPLEMENTED!!!!!!!!");
			/* fs_addNchannels(out_buf,&(buf_pointers,sysdata->num_channels);*/
			break;
	} /* end switch */
	
	return NULL;
}



signed short * add_cue_audio(signed short * out_buf,local_channel ** buf_pointers, 
							 int channel_count)
{
	switch (channel_count)
	{
		case 0: return NULL;
		case 1:
			memcpy(out_buf,buf_pointers[0]->cue_buffer.buf,DB_BUFSIZE_CHAR);
			return out_buf;
			break;
		case 2: 
			fs_add2channels(out_buf,
							&(buf_pointers[0]->cue_buffer),
							&(buf_pointers[1]->cue_buffer));
			return out_buf;
			break;
		case 3: 
			fs_add3channels(out_buf,&(buf_pointers[0]->cue_buffer),
							&(buf_pointers[1]->cue_buffer),&(buf_pointers[2]->cue_buffer));
			return out_buf;
			break;
		case 4: 
			fs_add4channels(out_buf,      &(buf_pointers[0]->cue_buffer),
							&(buf_pointers[1]->cue_buffer), &(buf_pointers[2]->cue_buffer),
							&(buf_pointers[3]->cue_buffer));
			return out_buf;
			break;
		case 5: 
			fs_add5channels(out_buf,      &(buf_pointers[0]->cue_buffer),
							&(buf_pointers[1]->cue_buffer), &(buf_pointers[2]->cue_buffer),
							&(buf_pointers[3]->cue_buffer), &(buf_pointers[4]->cue_buffer));
			return out_buf;
			break;
		case 6: 
			fs_add6channels(out_buf,      &(buf_pointers[0]->cue_buffer),
							&(buf_pointers[1]->cue_buffer), &(buf_pointers[2]->cue_buffer),
							&(buf_pointers[3]->cue_buffer), &(buf_pointers[4]->cue_buffer),
							&(buf_pointers[5]->cue_buffer));
			return out_buf;
			break;
		case 7: 
			fs_add7channels(out_buf,      &(buf_pointers[0]->cue_buffer),
							&(buf_pointers[1]->cue_buffer), &(buf_pointers[2]->cue_buffer),
							&(buf_pointers[3]->cue_buffer), &(buf_pointers[4]->cue_buffer),
							&(buf_pointers[5]->cue_buffer), &(buf_pointers[6]->cue_buffer));
			return out_buf;
			break;
		case 8: 
			fs_add8channels(out_buf,      &(buf_pointers[0]->cue_buffer),
							&(buf_pointers[1]->cue_buffer), &(buf_pointers[2]->cue_buffer),
							&(buf_pointers[3]->cue_buffer), &(buf_pointers[4]->cue_buffer),
							&(buf_pointers[5]->cue_buffer), &(buf_pointers[6]->cue_buffer),
							&(buf_pointers[7]->cue_buffer));
			return out_buf;
			break;  
		default:
			Error("add_cue_audio: fs_addNchannels NOT IMPLEMENTED!!!!!!!!");
			/* fs_addNchannels(out_buf,buf_pointers,sysdata->num_channels);*/
			break;
	} /* end switch */
	
	return NULL;
}


/*
  run_it - executes the channel listener loop, applies Fourier Synthesis when
           data is ready.

		   Note that two different timeout values are used with select.  
		   While audio is being played, the smaller value is being used. 
		   Once a timeout occurs, the larger value will be used. 
		   This way the audio pause command can be issued a short time 
		   after audio has stopped, and cleanup the audio output in case 
		   artifacts remain.  

  inputs - the input channel pipes
  outputs - the fourier synthesis of the pipes is sent to the soundcard.
 */
void run_it()
{
	static int select_result,result,channel_count,cue_count,numch;
	static int i,checknum;
	/*pointers to arrays of signed shorts*/
	static local_channel ** buf_pointers;
	static local_channel ** temp_buf_pointers;
	static local_channel ** cue_pointers;
	static local_channel ** temp_cue_pointers;
 
	static fd_set  temp_read_set;
	static struct  timeval timeout;
	static int     timeout_value_sec;
	static int     timeout_value_usec;

	/* temp channel */
	static struct  local_channel_s *temp_ch;
	static struct  local_channel_s *temp_cue;
	static struct  channel_buf_s   *temp_buf;
	static int     total_timeout_count,current_timeout_count;
	static int     local_cue;
	static int     pause_flag;
	static int     skipnum;

/* 	static int loop_count = 0; */

	/* init local vars */
	result = select_result = channel_count = cue_count = i = 0;
	loop_flag = TRUE;
	checknum = total_timeout_count = current_timeout_count = 0;
  
	/* set number of (io fd's to check) to be the max read fd + 1 */
	numch = local_channels[sysdata->num_channels-1].server_comm_fd + 1;
  
	/* allocate the temporary buf pointers */
	buf_pointers = (local_channel **) malloc(sizeof(local_channel*) * sysdata->num_channels);
	cue_pointers = (local_channel **) malloc(sizeof(local_channel*) * sysdata->num_channels);	

	for (i = 0; i < sysdata->num_channels; i++)
    {
		buf_pointers[i] = NULL;
    }
  
	free_channels_flag = TRUE;
	make_ch_symlink();

	local_cue = FALSE;
	pause_flag = 1;

	/* set initial timeout to be 5 seconds */
	timeout_value_sec = MIN_TIMEOUT_VALUE_SEC;
	timeout_value_usec = MIN_TIMEOUT_VALUE_USEC;

	while (loop_flag)
    {
		/* reset audio if necessary */
		if (reset_audio_flag)
		{
			reset_audio_flag = 0;

			if (main_audio != NULL) 
			{
				close_audio(main_audio);
				main_audio = NULL;
			}
			
			Debug("Re-opening main audio device.");
			
			if ((main_audio = init_audio(sysdata->main_audio_device,O_WRONLY,stdout_flag,
										 sysdata->num_main_buffs)) == NULL)
			{
				Error("Could not open main audio device.");
			}
			
			if (cue_flag)
			{
				Debug("Re-opening cue audio device...");
				
				if (cue_audio != NULL) 
				{
					close_audio(cue_audio);
					cue_audio = NULL;
				}
				
				if ((cue_audio = init_audio(sysdata->cue_audio_device,
											O_WRONLY/*  | O_NONBLOCK */,
											stdout_flag,sysdata->num_cue_buffs)) == NULL)
				{
					Error("Could not open cue audio device.");
				}
			}
			
			Debug("Audio is reset.");
		}

		/*******  Handle IO *******/

		/* reset file descriptor set */
		FD_ZERO(&temp_read_set);
		temp_read_set = read_ch_set;

		/* reset temp_buf_pointer and channel pointer */
		temp_buf_pointers = buf_pointers;
		temp_cue_pointers = cue_pointers;

		temp_ch = local_channels; 
		temp_cue = local_channels;
		temp_buf = NULL;
      
		channel_count = cue_count = 0;
      
		/* check to see if any descriptors are ready...
		   select returns -1 on an interrupt, which is handled elsewhere,
		   and returns 0 on timeout. */      

		timeout.tv_sec = timeout_value_sec;
		timeout.tv_usec = timeout_value_usec;

		select_result = select(numch,&temp_read_set,(fd_set*) 0,(fd_set*) 0, &timeout);
      
		/* verify that all pipes have enough data in them */
		if (select_result > 0)
		{
			skipnum = 0;
			pause_flag = 1;
			timeout_value_sec = MIN_TIMEOUT_VALUE_SEC;
			timeout_value_usec = MIN_TIMEOUT_VALUE_USEC;

			/* set through channels and see if data is ready */
			for (i=0; i < sysdata->num_channels;i++)
			{
				if (FD_ISSET(temp_ch->server_comm_fd,&temp_read_set))
				{
					/* get amount of data in the buffer */
					if (ioctl(temp_ch->server_comm_fd,FIONREAD,&result) != 0)
					{
						perror("FIONREAD FAILED!!!  ");
					}
					else
					{
						if (result < DB_BUFSIZE_CHAR)
						{
							/* sleep for 5000 miroseconds and wait for data to arrive on the pipe.
							   the 5000 is becuase this is slightly less than
							   the amount of sound data in a 1024 byte buffer */
							int usec = 5000;
							struct timeval tv;

							temp_ch->skip_count++;
							
							tv.tv_sec = usec / 1000000;
							usec -= tv.tv_sec * 1000000;
							tv.tv_usec = usec;
							select(0, NULL, NULL, NULL, &tv);

							/* force the for loop to exit to minimize sleeps */
							i = sysdata->num_channels;
														
							/* to prevent pausing all output, if less than a buffers worth
                               of data is sitting in the pipe for more than SKIP_MAX iterations,
							   then clean the pipe out */
							if (temp_ch->skip_count < sysdata->skip_max) skipnum++;
						}
					}
				}
				
				temp_ch++;
			}

			/* if there is a pipe without enough data, then skip the I/O section of the
               loop so that the pipe has time to catch up */
			if (skipnum) continue;

			temp_ch = local_channels;

  			/* check each channel to see if data is ready */ 
			for (i=0; i < sysdata->num_channels;i++)
			{
				/* grab cue value for use inside the critical section */
				local_cue = temp_ch->cue;

				if (FD_ISSET(temp_ch->server_comm_fd,&temp_read_set))
				{
					switch (temp_ch->channel_type)
					{
						case PIPE_CHANNEL:
							/* get buffer to read pipe data to, always use buffer1 */
							temp_buf = &(temp_ch->buffer);
							result = read(temp_ch->server_comm_fd,temp_buf->buf,
										  DB_BUFSIZE_CHAR);
							temp_buf->len = result;
							temp_ch->skip_count = 0;
							break;
						case SOCKET_CHANNEL:
							Error("Socket channels not yet implemented.");
							break;
						default:
							Error("Unknown channel type.");
							break;
					}

					/* if we read data, then add it to the buffer array */
					if (result > 0)
					{
						/* remember buffer cause temp_ch->buffer-> requires a useless
						   extra addition... depends on how smart the compiler is... */
						temp_buf = &(temp_ch->buffer);
						temp_buf->data_ready = TRUE;
						
						/* see if this channel was previously free, if so then update 
						   the free channel symlink also, toss out this buffer of info 
						   because it is the init buffer written by DBAudio_Init() */
						if (temp_ch->is_symlink)
						{
							if (Check_Debug())
							{
								Debug("New data on %s",temp_ch->comm_filename);
								PrintChannelSettings(temp_ch);
							}
							temp_ch->is_symlink = FALSE;
							temp_ch->free_channel = FALSE;
							make_ch_symlink();

							/* clear cue buffer data for synch.*/
							if (cue_flag)
							{
								if (local_cue)
								{
									result = read(temp_ch->server_cue_fd,temp_buf->buf,
												  DB_BUFSIZE_CHAR);
								}
							}
						}
						else
						{
							/* calculate check value for buffer zeroing */
							checknum = DB_BUFSIZE_CHAR - temp_buf->len;
							/* if the max # of bytes wasn't read, then blank the tail of the buffer*/
							if (checknum)
							{ 
								Debug("cleared %d bytes",checknum);
								/* divide by two since buf is signed shorts which are 2 bytes long */

								memset(temp_buf->buf + (temp_buf->len / 
														sizeof(signed short)),0,checknum);
							}
							
							/* keep a pointer to the buffer that has data_ready */
							temp_buf_pointers[channel_count] = temp_ch;
							channel_count++;
							
							/* we read the cue data here becuase, cue data exists only
							   if data exists on the main input. It is easier to use the
							   main input pipe to detect data ready. */
							if (cue_flag && local_cue)
							{
								temp_buf = &(temp_ch->cue_buffer);
								result = read(temp_ch->server_cue_fd,temp_buf->buf,
											  DB_BUFSIZE_CHAR);
								
								if (result > 0)
								{									
									temp_buf->len = result;
									temp_buf->data_ready = TRUE;
									
									/* calculate check value for buffer zeroing*/
									checknum = DB_BUFSIZE_CHAR - temp_buf->len;
									/* if the max # of bytes wasn't read, then blank the tail of the buffer*/
									if (checknum)
									{
										/* divide by two since buf is signed shorts which are 2 bytes long */
										memset(temp_buf->buf + (temp_buf->len / 
																sizeof(signed short)),0,checknum);
									}
									
									/* keep a pointer to the buffer that has data_ready */
									temp_cue_pointers[cue_count] = temp_ch;
									cue_count++;
								} /* end if result > 0 */
								
							}  /* end cue_flag */
						}
					}  /*end if result */
					else 
					{
						/* Now there is a probable EOF situation so remove the pipe, and reopen it */
						Debug("Handling pipe EOF: %s",temp_ch->comm_filename);
		      
						FD_CLR(temp_ch->server_comm_fd,&read_ch_set);

						reset_channel(temp_ch,TRUE);

						/*check the symlink*/
						if (!free_channels_flag)
						{
							Debug("Channels in use is now less than Maximum.");
							free_channels_flag = TRUE;
							make_ch_symlink();
						}
					}
				}        /*end fd_set check*/

				temp_ch++;
			}  /*end for*/
			
			/*write main channel data*/			
			if (add_main_audio(output_buf,buf_pointers,channel_count) != NULL)
			{
				/*write cue data*/
				if (cue_flag)
				{
					if (add_cue_audio(cue_output_buf,cue_pointers,cue_count) != NULL)
					{
						/* handle cue split */
						if (sysdata->cue_split)
						{
							int  tempval, tempval2;
							short * tempout, *tempout2;
							int i;

							tempout = (short *) output_buf;
							tempout2 = (short *) cue_output_buf;

							tempout = output_buf;
							tempout2 = cue_output_buf;
							
							for(i = 0; i < DB_BUFSIZE_SHORT/2; i++)
							{
								/* get output sample */
								tempval = (int)(((float)tempout[0]) + ((float)(tempout[1])));
								/* get cue sample */
								tempval2 = (int)(((float)tempout2[0]) + ((float)(tempout2[1])));
								
								if(tempval > 32767) {tempval = 32767;}
								if(tempval < -32688) {tempval = -32767;}
								if(tempval2 > 32767) {tempval2 = 32767;}
								if(tempval2 < -32688) {tempval2 = -32767;}

								/* add to cue buffer */
								*tempout2 = (short)tempval;
								tempout2++;
								*tempout2 = (short)tempval2;
								tempout2++;
								
								tempout++;
								tempout++;
							}
						}
						
						if (single_output_flag)
						{
							memcpy(output_buf,cue_output_buf,DB_BUFSIZE_CHAR);
						}
						else
						{
							if (write_audio(cue_audio,cue_output_buf,DB_BUFSIZE_CHAR) <= 0)
							{
								perror("error outputing cue data ");
							}
						}
					}
					else /* add_cue_audio */
					{
						if (single_output_flag)
						{
							/* create mono output of master and set cue to zero */

							int  tempval, tempval2;
							short * tempout, *tempout2;
							int i;

							tempout = (short *) output_buf;
							tempout2 = (short *) cue_output_buf;

							tempout = output_buf;
							tempout2 = cue_output_buf;
							
							for(i = 0; i < DB_BUFSIZE_SHORT/2; i++)
							{
								/* get output sample */
								tempval = (int)(((float)tempout[0]) + ((float)(tempout[1])));
								/* get cue sample */
								tempval2 = (int)(((float)tempout2[0]) + ((float)(tempout2[1])));
								
								if(tempval > 32767) {tempval = 32767;}
								if(tempval < -32688) {tempval = -32767;}
								if(tempval2 > 32767) {tempval2 = 32767;}
								if(tempval2 < -32688) {tempval2 = -32767;}

								/* add to cue buffer */
								*tempout = (short)tempval;
								tempout++;
								*tempout = (short)0;
								tempout++;
								
								tempout2++;
								tempout2++;
							}
						}
						else
						{
							write_audio(cue_audio,empty_buf,DB_BUFSIZE_CHAR);
						}
/* 						memset(cue_output_buf,0,DB_BUFSIZE_CHAR); */
					} /* add cue audio */
				}  /* end write cue data */

				if (write_audio(main_audio,output_buf,DB_BUFSIZE_CHAR) <= 0)
				{
					perror("error outputing main data ");
				}

				/* if we are recording, write data to file */
				if (record_fid != NULL)
				{
					/* make sure there is space in the file, Max is ~4GB */
					if (record_count < (MAX_WAV_LENGTH - DB_BUFSIZE_CHAR))
					{
						record_count += fwrite(output_buf,1,DB_BUFSIZE_CHAR,record_fid);
					}
					else
					{
						/* out of space, stop recording */
						raise(SIGUSR2);
						Error("DBFSD: ran out of space while recording output.");
					}
				}

			} /* add_main_audio */
		}  /*end if select_result*/
		else 
		{
			if (select_result == 0)
			{
				Debug("Timeout total: %d  since last sample: %d (minutes)",total_timeout_count,current_timeout_count);
				total_timeout_count++;
				current_timeout_count++;

				/* since no audio has occured, set longer timeout */
				timeout_value_sec = MAX_TIMEOUT_VALUE_SEC;
				timeout_value_usec = MAX_TIMEOUT_VALUE_USEC;

				if (pause_flag)
				{
					Debug("Pausing main audio...");
					pause_audio(main_audio);
					
					if (cue_flag)
					{
						Debug("Pausing cue audio...");
						pause_audio(cue_audio);
					}

					pause_flag = 0;
				}
			}

			/*received an error on select, check which one it is.*/
			if (select_result == -1)
			{
				switch (errno)
				{
					case EBADF:
						Error("EBADF: invalid descriptors for select.\n");
						break;
					case EINTR:
						Debug("EINTR: select returned due to interrupt.\n");
						break;
					case EINVAL:
						Error("EINVAL: invalid parameters to select.\n");
						break;
					default: Error("Unknown error on select.\n"); break;
				}
			} /*end if select_result*/
		}  /*end else*/
	} /* end while */
}



/***************************************************************************/
/*                              MAIN                                       */
/***************************************************************************/

int main (int argc, char *argv[])
{
	/*daemon init's*/
#ifdef DBMIX_DEBUG
	printf("WARNING!!!  This program has been compiled with environment options specific to the head DBMix developer.  It may not work on your system!!!\n");
#endif

	sysdata = NULL;
	local_channels = NULL;

	init_sysdata();
	Debug("sysdata allocated.");
	parse_cmdline(argc,argv);

	Debug("cmdline parsed.");
	Debug("pid is: %d\n",sysdata->pid);

	Debug("Begining initialization...\n");

   	/*init audio, this *must* be done before the call to init_channels*/

	/* open audio device */
	/* if a device was not specified on the command line, then
	   use the hardcoded defaults. */
	Debug("Opening master audio device... %s...",sysdata->main_audio_device);
		
	main_audio = cue_audio = NULL;

	if ((main_audio = init_audio(sysdata->main_audio_device,O_WRONLY,stdout_flag,
								 sysdata->num_main_buffs)) == NULL)
	{
		Error("Could not open main audio device.");
		return -1;
	}
	
	if (single_output_flag)
	{
		Debug("WARNING!!! Cue will be output as a channel of the master output!!!!\n");
		cue_audio = NULL;
	}
	else
	{
		if (cue_flag)
		{
			Debug("Opening cue audio device...");
			
			if ((cue_audio = init_audio(sysdata->cue_audio_device,
										O_WRONLY/*  | O_NONBLOCK */,
										0,sysdata->num_cue_buffs)) == NULL)
			{
				Error("Could not open cue audio device.");
				return -1;
			}
		}
	}

	Debug("audio initialized.");

	if (init_channels() != SUCCESS)
	{shutdown_dbfsd(); return -1;}

	Debug("channels initialized.");
  
  /*install signal handlers*/
	reset_audio_flag = 0;

	signal(SIGINT,sigexit);
	signal(SIGTERM,sigexit);
	signal(SIGCONT,sigcont);
	signal(SIGHUP,sighup);
	signal(SIGUSR1,sigusr1);
	signal(SIGUSR2,sigusr2);

	Debug("Initialization complete... running... \n");
  
	run_it();

	Debug("Run_it complete.");

	shutdown_dbfsd();

	return 0;
}






