#!/usr/bin/perl -w
#
# basic.t -- Basic tests for podlators.
#
# Copyright 2001, 2002, 2004, 2006, 2009 by Russ Allbery <rra@stanford.edu>
#
# This program is free software; you may redistribute it and/or modify it
# under the same terms as Perl itself.

BEGIN {
    chdir 't' if -d 't';
    if ($ENV{PERL_CORE}) {
        @INC = '../lib';
    }
    unshift (@INC, '../blib/lib');
    $| = 1;
}

use strict;

use Test::More tests => 15;

BEGIN {
    use_ok ('Pod::Man');
    use_ok ('Pod::Text');
    use_ok ('Pod::Text::Overstrike');
    use_ok ('Pod::Text::Termcap');
}

# Find the path to the test source files.  This requires some fiddling when
# these tests are run as part of Perl core.
sub source_path {
    my $file = shift;
    if ($ENV{PERL_CORE}) {
        require File::Spec;
        my $updir = File::Spec->updir;
        my $dir = File::Spec->catdir ($updir, 'lib', 'Pod', 't');
        return File::Spec->catfile ($dir, $file);
    } else {
        return $file;
    }
}

# Hard-code a few values to try to get reproducible results.
$ENV{COLUMNS} = 80;
$ENV{TERM} = 'xterm';
$ENV{TERMCAP} = 'xterm:co=80:do=^J:md=\E[1m:us=\E[4m:me=\E[m';

# Map of translators to file extensions to find the formatted output to
# compare against.
my %translators = ('Pod::Man'              => 'man',
                   'Pod::Text'             => 'txt',
                   'Pod::Text::Color'      => 'clr',
                   'Pod::Text::Overstrike' => 'ovr',
                   'Pod::Text::Termcap'    => 'cap');

# Set default options to match those of pod2man and pod2text.
our %options = (sentence => 0);

for my $module (sort keys %translators) {
  SKIP: {
        if ($module eq 'Pod::Text::Color') {
            eval { require Term::ANSIColor };
            skip 'Term::ANSIColor not found', 3 if $@;
            require_ok ('Pod::Text::Color');
        }
        my $parser = $module->new (%options);
        isa_ok ($parser, $module, 'Parser object');

        # For Pod::Man, strip out the autogenerated header up to the .TH title
        # line.  That means that we don't check those things; oh well.  The
        # header changes with each version change or touch of the input file.
        open (OUT, '> out.tmp') or die "Cannot create out.tmp: $!\n";
        $parser->parse_from_file (source_path ('basic.pod'), \*OUT);
        close OUT;
        if ($module eq 'Pod::Man') {
            open (TMP, 'out.tmp') or die "Cannot open out.tmp: $!\n";
            open (OUTPUT, "> out.$translators{$module}")
                or die "Cannot create out.$translators{$module}: $!\n";
            local $_;
            while (<TMP>) { last if /^\.nh/ }
            print OUTPUT while <TMP>;
            close OUTPUT;
            close TMP;
            1 while unlink 'out.tmp';
        } else {
            rename ('out.tmp', "out.$translators{$module}")
                or die "Cannot rename out.tmp: $!\n";
        }

        # Slurp the output and expected output and compare them.
        my ($master, $output);
        {
            local $/;
            open (MASTER, source_path ("basic.$translators{$module}"))
                or die "Cannot open basic.$translators{$module}: $!\n";
            open (OUTPUT, "out.$translators{$module}")
                or die "Cannot open out.$translators{$module}: $!\n";
            $master = <MASTER>;
            $output = <OUTPUT>;
            close MASTER;
            close OUTPUT;
        }

        # OS/390 is EBCDIC, which uses a different character for ESC
        # apparently.  Try to convert so that the test still works.
        if ($^O eq 'os390' and $module eq 'Pod::Text::Termcap') {
            $output =~ tr/\033/\047/;
        }
        if (ok ($master eq $output, "$module output is correct")) {
            1 while unlink "out.$translators{$module}";
        } else {
            diag ("Non-matching output left in out.$translators{$module}\n");
        }
    }
}
