# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with
code in this repository.

## Project Overview

Devel::Cover is a Perl code coverage analysis tool that provides statement,
branch, condition, subroutine, and pod coverage metrics. The module replaces
Perl ops with functions that count execution to determine code coverage. It
includes various reporting formats including HTML, text, and vim integration.

## Build and Test Commands

This project uses ExtUtils::MakeMaker for building and testing:

### Basic Build Process

```bash
perl Makefile.PL
make
make test
make install
```

### Test Commands

```bash
# Run all tests
make test

# Run tests with verbose output
make test TEST_VERBOSE=1

# Run specific test files
make test TEST_FILES='t/specific.t'

# Run tests with coverage using yath (preferred test runner)
yath test -j20 --no-color -v -T --term-width=200

# Run a single test with yath
yath test t/specific.t -j20 --no-color -v -T --term-width=200
```

### Coverage Analysis

```bash
# Basic coverage for uninstalled module
cover -test

# Manual coverage collection
cover -delete
HARNESS_PERL_SWITCHES=-MDevel::Cover make test
cover

# Coverage for a specific program
perl -MDevel::Cover yourprog.pl args
cover
```

### Development Commands

```bash
# Generate ctags
make tags

# Show version
make show_version

# Create test files from tests/ directory
# (automatically done by Makefile.PL)

# Self-coverage analysis
make self_cover

# Create distribution
make dist
```

## Code Quality Tools

### Linting and Formatting

- **perlcritic**: Configured via `.perlcriticrc` (severity 2, allows unsafe)
- **perltidy**: Configured via `.perltidyrc` (2-space indentation, specific
formatting rules)
- Run both tools on all modified files before committing

### Testing Requirements

- Tests must use Test2 system (not Test::More)
- All new code requires tests in the `t/` directory
- Tests should be run with `yath` test runner
- Coverage verification using Devel::Cover with JSON reports

## Architecture Overview

### Core Modules

**Coverage Collection**:

- `Devel::Cover` - Main module that hooks into Perl's op execution
- `Devel::Cover::Op` - Handles operation coverage tracking
- `Devel::Cover::DB` - Database storage and management
- `Devel::Cover::Inc` - Include path management (auto-generated)

**Coverage Criteria**:

- `Devel::Cover::Statement` - Statement coverage
- `Devel::Cover::Branch` - Branch coverage
- `Devel::Cover::Condition*` - Condition coverage (multiple variants)
- `Devel::Cover::Subroutine` - Subroutine coverage
- `Devel::Cover::Pod` - POD documentation coverage

**Storage Backends** (`Devel::Cover::DB::IO::*`):

- `Storable` - Binary format (core module)
- `JSON` - JSON format (preferred if available)
- `Sereal` - Sereal format (fastest if available)

**Reporting System** (`Devel::Cover::Report::*`):

- `Html*` - Multiple HTML report variants (basic, minimal, subtle)
- `Text*` - Text-based reports
- `Vim` - Vim editor integration
- `Json` - JSON output format
- `Compilation` - Compilation error format

### Binary Tools

- `cover` - Main coverage report generator
- `cpancover` - CPAN coverage analysis tool
- `gcov2perl` - Convert gcov files to Devel::Cover format

### Test Infrastructure

- `tests/` - Test files that get converted to `t/e2e/` during build
- `test_output/` - Expected output for different Perl versions
- `t/e2e/` - End-to-end tests (generated from tests/)
- `t/internal/` - Internal module tests

## Development Notes

### Code Generation

- `lib/Devel/Cover/Inc.pm` is auto-generated by `Makefile.PL`
- Test files in `t/e2e/` are auto-generated from `tests/` directory
- The build process customizes tests for the current Perl version

### XS Component

- Contains C code (`Cover.xs`, `Cover.c`) for low-level Perl integration
- Uses typemap in `utils/typemap`

### Version Compatibility

- Supports Perl 5.12+
- Has version-specific test outputs for different Perl versions
- Self-adjusts for different Perl installations

### Docker Support

- Multiple Docker configurations in `docker/` directory
- Support for different Perl versions and development environments
